package gov.va.genisis2;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.log4j.Logger;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.va.genisis2.ts.common.dto.TripleDTO;

/**
 * @author PII
 *
 */
public class CliScript {

	private static final Logger LOGGER = Logger.getLogger(CliScript.class);
	
	private static final Pattern VALID_IP_ADDRESS = Pattern.compile(
	        "^(([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\.){3}([01]?\\d\\d?|2[0-4]\\d|25[0-5])$");
	
	public static void main(String[] args) {
		
		CommandClient commandClient = new CommandClient(
				new PropertyManager(CliScriptConstants.PROPERTY_FILE_LOCATION));
		
		CommandLineParser parser = new DefaultParser();
		CommandLine cmdLine = null;
		Options options = null;
		try {
			options = createCommandLineOptions();
			cmdLine = parser.parse(options, args);
		} catch (ParseException e) {
			LOGGER.error("Couldn't parse command line arguments "+e);
			System.err.println("Error: "+e);
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("java -jar tscript-jar-with-dependencies.jar", options);
			System.exit(0);
		}
		if( cmdLine.hasOption( CliScriptConstants.HELP ) )  {
			LOGGER.debug("help message");
			HelpFormatter formatter = new HelpFormatter();
			formatter.printHelp("java -jar ts-cli-script-jar-with-dependencies.jar", options);
			System.exit(0);
		}
		
		if( cmdLine.hasOption( CliScriptConstants.GENERATE )) {
			LOGGER.debug("Generate existing sets of subjects, predicates, and objects");
			String endpoint = cmdLine.getOptionValue(CliScriptConstants.GENERATE);
			
			if( endpoint == null ) {
				endpoint = CliScriptConstants.DEFAULT_SPARQL_ENDPOINT;
			}
			commandClient.generateExistingTripleFiles(endpoint);
		}
		
		if( cmdLine.hasOption( CliScriptConstants.CREATE )) {
			LOGGER.debug("Create triples");
			Integer numberOfTriples = 1;
			try {
				numberOfTriples = Integer.parseInt( cmdLine.getOptionValue(CliScriptConstants.CREATE) );
			}catch( NumberFormatException ne ) {
				LOGGER.error("Cannot parse the number of triples...defaulting to one");
			}finally {
				commandClient.createTriples( numberOfTriples );
			}
		}
		if( cmdLine.hasOption( CliScriptConstants.EXISTS ) )  {
			LOGGER.debug("Checking if triples exist");
			
			List<TripleDTO> triples = null;
			ObjectMapper mapper = new ObjectMapper();

			String[] values = cmdLine.getOptionValues( CliScriptConstants.EXISTS );
			
			if( cmdLine.getOptionValue(CliScriptConstants.EXISTS) != null ) {
				//read json values from a file
				try {
					if( validateIpAddress(values[0]) ) {
						//they have entered in an ip address and no file name
						System.err.println(CliScriptConstants.EXISTS_HELP);
						System.exit(-1);
					}
					//make a file from the first argument
					triples = mapper.readValue(new File(values[0]), 
							new TypeReference<List<TripleDTO>>(){});
					String endpoint = CliScriptConstants.DEFAULT_SPARQL_ENDPOINT;
					if( values[1] != null && validateIpAddress( values[1] )) {
						endpoint = constructSpqarqlEndpoint(values[1]);
					}
					commandClient.checkIfTriplesExists( triples, false, endpoint );
				} catch (IOException e) {
					LOGGER.error(e.getMessage());
				}
			}
			else {
				//read from stdin which means the default endpoint should be used
				try {
					triples = mapper.readValue(new BufferedReader(
							new InputStreamReader(System.in)), 
							new TypeReference<List<TripleDTO>>(){});
					commandClient.checkIfTriplesExists( triples, true, CliScriptConstants.DEFAULT_SPARQL_ENDPOINT);
				} catch (IOException e) {
					LOGGER.error(e.getMessage());
				}
			}
		}
	}
	

	/**
	 * Create the option objects used as command line arguments
	 * @return org.apache.commons.cli.Options
	 */
	private static Options createCommandLineOptions() {
		Options options = new Options();

		Option helpOption = Option.builder(CliScriptConstants.HELP).
				required(false).
				desc("Displays the help message").
				build();
		
		//establish mutually exclusive group for required create, exists, generate
		OptionGroup mutualExclusiveOptions = new OptionGroup();
		mutualExclusiveOptions.setRequired(true);

		Option createOption = Option.builder(CliScriptConstants.CREATE).
				numberOfArgs(1).
				type(Integer.class).
				desc("Generate a JSON file of triples").
				build();
		
		Option existsOption = Option.builder(CliScriptConstants.EXISTS).
				numberOfArgs(2).
				optionalArg(true).
				desc(CliScriptConstants.EXISTS_HELP).
				build();
				
				
		Option generateOption = Option.builder(CliScriptConstants.GENERATE).
				numberOfArgs(1).
				optionalArg(true).
				desc("Will generate 3 text files representing triples for all existing subjects, predicates, "
						+ "and objects given a fuseki ip address location. If no IP given will default to: "
						+ CliScriptConstants.DEFAULT_SPARQL_ENDPOINT).
				build();
		
		options.addOption(helpOption);
		
		mutualExclusiveOptions.addOption(createOption);
		mutualExclusiveOptions.addOption(existsOption);
		mutualExclusiveOptions.addOption(generateOption);
		
		options.addOptionGroup(mutualExclusiveOptions);

		return options;
		
	}
	
	/**
	 * Returns true if the given string is valid ip address
	 * @param ip
	 * @return
	 */
	public static boolean validateIpAddress(String ip) {
		 return VALID_IP_ADDRESS.matcher(ip).matches();
	}
	
	/**
	 * Construct a sparql endpoint given an IP address
	 * @param ip
	 * @return
	 */
	public static String constructSpqarqlEndpoint(String ip) {
		StringBuilder builder = new StringBuilder();
		
		return builder.append("http://")
		.append(ip)
		.append(":3030/ds/sparql").toString();
		
	}
}
