package gov.va.genisis2.ts.service.impl;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrInputDocument;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.ts.converter.SolrConceptDocumentConverter;
import gov.va.genisis2.ts.dto.ConceptCardDTO;
import gov.va.genisis2.ts.service.ISolrService;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;

/**
 * 
 * @author PII
 *
 */
@Service
public class SolrService implements ISolrService {

	private static final Logger LOGGER = LogManager.getLogger(SolrService.class);

	private static final String ID = "id";
	private static final String LABEL = "Label";
	private static final String DEFINITION = "Definition";
	private static final String ALT_DEFINITION = "Alt_Definition";
	private static final String SYNONYMS = "Synonyms";
	private static final String PARENT = "Parent";
	private static final String CHILD = "Child";
	private static final String SOURCE_ONTOLOGY = "Source_Ontology";
	private static final String DATA_ELEMENT_MAPPING = "Data_Element_Mapping";
	private static final String VERSION = "_version_";
	
	//Solr query needs to be escaped
	private static final Pattern SPECIAL_CHARS = Pattern.compile("[{}()\\[\\]:+*?^$\\\\|]");

	@Autowired
	private SolrConceptDocumentConverter documentConverter;

	@Autowired
	private TSPropertiesUtil propsUtil;

	@SuppressWarnings("unused")
	@Override
	public void addTripleToSolrDocument(List<ConceptCardDTO> conceptCards, SolrClient solrClient) {

		Map<String, SolrConceptDocument> documentMap = new HashMap<String, SolrConceptDocument>();
		for (ConceptCardDTO concept : conceptCards) {

			SolrConceptDocument solrDocument = documentConverter.convert(concept);

			if (!documentMap.containsKey(solrDocument.getId())) {
				// check for data element mapping here??
				documentMap.put(solrDocument.getId(), solrDocument);
			} else {
				SolrConceptDocument existingDoc = documentMap.get(solrDocument.getId());

				// coalese with the new document here
				SolrConceptDocument combinedDoc = combineConceptDocument(solrDocument, existingDoc);

				// place the document back into the map
				documentMap.put(combinedDoc.getId(), combinedDoc);
			}
		}
		// here you should have a map that contains batch change of documents
		// this list should have unique ids

		// now on to the the work...

		for (Map.Entry<String, SolrConceptDocument> entry : documentMap.entrySet()) {
			SolrConceptDocument document = entry.getValue();
			SolrQuery solrQuery = new SolrQuery(ID + ":" + 
					solrRegexQuery(document.getId()));

			try {
				QueryResponse response = solrClient.query(solrQuery);
				UpdateResponse updateResponse;

				Optional<SolrDocument> oldDoc = resolve(() -> response.getResults().get(0));

				if (oldDoc.isPresent()) {

					// delete the old guy!
					updateResponse = solrClient.deleteById(document.getId());
					solrClient.commit(); //commit changed immediately

					SolrDocument theOldDoc = oldDoc.get();

					// update the old documents
					theOldDoc.setField(DEFINITION, document.getDefinition());
					theOldDoc.setField(ALT_DEFINITION, document.getAltDefinition());
					theOldDoc.setField(SYNONYMS, document.getSynonyms());
					theOldDoc.setField(PARENT, document.getParent());
					theOldDoc.setField(CHILD, document.getChild());

					updateResponse = solrClient.add(toSolrInputDocument(theOldDoc));

				} else {
					// the document doesnt exist so create a new one and add
					SolrInputDocument newInputDoc = new SolrInputDocument();

					newInputDoc.addField(ID, document.getId());
					newInputDoc.addField(LABEL, document.getLabel());
					newInputDoc.addField(DEFINITION, document.getDefinition());
					newInputDoc.addField(ALT_DEFINITION, document.getAltDefinition());
					newInputDoc.addField(SYNONYMS, document.getSynonyms());
					newInputDoc.addField(PARENT, document.getParent());
					newInputDoc.addField(CHILD, document.getChild());
					newInputDoc.addField(SOURCE_ONTOLOGY, document.getSourceOntology());
					newInputDoc.addField(DATA_ELEMENT_MAPPING, document.getDataElementMapping());

					updateResponse = solrClient.add(newInputDoc);

				}
				updateResponse = solrClient.commit();
			} catch (SolrServerException | IOException e) {
				LOGGER.error(e.getMessage());
			}
		}
	}
	
	/**
	 * Combines new document information into an existing document
	 * @param conceptUris
	 * @param solrClient
	 * @return Old document with the combined new information
	 */
	@Override
	public void deleteTripleToSolrDocument(List<String> conceptUris, SolrClient solrClient) {
		
		UpdateResponse response;
		try {
			 response = solrClient.deleteById(conceptUris
					 .stream()
					 .distinct()
					 .collect(Collectors.toList()));
			 
			 solrClient.commit();
		}catch (SolrServerException | IOException e) {
			LOGGER.error(e.getMessage());
		}
	}

	/**
	 * Combines new document information into an existing document
	 * @param newDoc
	 * @param oldDoc
	 * @return Old document with the combined new information
	 */
	public SolrConceptDocument combineConceptDocument(SolrConceptDocument newDoc, SolrConceptDocument oldDoc) {
		oldDoc.setDefinition(oldDoc.getDefinition() + " " +newDoc.getDefinition());
		oldDoc.setAltDefinition(oldDoc.getAltDefinition() + " " + newDoc.getAltDefinition());
		oldDoc.setSynonyms(oldDoc.getSynonyms() + " " + newDoc.getSynonyms());
		oldDoc.setParent(oldDoc.getParent() + " " + newDoc.getParent());
		oldDoc.setChild(oldDoc.getChild() + " " + newDoc.getChild());

		return oldDoc;
	}

	/**
	 * Resolves a supplier function to check for nulls
	 * @param resolver
	 * @return the optional object or null if found
	 */
	private static <T> Optional<T> resolve(Supplier<T> resolver) {
		try {
			T result = resolver.get();
			return Optional.ofNullable(result);
		} catch (Exception e) {
			return Optional.empty();
		}
	}

	/**
	 * Convert a SorlDocument into an SolrInputDocument for transmission
	 * 
	 * @param solrDoc
	 * @return SolrInputDocument 
	 */
	public SolrInputDocument toSolrInputDocument(SolrDocument solrDoc) {
		SolrInputDocument doc = new SolrInputDocument();

		for (String name : solrDoc.getFieldNames()) {
			doc.addField(name, solrDoc.getFieldValue(name));
		}
		//ignore versioning information for now
		doc.setField(VERSION, 0);
		if( isDataElementPrefix( (String) solrDoc.get(ID)))
			doc.setField(DATA_ELEMENT_MAPPING, true, 100); //adding boost value
		return doc;
	}
	
	public  boolean isDataElementPrefix(String uri) {
		return uri.startsWith(propsUtil.getBaselineSurveyPrefix());
	}
	
	/**
	 * Regex used to escape characters for a solr query
	 * @param query
	 * @return escaped query form
	 */
	private static String solrRegexQuery(String query) {
		return SPECIAL_CHARS.matcher(query).replaceAll("\\\\$0");
	}
}
