package gov.va.genisis2.ts.service.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.jena.query.QueryExecution;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.genisis2.ts.common.dto.TripleDTO;
import gov.va.genisis2.ts.common.exception.TSInernalSystemException;
import gov.va.genisis2.ts.fuseki.FusekiClient;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;

@Component
public class MvpUriGenHelper {
	private static final Logger LOGGER = LogManager.getLogger(MvpUriGenHelper.class);

	@Autowired
	private TSPropertiesUtil propsUtil;

	@Autowired
	private FusekiClient fusekiClient;

	public void populateUrisForNewConcepts(List<TripleDTO> triples) {
		Map<String, String> uriHashMap = new HashMap<>();

		try {
			if (null == triples || triples.isEmpty()) {
				LOGGER.error("triples list is emptry");
			} else {
				for (TripleDTO triple : triples) {
					// replace _?xx in subject
					String value = triple.getS();
					String updatedValue = this.generateUri(value, uriHashMap);
					
					if (null != updatedValue) {
						triple.setS(updatedValue);
					}
					
					// replace _?xx in predicate
					value = triple.getP();
					updatedValue = this.generateUri(value, uriHashMap);
					
					if (null != updatedValue) {
						triple.setP(updatedValue);
					}
					
					// replace _?xx in object
					value = triple.getO();
					updatedValue = this.generateUri(value, uriHashMap);
					
					if (null != updatedValue) {
						triple.setO(updatedValue);
					}
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in populateUrisForNewConcepts", e);
			throw new TSInernalSystemException("Error in populateUrisForNewConcepts", e);
		}

	}

	/**
	 * Adds angular brackets to resources.. This is done because SPARQL needs
	 * resource to be in angular brackets
	 * 
	 * @param uri
	 * @return
	 */
	public String quoteUri(String uri) {
		if (uri.startsWith("<")) {
			return uri;
		}

		return "<" + uri + ">";
	}

	/**
	 * Adds double quotes to literals.. This is done because SPARQL needs literals
	 * to be double quoted
	 * 
	 * @param query
	 * @return
	 */
	public String quoteLiteral(String query) {
		if (query.startsWith("<")) {
			return query;
		}
		return "\"" + query + "\"";
	}

	private String generateUri(String value, Map<String, String> uriHashMap) {
		String updatedValue = null;
		boolean valueHasAngularBracket = false;
		QueryExecution qe = null;

		try {
			// value starts with 'http://genisis.va.gov/mvp-schema'
			if (null != value
					&& ((valueHasAngularBracket = value.trim().startsWith("<") && value.trim().startsWith("<" + propsUtil.getMvpPrefix())) || value.trim().startsWith(propsUtil.getMvpPrefix()))
					&& value.trim().contains(propsUtil.getMvpUriDelimiter())) {
				if (valueHasAngularBracket) {
					value = value.substring(1, value.length() - 1);
				}

				// subject ends with "_?###" . where we accept from 1 to 3 decimal digits after
				// the '?'
				Pattern p = Pattern.compile(propsUtil.getMvpNewConceptRegex());
				Matcher m = p.matcher(value);
				if (m.find()) {
					// is subject exists in URIHashMap
					if (uriHashMap.containsKey(value)) {
						updatedValue = uriHashMap.get(value);
					} else {
						boolean existsInDatabase = true;
						while (existsInDatabase) {
							// generate 4 digit # and update subject with 4 digit #
							updatedValue = value.substring(0, value.indexOf("_?") + 1) + new Random().nextInt(9999);

							// http://genisis.va.gov/mvp-schema#Definition_#### exists in Jena
							String sparqlQuery = String.format(propsUtil.getGeneratedUriExistsSparqlQuery(), updatedValue, updatedValue, updatedValue);
							existsInDatabase = fusekiClient.performAsk(sparqlQuery, propsUtil.getSparqlEndpoint());
						}
						// add to URIHashMap
						uriHashMap.put(value, updatedValue);
						updatedValue = (valueHasAngularBracket) ? "<" + updatedValue + ">" : updatedValue;
					}
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in generateUri", e);
			throw new TSInernalSystemException("Error in generateUri", e);
		} finally {
			if (qe != null && !qe.isClosed()) {
				qe.close();
			}
		}

		return updatedValue;
	}

	// TODO: fix angular brackets hardcoding
	@SuppressWarnings("unused")
	private String getAskForm(TripleDTO triple) {
		if (null == triple) {
			return null;
		} else {
			return "ASK { " + quoteUri(triple.getS()) + " " + quoteUri(triple.getP()) + " " + quoteUri(triple.getO()) + " }";
		}
	}
}
