package gov.va.genisis2.ts.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.ts.common.exception.TSRuntimeException;
import gov.va.genisis2.ts.dto.ConceptCardDTO;
import gov.va.genisis2.ts.dto.ConceptCardMetaDTO;
import gov.va.genisis2.ts.dto.ConceptDescriptionDTO;
import gov.va.genisis2.ts.dto.ConceptPredicateUriValueDTO;
import gov.va.genisis2.ts.dto.ConceptRelationshipDTO;
import gov.va.genisis2.ts.dto.MetaInterOntologyLinkDTO;
import gov.va.genisis2.ts.dto.MetaPropertyDTO;
import gov.va.genisis2.ts.dto.MetaRelationshipDTO;
import gov.va.genisis2.ts.dto.RelationshipClassTypeDTO;
import gov.va.genisis2.ts.dto.SolrDTO;
import gov.va.genisis2.ts.dto.SolrDocDTO;
import gov.va.genisis2.ts.fuseki.FusekiClient;
import gov.va.genisis2.ts.service.IConceptCardService;
import gov.va.genisis2.ts.service.IMappingService;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;
import gov.va.genisis2.ts.utils.Uri2Name;

@Service
public class ConceptCardService implements IConceptCardService {

	private static final Logger LOGGER = LogManager.getLogger(ConceptCardService.class);

	@Autowired
	private FusekiClient fusekiClient;

	@Autowired
	private IMappingService mappingService;

	@Autowired
	private TSPropertiesUtil propsUtil;

	@Autowired
	private ConceptCardHelper conceptCardHelper;

	@Resource(name = "descDefinitonPredicateUriList")
	private List<String> descDefinitonPredicateUriList;

	@Resource(name = "descAltDefinitonPredicateUriList")
	private List<String> descAltDefinitonPredicateUriList;

	@Resource(name = "descSynonymPredicateUriList")
	private List<String> descSynonymPredicateUriList;

	@Resource(name = "relSuperClassPredicateUriList")
	private List<String> relSuperClassPredicateUriList;

	@Resource(name = "metaInterOntologyLinkPredicateUriList")
	private List<String> metaInterOntologyLinkPredicateUriList;

	@Resource(name = "mappingAlgorithmPredicateUriList")
	private List<String> mappingAlgorithmPredicateUriList;

	@Resource(name = "mappingValidationPredicateUriList")
	private List<String> mappingValidationPredicateUriList;

	@Resource(name = "mappingDataElementPredicateUriList")
	private List<String> mappingDataElementPredicateUriList;

	@Resource(name = "guidPredicateUriList")
	private List<String> guidPredicateUriList;

	@Resource(name = "notationUriList")
	private List<String> notationUriList;

	@Resource(name = "cbaseCodeUriList")
	private List<String> cbaseCodeUriList;

	@Resource(name = "commentUriList")
	private List<String> commentUriList;

	@Resource(name = "associatedGenesUriList")
	private List<String> associatedGenesUriList;

	@Resource(name = "questionTextUriList")
	private List<String> questionTextUriList;

	@Resource(name = "questionNumberUriList")
	private List<String> questionNumberUriList;

	@Resource(name = "elementTypeUriList")
	private List<String> elementTypeUriList;
	
	@Resource(name = "typeUriList")
	private List<String> typeUriList;
	
	@Override
	public ConceptCardDTO getConceptCardsOnConceptUri(String conceptUri) {
		if (null == conceptUri) {
			LOGGER.error("conceptUri is null");
			throw new TSRuntimeException("conceptUri is null");
		}

		QueryExecution qe = null;
		ResultSet resultSet = null;
		String conceptName;
		String ontologyName;
		ConceptCardDTO conceptCardDto = new ConceptCardDTO();
		ConceptRelationshipDTO relationships = new ConceptRelationshipDTO();
		List<RelationshipClassTypeDTO> subClassTypes = new ArrayList<>();
		ConceptDescriptionDTO conceptDesc = new ConceptDescriptionDTO();
		List<ConceptPredicateUriValueDTO> definitions = new LinkedList<>();
		List<ConceptPredicateUriValueDTO> altDefinitions = new LinkedList<>();
		List<ConceptPredicateUriValueDTO> synonyms = new ArrayList<>();
		List<RelationshipClassTypeDTO> superClassTypes = new ArrayList<>();
		ConceptCardMetaDTO conceptCardMeta = new ConceptCardMetaDTO();
		List<MetaInterOntologyLinkDTO> metaInterOntologyLinks = new ArrayList<>();
		List<MetaPropertyDTO> metaProperties = new ArrayList<>();
		List<MetaRelationshipDTO> metaRelationships = new ArrayList<>();
		ConceptPredicateUriValueDTO mvpAlgorithm = new ConceptPredicateUriValueDTO();
		ConceptPredicateUriValueDTO mvpValidation = new ConceptPredicateUriValueDTO();
		List<ConceptPredicateUriValueDTO> mappingDataElements = new ArrayList<ConceptPredicateUriValueDTO>();
		List<MetaPropertyDTO> genesAssociatedWithDiseases = new ArrayList<MetaPropertyDTO>();

		try {
			conceptCardDto.setUri(conceptUri);

			// Concept-Name SPARQL query
			String conceptNameSparqlQuery = String.format(propsUtil.getConceptNameSparqlQuery(), conceptUri);
			qe = QueryExecutionFactory.sparqlService(propsUtil.getSparqlEndpoint(), conceptNameSparqlQuery);
			resultSet = fusekiClient.performSelectQuery(qe, propsUtil.getSparqlEndpoint());

			// ResultSetFormatter.out(resultSet);

			if (null != resultSet) {
				// extract concept name from SOLR results
				while (resultSet.hasNext()) {
					QuerySolution binding = resultSet.nextSolution();
					conceptName = conceptCardHelper.getValueOfLiteralOrUri(binding, propsUtil.getConceptNameQueryHeadvarName());
					ontologyName = conceptCardHelper.getOntologyNameForConceptUri(conceptUri);

					conceptCardDto.setName(conceptName);
					conceptCardDto.setNameWithOntologyName(conceptName + " [" + ontologyName + "]");
					conceptCardDto.setOntologyName(ontologyName);
					conceptCardDto.setSource(ontologyName);
					conceptCardMeta.setSource(ontologyName);
					conceptCardMeta.setName(conceptName);
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in execution Concept-Name SPARQL query", e);
			throw e;
		} finally {
			if (qe != null && !qe.isClosed()) {
				qe.close();
			}
		}

		try {
			// Concept-SubType SPARQL query
			String conceptSubTypeSparqlQuery = String.format(propsUtil.getConceptSubtypeSparqlQuery(), conceptUri);
			qe = QueryExecutionFactory.sparqlService(propsUtil.getSparqlEndpoint(), conceptSubTypeSparqlQuery);
			resultSet = fusekiClient.performSelectQuery(qe, propsUtil.getSparqlEndpoint());

			if (null != resultSet) {
				// extract subclass types from SOLR results
				while (resultSet.hasNext()) {
					RelationshipClassTypeDTO subClass = new RelationshipClassTypeDTO();
					QuerySolution binding = resultSet.nextSolution();
					String uri = conceptCardHelper.getValueOfLiteralOrUri(binding, propsUtil.getSubtypeQueryHeadVarChildUri());
					subClass.setUri(uri);
					subClass.setName(conceptCardHelper.getValueOfLiteralOrUri(binding, propsUtil.getSubtypeQueryHeadVarChildName()));
					subClass.setIsMvpPredicate(isMvpPredicate(uri));
					subClassTypes.add(subClass);
				}

				// add subclass types to concept-card
				if (!subClassTypes.isEmpty()) {
					Collections.sort(subClassTypes);
					relationships.setSubClasses(subClassTypes);
					conceptCardDto.setRelationships(relationships);
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in execution Concept-SubType SPARQL query", e);
			throw e;
		} finally {
			if (qe != null && !qe.isClosed()) {
				qe.close();
			}
		}

		try {
			// Concept-Predicate SPARQL query
			String conceptPredicateSparqlQuery = String.format(propsUtil.getConceptPredicateSparqlQuery(), conceptUri, conceptUri);
			qe = QueryExecutionFactory.sparqlService(propsUtil.getSparqlEndpoint(), conceptPredicateSparqlQuery);
			resultSet = fusekiClient.performSelectQuery(qe, propsUtil.getSparqlEndpoint());

			if (null != resultSet) {
				final String SOLR_QR3_HEAD_VAR_P = propsUtil.getConceptPredicateQueryHeadVarP();
				final String SOLR_QR3_HEAD_VAR_O = propsUtil.getConceptPredicateQueryHeadVarO();
				final String SOLR_QR3_HEAD_VAR_N = propsUtil.getConceptPredicateQueryHeadVarN();

				while (resultSet.hasNext()) {
					ConceptPredicateUriValueDTO mvpDataDto;
					org.apache.jena.rdf.model.Resource predicateUriResource;
					ConceptPredicateUriValueDTO predicateUriValue;
					RDFNode predicateUriNode;
					String predicateUriString;
					MetaPropertyDTO metaProperty;

					QuerySolution binding = resultSet.nextSolution();

					if (binding.contains(SOLR_QR3_HEAD_VAR_P)) {
						predicateUriNode = binding.get(SOLR_QR3_HEAD_VAR_P);

						if (null != predicateUriNode && predicateUriNode.isURIResource()) {
							predicateUriResource = predicateUriNode.asResource();

							if (null != predicateUriResource) {
								predicateUriString = predicateUriResource.getURI();

								if (null != predicateUriString) {
									// extract concept description.definition from SOLR results
									if (isPredicateDefinitionUri(predicateUriString)) {
										predicateUriValue = new ConceptPredicateUriValueDTO();
										predicateUriValue.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										predicateUriValue.setPredicateUri(predicateUriString);
										predicateUriValue.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										definitions.add(predicateUriValue);
									} else if (isPredicateAltDefinitionUri(predicateUriString)) {
										// extract concept description.alt-definition from SOLR results
										predicateUriValue = new ConceptPredicateUriValueDTO();
										predicateUriValue.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										predicateUriValue.setPredicateUri(predicateUriString);
										predicateUriValue.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										altDefinitions.add(predicateUriValue);
									} else if (isPredicateSynonymUri(predicateUriString)) {
										// extract concept description.synonym from SOLR results
										predicateUriValue = new ConceptPredicateUriValueDTO();
										predicateUriValue.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										predicateUriValue.setPredicateUri(predicateUriString);
										predicateUriValue.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										synonyms.add(predicateUriValue);
									} else if (isPredicateRelSuperClassUri(predicateUriString)) {
										// extract concept relationships.superClasses from SOLR results
										RelationshipClassTypeDTO superClass = new RelationshipClassTypeDTO();
										superClass.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										superClass.setName(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_N));
										String oValue = conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O);
										if (null != oValue) {
											superClass.setUri(oValue);
											superClassTypes.add(superClass);
										}
									} else if (isPredicateMetaInterOntologyLinkUri(predicateUriString)) {
										// extract concept meta.interOntologyLinks from SOLR results
										MetaInterOntologyLinkDTO metaInterOntologyLink = new MetaInterOntologyLinkDTO();
										metaInterOntologyLink.setName(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_N));

										String interOntologyLinksUri = conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O);
										metaInterOntologyLink.setUri(interOntologyLinksUri);
										metaInterOntologyLink.setOntologyName(Uri2Name.extractOntology(interOntologyLinksUri));
										metaInterOntologyLink.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaInterOntologyLinks.add(metaInterOntologyLink);
									} else if (isPredicateMappingAlgorithmUri(predicateUriString)) {
										mvpAlgorithm.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										mvpAlgorithm.setPredicateUri(predicateUriString);
										mvpAlgorithm.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardMeta.setAlgorithm(mvpAlgorithm);
									} else if (isPredicateMappingValidationUri(predicateUriString)) {
										mvpValidation.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										mvpValidation.setPredicateUri(predicateUriString);
										mvpValidation.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardMeta.setValidation(mvpValidation);
									} else if (isPredicateMappingDataElementUri(predicateUriString)) {
										mvpDataDto = new ConceptPredicateUriValueDTO();
										mvpDataDto.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										mvpDataDto.setPredicateUri(predicateUriString);
										mvpDataDto.setValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										mappingDataElements.add(mvpDataDto);
									} else if (isPredicateGuidUri(predicateUriString)) {
										String guid = conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O);
										conceptCardMeta.setGuid(guid);
										conceptCardDto.setGuid(guid);
									} else if (isPredicateNotationUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setNotation(metaProperty);
									} else if (isPredicateCbaseCodeUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setCbaseCode(metaProperty);
									} else if (isPredicateCommentUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setComment(metaProperty);
									} else if (isPredicateAssociatedGenesUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										genesAssociatedWithDiseases.add(metaProperty);
									} else if (isPredicateQuestionTextUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setQuestionText(metaProperty);
									} else if (isPredicateQuestionNumberUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setQuestionNumber(metaProperty);
									} else if (isPredicateElementTypeUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setElementType(metaProperty);
									} else if (isPredicateTypeUri((predicateUriString))) {
										metaProperty = new MetaPropertyDTO();
										metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
										metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
										metaProperty.setPropertyUri(predicateUriString);
										metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
										conceptCardDto.setType(metaProperty);
									} else {
										// no "n" value - extract concept meta.properties from SOLR results
										if (!binding.contains(SOLR_QR3_HEAD_VAR_N)) {
											metaProperty = new MetaPropertyDTO();
											metaProperty.setIsMvpPredicate(isMvpPredicate(predicateUriString));
											metaProperty.setPropertyName(Uri2Name.extractNameFromUri(predicateUriString));
											metaProperty.setPropertyUri(predicateUriString);
											metaProperty.setPropertyValue(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
											metaProperties.add(metaProperty);
										} else {
											// there is "n" value - extract concept meta.relationships from SOLR results
											MetaRelationshipDTO metaRelationship = new MetaRelationshipDTO();
											metaRelationship.setIsMvpPredicate(isMvpPredicate(predicateUriString));
											metaRelationship.setRelationshipName(Uri2Name.extractNameFromUri(predicateUriString));
											metaRelationship.setRelationshipUri(predicateUriString);
											metaRelationship.setRelatedConcept(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_N));
											metaRelationship.setRelatedConceptUri(conceptCardHelper.getValueOfLiteralOrUri(binding, SOLR_QR3_HEAD_VAR_O));
											metaRelationships.add(metaRelationship);
										}
									}
								}
							}
						}
					}
				}

				Collections.sort(definitions);
				conceptCardMeta.setDefinitions(definitions);
				conceptDesc.setDefinitions(definitions);
				Collections.sort(altDefinitions);
				conceptDesc.setAltDefinitions(altDefinitions);
				Collections.sort(synonyms);
				conceptDesc.setSynonyms(synonyms);

				conceptCardDto.setDescription(conceptDesc);
				conceptCardDto.setDefinitions(definitions);
				conceptCardDto.setAltDefinitions(altDefinitions);
				conceptCardDto.setSynonyms(synonyms);

				Collections.sort(superClassTypes);
				relationships.setSuperClasses(superClassTypes);
				conceptCardDto.setRelationships(relationships);
				conceptCardDto.setSubClasses(subClassTypes);
				conceptCardDto.setSuperClasses(superClassTypes);

				Collections.sort(metaInterOntologyLinks);
				conceptCardMeta.setInterOntologyLinks(metaInterOntologyLinks);

				Collections.sort(metaProperties);
				conceptCardMeta.setProperties(metaProperties);

				Collections.sort(metaRelationships);
				conceptCardMeta.setRelationships(metaRelationships);
				Collections.sort(mappingDataElements);
				conceptCardMeta.setDataElements(mappingDataElements);
				conceptCardMeta.setDataOutcome(StringUtils.EMPTY);
				conceptCardMeta.setDataTransformation(StringUtils.EMPTY);
				conceptCardMeta.setCleaned(StringUtils.EMPTY);
				conceptCardMeta.setCleaningProcedure(StringUtils.EMPTY);
				conceptCardMeta.setDataSource(StringUtils.EMPTY);
				conceptCardMeta.setSensitivity(StringUtils.EMPTY);
				conceptCardMeta.setSpecificity(StringUtils.EMPTY);
				conceptCardMeta.setUnits(StringUtils.EMPTY);
				conceptCardMeta.setValidatedBy(StringUtils.EMPTY);

				conceptCardDto.setMeta(conceptCardMeta);
				conceptCardDto.setInterOntologyLinks(metaInterOntologyLinks);
				conceptCardDto.setProperties(metaProperties);
				conceptCardDto.setMetaRelationships(metaRelationships);
				conceptCardDto.setAlgorithm(mvpAlgorithm);
				conceptCardDto.setValidation(mvpValidation);
				conceptCardDto.setDataElements(mappingDataElements);
				conceptCardDto.setDataOutcome(StringUtils.EMPTY);
				conceptCardDto.setDataTransformation(StringUtils.EMPTY);
				conceptCardDto.setCleaned(StringUtils.EMPTY);
				conceptCardDto.setCleaningProcedure(StringUtils.EMPTY);
				conceptCardDto.setDataSource(StringUtils.EMPTY);
				conceptCardDto.setSensitivity(StringUtils.EMPTY);
				conceptCardDto.setSpecificity(StringUtils.EMPTY);
				conceptCardDto.setUnits(StringUtils.EMPTY);
				conceptCardDto.setValidatedBy(StringUtils.EMPTY);
				
				Collections.sort(genesAssociatedWithDiseases);
				conceptCardDto.setGenesAssociatedWithDiseases(genesAssociatedWithDiseases);
			}
		} catch (Exception e) {
			LOGGER.error("Error in execution Concept-Predicate SPARQL query", e);
			throw e;
		} finally {
			if (qe != null && !qe.isClosed()) {
				qe.close();
			}
		}

		return conceptCardDto;
	}

	@Override
	public void populateConceptName(SolrDTO solrDto) {
		if (null == solrDto || null == solrDto.getResponse() || null == solrDto.getResponse().getDocs()) {
			return;
		} else {
			List<SolrDocDTO> solrDocs = solrDto.getResponse().getDocs();
			for (SolrDocDTO solrDoc : solrDocs) {
				// append Ontology name for each concept
				solrDoc.setLabelWithOntologyName(solrDoc.getLabel() + " [" + conceptCardHelper.getOntologyNameForConceptUri(solrDoc.getId()) + "]");

				// check if a concept has mappings
				solrDoc.setHasMapping(mappingService.hasMapping(solrDoc.getId()));
			}
		}
	}

	@Override
	public void populateBaselineSurveyIndex(SolrDTO solrDto) {
		boolean searchForBaselineSurvey = true;
		int index = 0;
		if (null == solrDto || null == solrDto.getResponse() || null == solrDto.getResponse().getDocs()) {
			return;
		} else {
			List<SolrDocDTO> solrDocs = solrDto.getResponse().getDocs();
			for (SolrDocDTO solrDoc : solrDocs) {
				if (searchForBaselineSurvey && isConceptBaselineSurvey(solrDoc.getId())) {
					index++;
				} else {
					searchForBaselineSurvey = false;
				}
			}
		}
		solrDto.getResponse().setBaselineSurveyIndex(index);
	}

	private boolean isPredicateDefinitionUri(String otherLiteral) {
		return descDefinitonPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateAltDefinitionUri(String otherLiteral) {
		return descAltDefinitonPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateSynonymUri(String otherLiteral) {
		return descSynonymPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateRelSuperClassUri(String otherLiteral) {
		return relSuperClassPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateMetaInterOntologyLinkUri(String otherLiteral) {
		return metaInterOntologyLinkPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateMappingAlgorithmUri(String otherLiteral) {
		return this.mappingAlgorithmPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateMappingValidationUri(String otherLiteral) {
		return this.mappingValidationPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateMappingDataElementUri(String otherLiteral) {
		return this.mappingDataElementPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(otherLiteral)).findFirst().isPresent();
	}

	private boolean isPredicateGuidUri(String conceptUri) {
		return this.guidPredicateUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isPredicateNotationUri(String conceptUri) {
		return this.notationUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isPredicateCbaseCodeUri(String conceptUri) {
		return this.cbaseCodeUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isPredicateCommentUri(String conceptUri) {
		return this.commentUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isPredicateAssociatedGenesUri(String conceptUri) {
		return this.associatedGenesUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isPredicateQuestionTextUri(String conceptUri) {
		return this.questionTextUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}
	
	private boolean isPredicateQuestionNumberUri(String conceptUri) {
		return this.questionNumberUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}
	
	private boolean isPredicateElementTypeUri(String conceptUri) {
		return this.elementTypeUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}
	
	private boolean isPredicateTypeUri(String conceptUri) {
		return this.typeUriList.stream().filter(s -> s.equalsIgnoreCase(conceptUri)).findFirst().isPresent();
	}

	private boolean isConceptBaselineSurvey(String conceptUri) {
		return (null == conceptUri) ? false : conceptUri.toUpperCase().startsWith(propsUtil.getBaselineSurveyPrefix().toUpperCase());
	}

	private boolean isMvpPredicate(String predicateUri) {
		return (null == predicateUri) ? false : predicateUri.toUpperCase().startsWith(propsUtil.getMvpPrefix().toUpperCase());
	}
}
