package gov.va.genisis2.ts.converter;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

import gov.va.genisis2.ts.dto.ConceptCardDTO;
import gov.va.genisis2.ts.dto.ConceptPredicateUriValueDTO;
import gov.va.genisis2.ts.dto.RelationshipClassTypeDTO;
import gov.va.genisis2.ts.service.impl.ConceptCardHelper;
import gov.va.genisis2.ts.service.impl.SolrConceptDocument;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;

/**
 * @author PII
 *
 */
@Component
public class SolrConceptDocumentConverter implements Converter<ConceptCardDTO, SolrConceptDocument>{

	@Autowired
	private ConceptCardHelper conceptCardHelper;
	
	@Autowired
	private TSPropertiesUtil propsUtil;
	
	@Override
	public SolrConceptDocument convert(ConceptCardDTO conceptCardDto) {
		SolrConceptDocument document = new SolrConceptDocument();
		
		document.setId(conceptCardDto.getUri());
		document.setLabel(conceptCardDto.getName());
		
		document.setDefinition(
				String.join(" ", getValuesFromConceptPredicate(
						Optional.ofNullable( conceptCardDto
						.getDescription()
						.getDefinitions()))));
		
		document.setAltDefinition(
				String.join(" ", getValuesFromConceptPredicate(
						Optional.ofNullable( conceptCardDto
						.getDescription()
						.getAltDefinitions()))));
		
		document.setSynonyms(String.join(" ", getValuesFromConceptPredicate(
				Optional.ofNullable( conceptCardDto
				.getDescription()
				.getSynonyms()))));
		
		document.setParent(
				String.join(" ", getValuesFromConceptRelationship(
						Optional.ofNullable( conceptCardDto
						.getRelationships()
						.getSuperClasses()))));
		
		document.setChild(
				String.join(" ", getValuesFromConceptRelationship(
						Optional.ofNullable( conceptCardDto
						.getRelationships()
						.getSubClasses()))));
		
		document.setSourceOntology(
				conceptCardHelper
				.getOntologyNameForConceptUri(
						conceptCardDto.getUri()));
		/** 
		 * For build 7 it is necessary, to boost only mvp data elements
		 */
		if( isDataElementPrefix(conceptCardDto.getUri()))
			document.setDataElementMapping(true);
		
		return document;
	}
	
	/**
	 * Checks for an appropirate prefix for a give concept uri
	 * @param uri
	 * @return
	 */
	public boolean isDataElementPrefix(String uri) {
		return uri.startsWith(propsUtil.getBaselineSurveyPrefix());
	}

	public List<String> getValuesFromConceptPredicate( Optional<List<ConceptPredicateUriValueDTO>> predicateValues ) {
		if(predicateValues.isPresent() ) {
			return predicateValues
					.get()
					.stream()
					.map(ConceptPredicateUriValueDTO::getValue)
					.collect(Collectors.toList());
		}
		return Arrays.asList("");
	}
	
	public List<String> getValuesFromConceptRelationship(Optional<List<RelationshipClassTypeDTO>> relationshipValues ){
		if(relationshipValues.isPresent() ) {
			return relationshipValues
					.get()
					.stream()
					.map(RelationshipClassTypeDTO::getName)
					.collect(Collectors.toList());
		}
		return Arrays.asList("");
	}
}
