package gov.va.genisis2.bs.controller;

import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.owasp.esapi.ESAPI;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import gov.va.genisis2.bs.service.IBusinessService;
import gov.va.genisis2.ts.common.dto.BookmarkDTO;
import gov.va.genisis2.ts.common.dto.LabelDTO;
import gov.va.genisis2.ts.common.enums.ErrorEnum;
import gov.va.genisis2.ts.common.exception.TSDuplicateDataException;
import gov.va.genisis2.ts.common.exception.TSInernalSystemException;
import gov.va.genisis2.ts.common.exception.TSNoDataFoundException;

/**
 * Bookingmarking Service Controller. Main point to the bookmarking service
 * 
 * @author PII
 * @author PII
 *
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/")
@EnableJpaRepositories("gov.va.genisis2.bs.data.repository")
public class BookmarkingServiceController {

	private static final Logger LOGGER = LogManager.getLogger(BookmarkingServiceController.class);

	@Autowired
	private IBusinessService businessService;

	@RequestMapping(value = "/bookmarks", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity<List<BookmarkDTO>> getBookmarks() {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getBookmarks");
		}
		List<BookmarkDTO> bookmarkingDto = null;
		try {
			bookmarkingDto = businessService.getBookmarks();
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		if (null == bookmarkingDto) {
			LOGGER.error("bookmarkingDto is null");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<List<BookmarkDTO>>(bookmarkingDto, HttpStatus.OK);
		}
	}
	
//	@RequestMapping(value = "/bookmarks/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
//	public ResponseEntity<BookmarkDTO> getBookmark(@PathVariable Integer id) {
//		if (LOGGER.isInfoEnabled()) {
//			LOGGER.info("getBookmark");
//		}
//		BookmarkDTO bookmarkingDto = null;
//		try {
//			bookmarkingDto = businessService.getBookmark(id);
//		} catch (Exception e) {
//			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
//			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
//		}
//
//		if (null == bookmarkingDto) {
//			LOGGER.error("bookmarkingDto is null");
//			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
//		} else {
//			return new ResponseEntity<BookmarkDTO>(bookmarkingDto, HttpStatus.OK);
//		}
//	}

	@RequestMapping(value = "/bookmarks", method = RequestMethod.POST, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	public ResponseEntity<BookmarkDTO> createBookmark(@RequestBody BookmarkDTO bookmarkDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Create Bookmark ");
		}
		// TODO: Look into changing throw new catch paradigm..stack traces are expensive
		BookmarkDTO bookmarkingDto = null;
		try {
			bookmarkingDto = businessService.createBookmark(bookmarkDto);
		} catch (TSDuplicateDataException ex) {
			throw ex;
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		if (null == bookmarkingDto) {
			LOGGER.error("bookmarkingDto is null");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<BookmarkDTO>(bookmarkingDto, HttpStatus.CREATED);
		}
	}

	@RequestMapping(value = "/bookmarks/{id}", method = RequestMethod.DELETE, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	public ResponseEntity<Integer> deleteBookmark(@PathVariable Integer id) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Delete Bookmark ");
		}

		try {
			businessService.deleteBookmark(id);
		} catch (TSNoDataFoundException ex) {
			throw ex;
		}
		catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSNoDataFoundException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
		}
		return new ResponseEntity<Integer>(id, HttpStatus.OK);
	}

	@RequestMapping(value = "/bookmarks/{id}/labels", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity<BookmarkDTO> labelBookmarkedConcept(@PathVariable Integer id, @RequestBody LabelDTO labelDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Add label to bookmark");
		}
		BookmarkDTO returnedBookmark = null;
		try {
			// set the bookmark id to the label
			labelDto.setBookmarkId(id); // check for null here

			// add the label to the bookmark
			returnedBookmark = businessService.labelBookmark(labelDto);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}
		if (returnedBookmark == null) {
			LOGGER.error("Label could not be added");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<BookmarkDTO>(returnedBookmark, HttpStatus.CREATED);
		}
	}

	@RequestMapping(value = "/bookmarks/{id}/labels", method = RequestMethod.DELETE, consumes = "application/json", produces = "application/json")
	public ResponseEntity<BookmarkDTO> unLabelBookmarkedConcept(@PathVariable Integer id, @RequestParam(value = "labelName", required = true) String labelName) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Unlabel a bookmarked concept");
		}
		BookmarkDTO returnedBookmarkDto = null;
		try {
			// Create a labelDTO with string and id passed in
			LabelDTO l = new LabelDTO();
			l.setBookmarkId(id);
			l.setName(labelName);

			// add the label to the bookmark
			returnedBookmarkDto = businessService.unlabelBookmark(l);
		} catch (TSNoDataFoundException ex) {
			throw ex;
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}
		if (returnedBookmarkDto == null) {
			LOGGER.error("Label could not be added");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<BookmarkDTO>(returnedBookmarkDto, HttpStatus.OK);
		}
	}

	@RequestMapping(value = "/bookmarks/{username:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity<List<BookmarkDTO>> retreiveBookmarksByUsername(@PathVariable String username) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Retreive bookmared concepts with labels for given username");
		}

		List<BookmarkDTO> bookmarkDto = null;
		try {
			// add the label to the bookmark
			bookmarkDto = businessService.fetchAllBookmarksByUsername(username);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}
		if (bookmarkDto == null) {
			LOGGER.error("No bookmarks found");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<List<BookmarkDTO>>(bookmarkDto, HttpStatus.OK);
		}
	}

	@RequestMapping(value = "/labels/{username:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity<List<LabelDTO>> retreiveLabelsByUsername(@PathVariable String username) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Retreive labels for given username");
		}

		List<LabelDTO> labelDto = null;
		try {
			labelDto = businessService.fetchAllLabelsByUsername(username);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}
		if (labelDto == null) {
			LOGGER.error("No bookmarks found");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<List<LabelDTO>>(labelDto, HttpStatus.OK);
		}
	}

	@RequestMapping(value = "/bookmarks/labels/{label}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity<List<BookmarkDTO>> retreiveBookmarksByLabel(@PathVariable String label) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Retreive bookmared concepts with specific label");
		}

		List<BookmarkDTO> bookmarkDto = null;
		try {
			bookmarkDto = businessService.fetchAllBookmarksByLabel(label);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		if (bookmarkDto == null) {
			LOGGER.error("Bookmarks could not be found");
			throw new TSNoDataFoundException(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage());
		} else {
			return new ResponseEntity<List<BookmarkDTO>>(bookmarkDto, HttpStatus.OK);
		}
	}

	/**
	 * No data found exception.
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSNoDataFoundException.class })
	public ResponseEntity<Object> exceptionHandler(TSNoDataFoundException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML( ex.getMessage( )), HttpStatus.NOT_FOUND);
	}

	/**
	 * Duplicate data exception for create API calls
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSDuplicateDataException.class })
	public ResponseEntity<Object> exceptionHandler(TSDuplicateDataException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML( ex.getMessage( )), HttpStatus.BAD_REQUEST);
	}

	/**
	 * Internal System Error exception
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSInernalSystemException.class })
	public ResponseEntity<Object> exceptionHandler(TSInernalSystemException ex) {
		return new ResponseEntity<Object>(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
	}
}