import { Injectable } from '@angular/core';
import { NgRedux } from 'ng2-redux';
import { IAppState } from '../store';
import { Http, Response, Headers, RequestOptions } from '@angular/http';
import { environment } from '../../environments/environment';
import { ApiServices } from './api-services';
import { ConceptServices } from '../services/concept-services';
import { BookmarkServices } from '../services/bookmark-services';


import {
  SET_ADDING_DEFINITION_MVP_STATE,
  SET_ADDING_ALT_DEFINITION_MVP_STATE,
  SET_ADDING_SYNONYM_MVP_STATE,
  RESET_ADD_MVP_STATE,
  SET_CARD_DATA,
  CONCEPT_CARD_MVP_ADDED,
  CONCEPT_CARD_MVP_ADDING,
  DISPLAY_CONCEPT_CARD_DETAIL,
  SET_ADDING_SUPER_CLASS_MVP_STATE,
  SET_ADDING_SUB_CLASS_MVP_STATE,
  SET_ADDING_INTER_ONTOLOGY_MVP_STATE,
  SET_ADDING_PROPERTY_MVP_STATE,
  SET_ADDING_RELATIONSHIP_MVP_STATE,
  SET_ADDING_MAPPING_DATA_ELEMENT_MVP_STATE,
  SET_ADDING_MAPPING_ALGORITHM_MVP_STATE,
  SET_ADDING_MAPPING_VALIDATION_MVP_STATE,
  FETCH_BOOKMARKS,
  UPDATE_BOOKMARKED_MAPPING,
  FOCUS_ON,
} from '../actions';
@Injectable()
export class MvpServices {
  baseUri;
  mvpPrefix = 'http://URL/mvp-schema#';
  rdfPrefix = 'http://www.w3.org/2000/01/rdf-schema#'; 
  constructor(
    private ngRedux: NgRedux<IAppState>,
    private apiServices: ApiServices,
    private conceptServices: ConceptServices,
    private bookmarkServices: BookmarkServices,
    private http: Http) {
  }

  getState() {
    return this.ngRedux.getState();
  }

  // Master Service Event Handler function
  enableMvpAddFor(type = null, tabType) {
    switch(type) {
      case 'definition': 
        // do what ever redux stuff needed for definition
        this.ngRedux.dispatch({
          type: SET_ADDING_DEFINITION_MVP_STATE
        });
        break;
      case 'alt-definitions':
        this.ngRedux.dispatch({
          type: SET_ADDING_ALT_DEFINITION_MVP_STATE
        });
        break;
      case 'synonyms':
        this.ngRedux.dispatch({
          type: SET_ADDING_SYNONYM_MVP_STATE
        });
        break;
      case 'super-classes':
        this.ngRedux.dispatch({
          type: SET_ADDING_SUPER_CLASS_MVP_STATE
        });
        break;
      case 'sub-classes':
        this.ngRedux.dispatch({
          type: SET_ADDING_SUB_CLASS_MVP_STATE
        });
        break;
      case 'inter-ontologies':
        this.ngRedux.dispatch({
          type: SET_ADDING_INTER_ONTOLOGY_MVP_STATE
        });
        break;
      case 'properties':
        this.ngRedux.dispatch({
          type: SET_ADDING_PROPERTY_MVP_STATE
        });
        break;
      case 'relationships':
        this.ngRedux.dispatch({
          type: SET_ADDING_RELATIONSHIP_MVP_STATE
        });
        break;
      case 'data-elements':
        this.ngRedux.dispatch({
          type: SET_ADDING_MAPPING_DATA_ELEMENT_MVP_STATE
        });
        break;
      case 'algorithm':
        this.ngRedux.dispatch({
          type: SET_ADDING_MAPPING_ALGORITHM_MVP_STATE
        });
        break;
      case 'validation':
        this.ngRedux.dispatch({
          type: SET_ADDING_MAPPING_VALIDATION_MVP_STATE
        });
        break;
      default: 
        break;
      
    }
    this.focusOn(tabType);
  }

  resetMvpAddState(tabType) {
    this.ngRedux.dispatch({
      type: RESET_ADD_MVP_STATE
    });
    this.ngRedux.dispatch({
      type: FOCUS_ON,
      payload: tabType
    });
    
  }

  focusOn(tabType) {
    if (tabType != null) {
      this.ngRedux.dispatch({
        type: FOCUS_ON,
        payload: tabType
      });
    }
  }

  // Master Service Function
  addMvpConcept(options = {type: null}, tabType) {
    this.ngRedux.dispatch({
      type: CONCEPT_CARD_MVP_ADDING
    });
    switch(options.type) {
      case 'definition':
        this.executeAddingDefinition(options, tabType);
        break;
      case 'alt-definitions':
        this.executeAddingAltDefinitions(options, tabType);
        break;
      case 'synonyms':
        this.executeAddingSynonyms(options, tabType);
        break;
      case 'super-classes':
        this.executeAddingSuperClasses(options, tabType);
        break;
      case 'sub-classes':
        this.executeAddingSubClasses(options, tabType);
       break;
      case 'inter-ontologies':
        this.executeAddingInterOntologies(options, tabType);
        break;
      case 'properties':
        this.executeAddingProperties(options, tabType);
        break;
      case 'relationships':
        this.executeAddingRelationships(options, tabType);
        break;
      case 'data-elements':
        this.executeAddingMappingDataElements(options, tabType);
        break;
      case 'algorithm':
        this.executeAddingMappingAlgorithm(options, tabType);
        break;
      case 'validation':
        this.executeAddingMappingValidation(options, tabType);
        break;
      default:
        break;
    }
  }

  // Description tab sections can be consolidated to a switch statement since
  // everything after the triple creation is the same...
  executeAddingDefinition(options, tabType) {
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>', 
        p: '<' + this.mvpPrefix + 'Definition' + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  executeAddingAltDefinitions(options, tabType) {
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>', 
        p: '<' + this.mvpPrefix + 'AltDefinition' + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  executeAddingSynonyms(options, tabType) {
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>', 
        p: '<' + this.mvpPrefix + 'Synonym' + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  executeAddingSuperClasses(options, tabType) {
    let triple = {
        properties: [{
          s: '<' + this.getCurrentConceptUri() + '>',
          p: '<' + this.mvpPrefix + 'SubClassOf' + '>',
          o: '<' + options.object + '>'
        }]
      };
    this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        const rdata = JSON.parse(response.text());
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  executeAddingSubClasses(options, tabType) {
    var conceptNameUri = options.input.replace(/ /gi, '_');
    let triple = {
      properties: [{
        s: '<' + this.mvpPrefix + conceptNameUri + '_?01>',
        p: '<' + this.rdfPrefix + 'label>',
        o: options.input
      },
      {
        s: '<' + this.mvpPrefix + conceptNameUri + '_?01' + '>',
        p: '<' + this.mvpPrefix + 'SubClassOf' + '>',
        o: '<' + this.getCurrentConceptUri() + '>'
      }]
    };
    this.createAddTripleHttpRequest(triple)
    .subscribe((response: Response) => {
      if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
      }
    });
  }

  executeAddingInterOntologies(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: '<' + this.mvpPrefix + 'InterOntologyLink' + '>',
        o: '<' + options.object + '>'
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }
  
  executeAddingProperties(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: '<' + options.predicate + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }
  
  executeAddingRelationships(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: options.predicate,
        o: '<' + options.object + '>'
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  executeAddingMappingDataElements(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: '<' + this.mvpPrefix + 'MappingDataElement' + '>',
        o: '<' + options.object + '>'
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
          this.ngRedux.dispatch({
            type: UPDATE_BOOKMARKED_MAPPING,
            payload: {
              mappingFlag: true,
              conceptUri: this.getCurrentConceptUri()
            }
          });
        }
      });
  }

    executeAddingMappingAlgorithm(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: '<' + this.mvpPrefix + 'MappingAlgorithm' + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

    executeAddingMappingValidation(options, tabType){
    let triple = {
      properties: [{
        s: '<' + this.getCurrentConceptUri() + '>',
        p: '<' + this.mvpPrefix + 'MappingValidation' + '>',
        o: options.object
      }]
    };
    return this.createAddTripleHttpRequest(triple)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          this.conceptServices.reloadConceptCard(tabType);
        }
      });
  }

  getCurrentConceptUri() {
    return this.getState().conceptCard.conceptCardData.uri;
  }

  //http request to add triple
  createAddTripleHttpRequest(body) {
    const headers = this.apiServices.getHeader();
    const options = new RequestOptions({headers: headers});
    return this.http.post(this.apiServices.apiUrl() + '/update', body, options);
  }

  getPredicateLabelList() {
    let predicateList: Array<any> = 
       [
        {label: 'MVP-AltDefinition', uri: 'http://URL/mvp-schema#AltDefinition'},
        {label: 'MVP-Cleaned', uri: 'http://URL/mvp-schema#Cleaned'},
        {label: 'MVP-CleaningProcedure', uri: 'http://URL/mvp-schema#CleaningProcedure'},
        {label: 'MVP-DataOutcome', uri: 'http://URL/mvp-schema#DataOutcome'},
        {label: 'MVP-DataSource', uri: 'http://URL/mvp-schema#DataSource'},
        {label: 'MVP-DataTransform', uri: 'http://URL/mvp-schema#DataTransform'},
        {label: 'MVP-Definition', uri: 'http://URL/mvp-schema#Definition'},
        {label: 'MVP-ElementType', uri: 'http://URL/mvp-schema#ElementType'},
        {label: 'MVP-GUID', uri: 'http://URL/mvp-schema#GUID'},
        {label: 'MVP-InterOntologyLink', uri: 'http://URL/mvp-schema#InterOntologyLink'},
        {label: 'MVP-MappingAlgorithm', uri: 'http://URL/mvp-schema#MappingAlgorithm'},
        {label: 'MVP-DataElement', uri: 'http://URL/mvp-schema#DataElement'},
        {label: 'MVP-MappingValidation', uri: 'http://URL/mvp-schema#MappingValidation'},
        {label: 'MVP-Source', uri: 'http://URL/mvp-schema#Source'},
        {label: 'MVP-Sensitivity', uri: 'http://URL/mvp-schema#Sensitivity'},
        {label: 'MVP-Specificity', uri: 'http://URL/mvp-schema#Specificity'},
        {label: 'MVP-SubClassOf', uri: 'http://URL/mvp-schema#SubClassOf'},
        {label: 'MVP-Synonym', uri: 'http://URL/mvp-schema#Synonym'},
        {label: 'MVP-Units', uri: 'http://URL/mvp-schema#Units'},
        {label: 'MVP-ValidatedBy', uri: 'http://URL/mvp-schema#ValidatedBy'}];

    let labelPredicateObj: any = {label: 'RDF-Label', uri: this.rdfPrefix + 'label'};
    const newPredicateList = predicateList.concat();
    newPredicateList.push(labelPredicateObj);
    return this.hasPropertyPredicate() ? predicateList : newPredicateList;
  }

  hasPropertyPredicate() {
    var propertiesList = this.ngRedux.getState().conceptCard.conceptCardData.meta.properties;
    if (propertiesList !== null && propertiesList !== undefined && propertiesList.length > 0) {
        return propertiesList.find(propertyObj => propertyObj.propertyUri === (this.rdfPrefix + 'label')) != undefined;
    } else {
        return false;
    }
  }

  getMvpBookmarkedConceptList(){
    const conceptCardUri = this.getState().conceptCard.conceptCardData.uri;
    const bookmarkedConceptList = this.getState().bookmark.bookmarks;
    const newArray = bookmarkedConceptList.filter(
      bookmark => bookmark.conceptUri !== conceptCardUri);
    if (newArray != null && newArray.length > 0) {
      return newArray;
    } else {
      if (bookmarkedConceptList.length == 1 && bookmarkedConceptList[0].conceptUri === conceptCardUri) {
        return null;
      } else {
        return bookmarkedConceptList;
      }
    }
  }

  isConceptMvp(){
    //return this.getState().conceptCard.conceptCardData.uri != null ? this.getState().conceptCard.conceptCardData.uri.indexOf(this.mvpPrefix) > -1 : false
    return true;
  }


}