import { Component, OnInit } from '@angular/core';
import { NgRedux, select } from 'ng2-redux';
import { IAppState } from '../../store';
import { Response } from '@angular/http';

import { ConceptServices } from '../../services/concept-services';
import { BookmarkServices } from '../../services/bookmark-services';
import { SystemServices } from '../../services/system-services';
import {
  UPDATE_CONCEPT_CARD_TAB_SELECTION,
  PREVIOUS_CONCEPT_HISTORY,
  NEW_CONCEPT_HISTORY,
  CLEAR_CONCEPT_CARD_STATE,
  CONCEPT_CARD_POPULATING,
  SET_CARD_DATA,
  SET_CURRENT_CONCEPT_SELECTED,
  CONCEPT_CARD_POPULATED,
  DISABLE_CONCEPT_CARD_VIEW,
  ADD_BOOKMARK,
  SET_CURRENT_BOOKMARK_SELECTED,
  REMOVE_BOOKMARK,
  UPDATE_FILTERED_LIST,
  RESET_ADD_MVP_STATE,
  FOCUS_ON,
} from '../../actions';

@Component({
  selector: 'app-concept-card',
  templateUrl: './concept-card.component.html',
  styleUrls: ['./concept-card.component.scss']
})

export class ConceptCardComponent implements OnInit {
  @select(['system', 'focusedElement']) focusedElement;
  @select(['conceptCard', 'conceptCardData', 'description']) description;
  @select(['conceptCard', 'conceptCardData', 'relationships']) relationships;
  @select(['conceptCard', 'conceptCardData', 'meta']) meta;
  @select(['conceptCard', 'conceptCardData', 'mvp']) mvp;
  @select(['conceptCard', 'cardPopulated']) cardPopulated;
  constructor(
    private conceptServices: ConceptServices,
    private bookmarkServices: BookmarkServices,
    private ngRedux: NgRedux<IAppState>,
    private systemServices: SystemServices) { 
    this.conceptServices = conceptServices;
    this.bookmarkServices = bookmarkServices;
  }
  ngOnInit() {
  }

  getState() {
    return this.ngRedux.getState();
  }

  setTabSelection(selection) {
    this.ngRedux.dispatch({
      type: UPDATE_CONCEPT_CARD_TAB_SELECTION,
      payload: selection
    });
    this.ngRedux.dispatch({
          type: RESET_ADD_MVP_STATE
    });
  }

  getSelectedTab() {
    return this.getState().conceptCard.conceptCardState.selectedTab;
  }

  isDescriptionTab() {
    return this.getSelectedTab() === 'description';
  }

  isRelationshipsTab() {
    return this.getSelectedTab() === 'relationships';
  }

  isMetadataTab() {
    return this.getSelectedTab() === 'metadata';
  }

  isMVPTab() {
    return this.getSelectedTab() === 'mvp';
  }

  previousConcept() {
    if (this.getState().conceptCard.conceptHistory.length > 0) {
      this.ngRedux.dispatch({
        type: PREVIOUS_CONCEPT_HISTORY
      });
      this.ngRedux.dispatch({
        type: CONCEPT_CARD_POPULATING
      });
      const currentHistory = this.getState().conceptCard.conceptHistory;
      const previous = currentHistory[currentHistory.length - 1];
      if (currentHistory.length > 0) {
        this.ngRedux.dispatch({
          type: SET_CARD_DATA,
          payload: previous
        });
        this.ngRedux.dispatch({
          type: SET_CURRENT_CONCEPT_SELECTED,
          payload: previous
        });
        this.ngRedux.dispatch({
          type: CONCEPT_CARD_POPULATED
        });
      } else {
        this.ngRedux.dispatch({
          type: CLEAR_CONCEPT_CARD_STATE
        });
      }
    } else {
      this.ngRedux.dispatch({
        type: CLEAR_CONCEPT_CARD_STATE
      });
    }
    this.ngRedux.dispatch({
          type: RESET_ADD_MVP_STATE
    });
    if (this.canPreviousConcept()) {
      this.ngRedux.dispatch({
      type: FOCUS_ON,
      payload: 'disabledBackIcon'
    })
    }
  }

  bookmarkConcept() {
    this.bookmarkServices
    .addBookmark()
    .subscribe((response: Response) => {
      if (response.status === 201) { //changed check to look for 201 response instead of a 200
        setTimeout(() => {
          const rdata = JSON.parse(response.text());
          this.ngRedux.dispatch({
            type: ADD_BOOKMARK,
            payload: rdata.response
          });
          this.ngRedux.dispatch({
            type: FOCUS_ON,
            payload: 'unbookmarkIcon'
          });
          this.systemServices.postMessageUpdate();
        }, 1000);
      }
    });
  }

  unBookmarkConcept() {
    const bookmark = this.getBookmarkFromCurrentConcept();
    this.bookmarkServices
      .removeBookmark(bookmark.bookmarkId)
      .subscribe((response: Response) => {
        if (response.status === 200) {
          setTimeout(() => {
            const rdata = JSON.parse(response.text());
            this.ngRedux.dispatch({
               type: SET_CURRENT_BOOKMARK_SELECTED,
               payload: bookmark
            });
            this.ngRedux.dispatch({
              type: REMOVE_BOOKMARK,
              payload: rdata.response
            });
            this.ngRedux.dispatch({
              type: UPDATE_FILTERED_LIST
            });
            this.ngRedux.dispatch({
            type: FOCUS_ON,
            payload: 'bookmarkIcon'
            });
            this.systemServices.postMessageUpdate();
          }, 1000);
        }
      });
  }

  conceptCardPopulated() {
    const state = this.getState().conceptCard;
    if (
      state.cardPopulated &&
      state.conceptHistory.length > 0 &&
      !state.mvpRemoving &&
      !state.mvpAdding) {
      return true;
    } else {
      return false;
    }
  }

  conceptCardPopulatingMessage() {
    const state = this.getState();
    const conceptCard = state.conceptCard;
    const conceptSearch = state.conceptSearch;
    if (conceptCard.cardPopulating) {
      return 'Populating card...please wait.'
    } else if (conceptCard.mvpRemoving) {
      return 'Removing MVP...please wait.'
    } else if (conceptCard.mvpAdding) {
      return 'Adding MVP...please wait.'
    } else if (conceptCard.cardPopulated && !(conceptSearch.selectedConcept.length > 0)) {
      return 'Concept card could not be populated. Please try again.'
    } else {
      return 'Please search and select concept.'
    }
  }

  getCurrentConcept() {
    return this.getState().conceptSearch.selectedConcept;
  }

  retrieveConcept() {
  }

  disableConceptCard() {
    this.ngRedux.dispatch({
      type: DISABLE_CONCEPT_CARD_VIEW
    });
    this.ngRedux.dispatch({
      type: FOCUS_ON,
      payload: 'conceptCardNavIcon'
    })
  }

  isActiveTab(description) {
    var activeClass= 'nav-link active';
    var nonactiveClass= 'nav-link';
    if (description == this.getState().conceptCard.conceptCardState.selectedTab) {
      return { state : activeClass};
    } else {
      return {state: nonactiveClass};
    }
  }

  canPreviousConcept() {
    const conceptHistory = this.getState().conceptCard.conceptHistory;
    return conceptHistory.length > 1;
  }

  isBookmarked() {
    // this needs to change in mvp add
    const bookmarks = this.getState().bookmark.bookmarks;
    if (bookmarks.length > 0) {
      for (let bookmark of bookmarks) {
        if (bookmark.conceptUri === this.getCurrentConcept().uri) {
          return true;
        }
      }
    } else {
      return false;
    }
    return false;
  }

  getBookmarkFromCurrentConcept() {
    const bookmarks = this.getState().bookmark.bookmarks;
    for (let bookmark of bookmarks) {
      if (bookmark.conceptUri === this.getCurrentConcept().uri) {
        return bookmark;
      }
    }
    return null;
  }
}
