package gov.va.genisis2.dao.impl;

import gov.va.genisis2.dao.IRequestDao;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Join;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Projections;
import org.hibernate.query.Query;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.orm.hibernate5.HibernateTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class RequestDao.
 *
 */
@Repository
@Transactional(value="transactionManager")
public class RequestDao implements IRequestDao {
	
	/** The Logger. */
	private final org.slf4j.Logger Logger = LoggerFactory
			.getLogger(RequestDao.class);

	
	/** The hibernate template. */
	@Autowired
	@Qualifier("hibernateTemplate")
	private HibernateTemplate hibernateTemplate;
	
	
	
	/**
	 * Gets the hibernate template.
	 *
	 * @return the hibernate template
	 */
	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}



	/**
	 * Sets the hibernate template.
	 *
	 * @param hibernateTemplate the new hibernate template
	 */
	@Autowired
	public void setHibernateTemplate(HibernateTemplate hibernateTemplate) {
		this.hibernateTemplate = hibernateTemplate;
	}


	
	
	
	public int createRequest(final Request entity) {
		int id = 0;
		
		if (entity != null )
		entity.setId(this.maxRowValue());
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		EntityTransaction transaction = null;
		try {
			transaction = entityManager.getTransaction();
			transaction.begin();
			entityManager.persist(entity);
			transaction.commit();
			Logger.info("RequestDao:  Request Created.");
			id = entity.getId();
		} catch (RuntimeException e) {
			Logger.error("RequestDao:  Request Creation failed. Transaction Rolled back.");
			if (transaction != null)
				transaction.rollback();
		} finally {
			entityManager.close();
		}
		return id;
	}


	public List<Request> getRequestsByUID(String uid) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder
				.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		Predicate condition1 = criteriaBuilder.equal(root.get("createdBy"), uid);
	    Predicate condition2 = criteriaBuilder.equal(root.get("approverId"), uid);
	  //  Predicate condition3 = criteriaBuilder.notEqual(root.get("statusDescription"), "Draft");
	    Predicate conditions = criteriaBuilder.or(condition1, condition2 );
	    query.where(conditions);
	    
	    List<Request> requestList = entityManager
				.createQuery(query).getResultList();
		return requestList;
	}


	public Request getRequestsById(int id) {
		Request request = (Request) hibernateTemplate.get(
				Request.class, id);
		System.out.println("-------------*************-------------"+request.getId());
		return request;
	}

	
	public List<Request> getAllRequests() {
		List<Request> requestList = (List<Request>) hibernateTemplate
				.find("from Request");
		return requestList;
	}
	
	public List<Request> getAllRequestsByStudyApproval(int studyApprovalId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder
				.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get("studyApproval"), studyApprovalId));
		query.select(root);
		List<Request> requestList = entityManager
				.createQuery(query).getResultList();
		return requestList;
	}

	public List<Request> getAllRequestsByCommentHistory(int commentHistoryId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder
				.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get("commentHistory"), commentHistoryId));
		query.select(root);
		List<Request> requestList = entityManager
				.createQuery(query).getResultList();
		return requestList;
	}

	public List<Request> getAllRequestsByStaus(String status) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder
				.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get("statusDescription"), status));
		query.select(root);
		List<Request> requestList = entityManager
				.createQuery(query).getResultList();
		return requestList;
	}

	public int submitOrModify(Request entity) {
		hibernateTemplate.saveOrUpdate(entity);
		return entity.getId();
	}

	public RequestType getRequestType(int id) {
		RequestType requestType = (RequestType) hibernateTemplate.get(
				RequestType.class, id);
		return requestType;
	}

	public Source getSource(int id) {
		Source source = (Source) hibernateTemplate.get(Source.class, id);
		return source;
	}
	
	/**
	 * Max row value.
	 *
	 * @return the int
	 */
	@SuppressWarnings("unchecked")
	private int maxRowValue() {
		int maxrowVal =1;
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder
				.createQuery(Request.class);
		Root root = criteriaQuery.from(Request.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("id")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();
		if (result == null)
			return maxrowVal;
		else
		return result.intValue() +1 ;
	}

	public int submitOrModify(RequestHistory entity) {
		int historyLineNo=0;
		int maxRowHistory=0;
		
		if (StringUtils.equals(entity.getStatusDescription(), "Draft")) {
			historyLineNo = this.historyLineForRequestHistory(entity
					.getRequest().getId());
			if (historyLineNo > 0) {
				entity.setHistLineNo(historyLineNo);
			} else {
				maxRowHistory = maxRowValueHistory();
				entity.setHistLineNo(maxRowHistory);
			}

		} else {
			maxRowHistory = maxRowValueHistory();
			entity.setHistLineNo(maxRowHistory);
		}

		hibernateTemplate.saveOrUpdate(entity);
		return entity.getHistLineNo();
	}

	/**
	 * Max row value history.
	 *
	 * @return the int
	 */
	@SuppressWarnings("unchecked")
	private int maxRowValueHistory() {
		int maxRowVal =1;
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder
				.createQuery(RequestHistory.class);
		Root root = criteriaQuery.from(RequestHistory.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("histLineNo")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();
		
		if (result != null) {
			maxRowVal=result.intValue() +1 ;
		}
	/*	if (result == null)
			return maxrowVal;
		else
		return result.intValue() +1 ;*/
		return maxRowVal;
	}
	
	/**
	 * History line for request history.
	 *
	 * @param requestId the request id
	 * @return the int
	 */
	@SuppressWarnings("unchecked")
	private int historyLineForRequestHistory(int requestId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder
				.createQuery(RequestHistory.class);
		Root root = criteriaQuery.from(RequestHistory.class);

		Predicate condition1 = criteriaBuilder.equal(
				root.get("request").get("id"), requestId);
		Predicate condition2 = criteriaBuilder.equal(
				root.get("statusDescription"), "Draft");

		Predicate conditions = criteriaBuilder.and(condition1, condition2);

		criteriaQuery.where(conditions);

		RequestHistory result = null;

		Query query = (Query) entityManager.createQuery(criteriaQuery);
		try {
			//result = (Integer) query.getSingleResult();
			 result = (RequestHistory) query.getSingleResult();
		} catch (NoResultException e) {
			Logger.error("Request Id : " + requestId + "  Draft action  in Request History has issues. Could me mutiple instances" );
		}
		
		int histLineNo =0;
		
		if (result != null){
			histLineNo = result.getHistLineNo();
		}
		Logger.info("Request Id : " + requestId + "  Draft present in RequestHistory. HistLineNo: " +histLineNo );
		return histLineNo;
	}
	
	public List<RequestHistory> getRequestHistoryById(int requestId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory()
				.createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<RequestHistory> criteria = builder
				.createQuery(RequestHistory.class);
		Root<RequestHistory> org = criteria.from(RequestHistory.class);
		criteria.select(org);
		criteria.where(builder.and(builder.equal(org.get("request").get("id"),
				requestId)));
		criteria.orderBy(builder.desc(org.get("createdOn")));
		List<RequestHistory> histortList = entityManager.createQuery(criteria)
				.getResultList();
		return histortList;
	}

	public List<Source> getDataSources() {
		List<Source> list = (List<Source>) hibernateTemplate
				.find("from Source");
		return list;
	}

	public List<RequestType> getRequestTypes() {
		List<RequestType> list = (List<RequestType>) hibernateTemplate
				.find("from RequestType");
		return list;
	}
}
