package gov.va.genisis2.controller;

import gov.va.genisis2.bo.BusinessService;
import gov.va.genisis2.exceptions.ErrorResponse;
import gov.va.genisis2.exceptions.Genisis2RestException;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.model.CommentHistory;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;

import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;


/**
 * The Class Genisis2Controller.
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/services")
public class Genisis2Controller {

	/** The Logger. */
	private final org.slf4j.Logger Logger = LoggerFactory
			.getLogger(Genisis2Controller.class);

	/** The Constant REQUEST_SENT. */
	private final static String REQUEST_SENT = "Sent";
	
	/** The Constant REQUEST_DENIED. */
	private final static String REQUEST_DENIED = "Denied";
	
	/** The Constant REQUEST_RETURNED. */
	private final static String REQUEST_RETURNED = "Returned";
	
	/** The Constant REQUEST_UPDATE. */
	private final static String REQUEST_UPDATE = "Update";
	
	/** The Constant REQUEST_SUBMIT. */
	private final static String REQUEST_SUBMIT = "Submitted";

	/** The Constant REQUEST_ACCEPTED. */
	private final static String REQUEST_ACCEPTED = "RequestAccepted";
	
	/** The Constant REQUEST_NOT_ACCEPTED. */
	private final static String REQUEST_NOT_ACCEPTED = "RequestNotAccepted";
	
	/** The Constant RESULTS_ACCEPTED. */
	private final static String RESULTS_ACCEPTED = "ResultsAccepted";
	
	/** The Constant RESULTS_NOT_ACCEPTED. */
	private final static String RESULTS_NOT_ACCEPTED = "ResultsNotAccepted";

	/** The business service. */
	private BusinessService businessService;

	/* L D A P    R E S T     C A L L S */

	/**
	 * Gets the login details by email.
	 *
	 * @param email the email
	 * @return the login details by email
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/users/{email}/email", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getLoginDetailsByEmail(@PathVariable String email)  throws Genisis2RestException {
		Logger.info(" Get ldap user by email.");
		ResponseWrapper wrapper = businessService
				.ldapGetLoginDetailsByEmail(email);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the login details by id.
	 *
	 * @param id the id
	 * @return the login details by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/users/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getLoginDetailsById(@PathVariable String id) throws Genisis2RestException {
		Logger.info(" Get ldap user by Id.");
		ResponseWrapper wrapper = businessService.ldapGetLoginDetailsById(id);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Creates the request.
	 *
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	/* R E Q U E S T       R E S T      C A L L S */
	@RequestMapping(value = "/requests", method = RequestMethod.POST, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	public ResponseEntity createRequest(@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Create Request ");
		ResponseWrapper wrapper = businessService.createRequest(request);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the all requests by study approval.
	 *
	 * @param studyApprovalId the study approval id
	 * @return the all requests by study approval
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{studyApprovalId}/studyApproval", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequestsByStudyApproval  (
			@PathVariable int studyApprovalId) throws Genisis2RestException {
		Logger.info("getAllRequestsByStudyApproval ");
		ResponseWrapper wrapper = businessService
				.getAllRequestsByStudyApproval(studyApprovalId);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the all requests by comment history.
	 *
	 * @param commentHistoryId the comment history id
	 * @return the all requests by comment history
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
/*	@RequestMapping(value = "/requests/{CommentHistoryId}/commentHistory", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequestsByCommentHistory  (
			@PathVariable int commentHistoryId) throws Genisis2RestException {
		Logger.info("getAllRequestsByCommentHistory ");
		ResponseWrapper wrapper = businessService
				.getAllRequestsByCommentHistory(commentHistoryId);
		return createReposneWrapper(wrapper);
	}
	*/
	/**
	 * Gets the requests by id.
	 *
	 * @param id the id
	 * @return the requests by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsById(@PathVariable int id) throws Genisis2RestException {
		Logger.info("Get Request  by Id ");
		ResponseWrapper wrapper = businessService.getRequestByID(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the requests by user.
	 *
	 * @param uid the uid
	 * @return the requests by user
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{uid}/createdBy", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsByUser(@PathVariable String uid) throws Genisis2RestException {
		Logger.info("Get Requests  by user ");
		ResponseWrapper wrapper = businessService.getRequestsByUID(uid);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all requests.
	 *
	 * @return the all requests
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequests() throws Genisis2RestException {
		Logger.info("Get All Requests ");
		ResponseWrapper wrapper = businessService.getAllRequests();
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by data manager.
	 *
	 * @param id the id
	 * @return the request by data manager
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/dataManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByDataManager(@PathVariable String id) throws Genisis2RestException {
		Logger.info("Request by DataManger  ");
		ResponseWrapper wrapper = businessService.getRequestsByUID(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by status.
	 *
	 * @param status the status
	 * @return the request by status
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{status}/status", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByStatus(@PathVariable String status) throws Genisis2RestException {
		Logger.info("Requests by Status /approved/status");
		ResponseWrapper wrapper = businessService.getAllRequestsByStaus(status);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Request Update");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_UPDATE);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Submit request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/submit", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity submitRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Request Submit ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_SUBMIT);
		return createReposneWrapper(wrapper);

	}

	/**
	 * Sent request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/sent", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity sentRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Request Sent/Approved ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_SENT);
		return createReposneWrapper(wrapper);

	}

	/**
	 * Deny request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/deny", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity denyRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Request Deny  ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_DENIED);
		return createReposneWrapper(wrapper);

	}

	/**
	 * Return request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/return", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity returnRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info(" Request Returned  ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_RETURNED);
		return createReposneWrapper(wrapper);

	}

	/**
	 * Fulfill request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/acceptdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity fulfillRequest(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException  {
	Logger.info("Accept Data");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_ACCEPTED);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Un fulfill request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/denydata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity unFulfillRequest(@PathVariable int id,
			@RequestBody Request request)throws Genisis2RestException {
		Logger.info("Reject Data");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_NOT_ACCEPTED);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Reject data.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/rejectdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity rejectData(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
	   Logger.info("Reject Results");
		ResponseWrapper wrapper = businessService.persist(id, request,
				RESULTS_NOT_ACCEPTED);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Accept data.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/confirmdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity acceptData(@PathVariable int id,
			@RequestBody Request request) throws Genisis2RestException {
		Logger.info("Accept Results");
		ResponseWrapper wrapper = businessService.persist(id, request,
				RESULTS_ACCEPTED);
		return createReposneWrapper(wrapper);
	}

	
	/* S T U D Y    A P P R O V A L     R E S T     C A L L S */

	/**
	 * Creates the study approval.
	 *
	 * @param studyApproval the study approval
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity createStudyApproval(
			@RequestBody StudyApproval studyApproval)throws Genisis2RestException {
		Logger.info("Create Study Approval");
		ResponseWrapper wrapper = businessService
				.createStudyApproval(studyApproval);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update study approval.
	 *
	 * @param id the id
	 * @param studyApproval the study approval
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateStudyApproval(@PathVariable int id,
			@RequestBody StudyApproval studyApproval) throws Genisis2RestException {
		if (id > 0) {
			studyApproval.setId(id);
			ResponseWrapper wrapper = businessService
					.updateStudyApproval(studyApproval);
			Logger.info("Updated Study Approval");
			return createReposneWrapper(wrapper);
		} else {
			ResponseWrapper wrapper = new ResponseWrapper();
			Logger.info("No ID Found");
			wrapper.setSuccess(false);
			wrapper.setMessage("No ID Found");
			return new ResponseEntity<ResponseWrapper>(wrapper,
					HttpStatus.NOT_MODIFIED);
		}
	}

	/**
	 * Gets the study approval by id.
	 *
	 * @param id the id
	 * @return the study approval by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalById(@PathVariable int id) throws Genisis2RestException {
		Logger.info("Study Approval by Id");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByID(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the study approval by user.
	 *
	 * @param uid the uid
	 * @return the study approval by user
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{uid}/{user}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalByUser(@PathVariable String uid) throws Genisis2RestException {
		Logger.info("Study Approval by User");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByUID(uid);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all study approvals.
	 *
	 * @return the all study approvals
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllStudyApprovals() throws Genisis2RestException {
		Logger.info("All Study Approvals");
		ResponseWrapper wrapper = businessService.getStudyApprovals();
		return createReposneWrapper(wrapper);
	}
	
	/* C O M M E N T    H I S T O R Y     R E S T    C A L L S */
	
	/**
	 * Creates the comment history.
	 *
	 * @param commentHistory the comment history
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/comment", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity createCommentHistory(
			@RequestBody CommentHistory commentHistory)throws Genisis2RestException {
		Logger.info("Create Comment History");
		ResponseWrapper wrapper = businessService
				.createCommentHistory(commentHistory);
		return createReposneWrapper(wrapper);
	}

	
	/**
	 * Update comment history.
	 *
	 * @param id the id
	 * @param commentHistory the comment history
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
/*	@RequestMapping(value = "/comment/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateCommentHistory(@PathVariable int id,
			@RequestBody CommentHistory commentHistory) throws Genisis2RestException {
		if (id > 0) {
			commentHistory.setCommentId(id);
			ResponseWrapper wrapper = businessService
					.updateCommentHistory(commentHistory);
			Logger.info("Updated Comment History");
			return createReposneWrapper(wrapper);
		} else {
			ResponseWrapper wrapper = new ResponseWrapper();
			Logger.info("No ID Found");
			wrapper.setSuccess(false);
			wrapper.setMessage("No ID Found");
			return new ResponseEntity<ResponseWrapper>(wrapper,
					HttpStatus.NOT_MODIFIED);
		}
	}*/

	/**
	 * Gets the comment history by id.
	 *
	 * @param id the id
	 * @return the comment history by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	/*@RequestMapping(value = "/comment/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getCommentHistoryById(@PathVariable int id) throws Genisis2RestException {
		Logger.info("Comment History by Id");
		ResponseWrapper wrapper = businessService.getCommentHistoriesByID(id);
		return createReposneWrapper(wrapper);
	}*/
	

	/**
	 * Gets the comment histories by user.
	 *
	 * @param uid the uid
	 * @return the comment histories by user
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
/*	@RequestMapping(value = "/comment/{uid}/{user}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getCommentHistoriesByUser(@PathVariable String uid) throws Genisis2RestException {
		Logger.info("Comments by User");
		ResponseWrapper wrapper = businessService.getCommentHistoriesByUID(uid);
		return createReposneWrapper(wrapper);
	}*/

	/**
	 * Gets the all comment histories.
	 *
	 * @return the all comment histories
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
/*	@RequestMapping(value = "/comment", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllCommentHistories() throws Genisis2RestException {
		Logger.info("All Comment Histories");
		System.out.println("All Comment Histories");
		ResponseWrapper wrapper = businessService.getCommentHistories();
		return createReposneWrapper(wrapper);
	}
*/

	/* ACTIVITI R E S T     C A L L S */
	/**
	 * Gets the request tracking by id.
	 *
	 * @param id the id
	 * @return the request tracking by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	// Request History 
	@RequestMapping(value = "/requests/{id}/log", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestTrackingById(@PathVariable int id) throws Genisis2RestException {
		Logger.info("Get Request Tracking by Id ");
		ResponseWrapper wrapper = businessService.getRequestTrackingByID(id);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the request data sources.
	 *
	 * @return the request data sources
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/datasources", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestDataSources()throws Genisis2RestException {
		Logger.info("All Request Data Sources");
		ResponseWrapper wrapper = businessService.getRequestDataSources();
		return createReposneWrapper(wrapper);
	}
		
	/**
	 * Gets the request types.
	 *
	 * @return the request types
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requestTypes", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestTypes() throws Genisis2RestException {
		Logger.info("All Request Types");
		ResponseWrapper wrapper = businessService.getRequestTypes();
		return createReposneWrapper(wrapper);
		
	}
	
	/**
	 * Gets the business service.
	 *
	 * @return the business service
	 */
	public BusinessService getBusinessService() {
		return businessService;
	}

	/**
	 * Sets the business service.
	 *
	 * @param businessService the new business service
	 */
	@Autowired
	public void setBusinessService(BusinessService businessService) {
		this.businessService = businessService;
	}
		
	/**
	 * Creates the reposne wrapper.
	 *
	 * @param wrapper the wrapper
	 * @return the response entity
	 */
	private ResponseEntity<ResponseWrapper> createReposneWrapper(ResponseWrapper wrapper)
	{
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Pragma","No-cache");     
		responseHeaders.add("Cache-Control","no-cache");   
		return new ResponseEntity<ResponseWrapper>(wrapper,responseHeaders, HttpStatus.OK);
		
	}
	
	/**
	 * Exception handler.
	 *
	 * @param ex the ex
	 * @return the response entity
	 */
	@ExceptionHandler(Genisis2RestException.class)
	public ResponseEntity<ErrorResponse> exceptionHandler(Exception ex) {
		ErrorResponse error = new ErrorResponse();
		error.setErrorCode(HttpStatus.PRECONDITION_FAILED.value());
		error.setMessage(ex.getMessage());
		return new ResponseEntity<ErrorResponse>(error, HttpStatus.OK);
	}
}
