package gov.va.genisis2.bo;

import gov.va.genisis2.model.LDAPLogger;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.model.CommentHistory;
import gov.va.genisis2.model.WorkflowStatus;
import gov.va.genisis2.service.impl.LdapService;
import gov.va.genisis2.service.impl.LookUpService;
import gov.va.genisis2.service.impl.RequestService;
import gov.va.genisis2.service.impl.StudyApprovalService;
import gov.va.genisis2.service.impl.CommentHistoryService;
import gov.va.genisis2.util.ldap.Transformer;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * The Class BusinessService.
 */
public class BusinessService implements IBusiness {
	
	/** The Logger. */
	private final org.slf4j.Logger Logger = LoggerFactory
			.getLogger(BusinessService.class);
	
	/** The newline. */
	public static String newline = "<br />";
	
	/** The Constant NO_RECORD_FOUND. */
	private final static String NO_RECORD_FOUND = "No records found.";
	
	/** The Constant REQUEST_SENT. */
	private final static String REQUEST_SENT = "Sent";
	
	/** The Constant REQUEST_DENIED. */
	private final static String REQUEST_DENIED = "Denied";
	
	/** The Constant REQUEST_RETURNED. */
	private final static String REQUEST_RETURNED = "Returned";
	
	/** The Constant REQUEST_UPDATE. */
	private final static String REQUEST_UPDATE = "Update";
	
	/** The Constant REQUEST_SUBMIT. */
	private final static String REQUEST_SUBMIT = "Submitted";
	
	/** The Constant REQUEST_ACCEPTED. */
	private final static String REQUEST_ACCEPTED = "RequestAccepted";
	
	/** The Constant REQUEST_NOT_ACCEPTED. */
	private final static String REQUEST_NOT_ACCEPTED = "RequestNotAccepted";
	
	/** The Constant RESULTS_ACCEPTED. */
	private final static String RESULTS_ACCEPTED = "ResultsAccepted";
	
	/** The Constant RESULTS_NOT_ACCEPTED. */
	private final static String RESULTS_NOT_ACCEPTED="ResultsNotAccepted";
	
	
	/** The request service. */
	private RequestService requestService;
	
	/** The study approval service. */
	private StudyApprovalService studyApprovalService;
	
	/** The comment history service. */
	private CommentHistoryService commentHistoryService;
	
	/** The ldap service. */
	private LdapService ldapService;
	
	/** The transformer. */
	private Transformer transformer;
	
	/** The activiti service. */
	private ActivitiService activitiService;
	
	
	private LookUpService  lookUpService;
	
	
	public LookUpService getLookUpService() {
		return lookUpService;
	}

	@Autowired
	public void setLookUpService(LookUpService lookUpService) {
		this.lookUpService = lookUpService;
	}

	/**
	 * Gets the activiti service.
	 *
	 * @return the activiti service
	 */
	public ActivitiService getActivitiService() {
		return activitiService;
	}

    /**
     * Sets the activiti service.
     *
     * @param activitiService the new activiti service
     */
    @Autowired
	public void setActivitiService(ActivitiService activitiService) {
		this.activitiService = activitiService;
	}


	/**
	 * Gets the transformer.
	 *
	 * @return the transformer
	 */
	public Transformer getTransformer() {
		return transformer;
	}


	/**
	 * Sets the transformer.
	 *
	 * @param transformer the new transformer
	 */
	@Autowired
	public void setTransformer(Transformer transformer) {
		this.transformer = transformer;
	}

	/**
	 * Gets the request service.
	 *
	 * @return the request service
	 */
	public RequestService getRequestService() {
		return requestService;
	}

	/**
	 * Sets the request service.
	 *
	 * @param requestService the new request service
	 */
	@Autowired
	public void setRequestService(RequestService requestService) {
		this.requestService = requestService;
	}

	/**
	 * Gets the study approval service.
	 *
	 * @return the study approval service
	 */
	public StudyApprovalService getStudyApprovalService() {
		return studyApprovalService;
	}
	
	/**
	 * Sets the study approval service.
	 *
	 * @param studyApprovalService the new study approval service
	 */
	@Autowired
	public void setStudyApprovalService(
			StudyApprovalService studyApprovalService) {
		this.studyApprovalService = studyApprovalService;
	}
	
	/**
	 * Gets the comment history service.
	 *
	 * @return the comment history service
	 */
	public CommentHistoryService getCommentHistoryService() {
		return commentHistoryService;
	}
	
	/**
	 * Sets the comment history service.
	 *
	 * @param commentHistoryService the new comment history service
	 */
	@Autowired
	public void setCommentHistoryService(
			CommentHistoryService commentHistoryService) {
		this.commentHistoryService = commentHistoryService;
	}
	
	/**
	 * Gets the ldap service.
	 *
	 * @return the ldap service
	 */
	public LdapService getLdapService() {
		return ldapService;
	}

	/**
	 * Sets the ldap service.
	 *
	 * @param ldapService the new ldap service
	 */
	@Autowired
	public void setLdapService(LdapService ldapService) {
		this.ldapService = ldapService;
	}
	
	
	public ResponseWrapper<?> createStudyApproval(StudyApproval studyApproval) {
		Logger.info("BusinessService:  createStudyApproval");
		ResponseWrapper wrapper = new ResponseWrapper();
		studyApproval=BusinessHelper.handleStudyApprovalPayload(studyApproval);
		int id = getStudyApprovalService().createStudyApproval(studyApproval);
		if (id > 0) {
			wrapper = this.getStudyApprovalsByID(id);
			wrapper.setSuccess(true);
			wrapper.setMessage(null); }
		else {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays
							.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Study Approval");
		}
		return wrapper;
	}
	
		
	/**
	 * Update study approval.
	 *
	 * @param studyApproval the study approval
	 * @return the response wrapper
	 */
	public ResponseWrapper<?> updateStudyApproval(StudyApproval studyApproval) {
		Logger.info("BusinessService: updateStudyApproval");
		ResponseWrapper wrapper = new ResponseWrapper();
		studyApproval = BusinessHelper.handleStudyApprovalPayload(studyApproval);
		int id = studyApproval.getId();
		if (id > 0) {
			getStudyApprovalService().updateStudyApproval(studyApproval);
			wrapper = this.getStudyApprovalsByID(id);
		} else {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to update Study Approval");
		}
		return wrapper;
	}
	
		
	public ResponseWrapper<?> getStudyApprovalsByID(int id) {
		Logger.info("BusinessService:  getStudyApprovalsByID:"+id);
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval = getStudyApprovalService()
				.getStudyApprovalsByID(id);
		if (studyApproval != null) {
			List<StudyApproval> studyApprovalList = BusinessHelper
					.prepareStudyApprovalResponsePayload(Arrays
							.asList(studyApproval));
			wrapper.setResponse(studyApprovalList.get(0));
		}
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	public ResponseWrapper getStudyApprovalsByUID(String uid) {
		Logger.info("BusinessService:  getStudyApprovals by user:"+uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApproval> studyApprovalList = getStudyApprovalService()
				.getStudyApprovalsByUID(uid);
		studyApprovalList = BusinessHelper
				.prepareStudyApprovalResponsePayload(studyApprovalList);
		wrapper.setResponse(studyApprovalList);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	
	public ResponseWrapper<?> getStudyApprovals() {
		Logger.info("BusinessService:  getStudyApprovals");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApproval> studyApprovalList = getStudyApprovalService()
				.getStudyApprovals();
		studyApprovalList = BusinessHelper
				.prepareStudyApprovalResponsePayload(studyApprovalList);
		wrapper.setResponse(studyApprovalList);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	/**
	 * Gets the comment histories.
	 *
	 * @return the comment histories
	 */
	public ResponseWrapper<?> getCommentHistories() {
		Logger.info("BusinessService:  getCommentHistories");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<CommentHistory> commentHistoryList = getCommentHistoryService()
				.getCommentHistories();
		commentHistoryList = BusinessHelper
				.prepareCommentHistoryResponsePayload(commentHistoryList);
		wrapper.setResponse(commentHistoryList);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	/**
	 * Gets the request data sources.
	 *
	 * @return the request data sources
	 */
	public ResponseWrapper<?> getRequestDataSources() {
		Logger.info("BusinessService:  getDataSources");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Source> listDatasources = getRequestService().getDataSources();
		wrapper.setResponse(listDatasources);
		createResponseWrapper(wrapper);
		return wrapper;
	}

	/**
	 * Gets the request types.
	 *
	 * @return the request types
	 */
	public ResponseWrapper<?> getRequestTypes() {
		Logger.info("BusinessService:  getRequestTypes");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestType> listRequestTypes = getRequestService()
				.getRequestTypes();
		wrapper.setResponse(listRequestTypes);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	/**
	 * Creates the response wrapper.
	 *
	 * @param wrapper the wrapper
	 * @return the response wrapper
	 */
	private ResponseWrapper createResponseWrapper(ResponseWrapper wrapper) {
		Logger.info("BusinessService:  creating Response Wrapper");
		if (wrapper == null) {
			wrapper.setMessage("Internal System Error. Could not perform requested Opeartion");
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
			return wrapper;
		}

		if (wrapper.getResponse() == null) {
			wrapper.setSuccess(true);
			wrapper.setResponse(null);
			wrapper.setMessage(NO_RECORD_FOUND);
			return wrapper;

		} else {
			Object object = wrapper.getResponse();
			try {
				if (object instanceof ArrayList<?> || object instanceof StudyApproval
						|| object instanceof CommentHistory || object instanceof Request) {
					if (object instanceof ArrayList<?> && ((ArrayList<?>) object).get(0) instanceof StudyApproval) {
						if (((ArrayList) object).size() == 1) {
							StudyApproval studyApproval = (StudyApproval) ((ArrayList<?>) object).get(0);
							wrapper.setResponse(studyApproval);
							wrapper.setSuccess(true);
							wrapper.setMessage(null);
							return wrapper;
						} else {
							wrapper.setSuccess(true);
							wrapper.setMessage(null);
						}

					} else if (object instanceof ArrayList<?>
							&& ((ArrayList<?>) object).get(0) instanceof CommentHistory) {
						if (((ArrayList) object).size() == 1) {
							CommentHistory commentHistory = (CommentHistory) ((ArrayList<?>) object).get(0);
							wrapper.setResponse(commentHistory);
							wrapper.setSuccess(true);
							wrapper.setMessage(null);
							return wrapper;
						} else {
							wrapper.setSuccess(true);
							wrapper.setMessage(null);
						}

					} else {
						wrapper.setResponse(object);
						wrapper.setSuccess(true);
						wrapper.setMessage(null);
						return wrapper;
					}

				} else {
					wrapper.setSuccess(true);
					wrapper.setResponse(null);
					wrapper.setMessage(NO_RECORD_FOUND);
				}
			} catch (NullPointerException ex) {
				ex.printStackTrace();
			}
		}
		return wrapper;
	}

	
	public ResponseWrapper<?> ldapGetLoginDetailsByEmail(String email) {
		Logger.info("BusinessService:  ldap user by email:" + email);
		ResponseWrapper wrapper = new ResponseWrapper();
		String user = ldapService.getUserDetails(email);
		wrapper = this.populateUserDetails(user);
		return wrapper;
	}

	
	public ResponseWrapper<?> ldapGetLoginDetailsById(String id) {
		Logger.info("BusinessService:  ldap user by Id:" + id);
		ResponseWrapper wrapper = new ResponseWrapper();
		String user = ldapService.getUserDetails("cn", id);
		wrapper = this.populateUserDetails(user);
		return wrapper;
	}

	/**
	 * Populate user details.
	 *
	 * @param user the user
	 * @return the response wrapper
	 */
	private ResponseWrapper<?> populateUserDetails(String user) {
		ResponseWrapper wrapper = new ResponseWrapper();
		try {
			Map<String, String> managerObject = new HashMap<String, String>();
			Map<String, String> userObject = new HashMap<String, String>();
			LDAPLogger ldapLogger = new LDAPLogger();

			if (!StringUtils.isBlank(user) && StringUtils.contains(user, ":")) {
				userObject = transformer.getLdapKeyValuePairs(user, userObject);
				String role = transformer.getRoleApprover(userObject.get("ou"),
						"ou");
				String manager = transformer.getRoleApprover(
						userObject.get("o"), "cn");

				String managerDetails = ldapService.getUserDetails("cn",
						manager);
				if (!StringUtils.isBlank(managerDetails)) {
					managerObject = transformer.getLdapKeyValuePairs(
							managerDetails, managerObject);
					managerObject.put("role", "approver");
				}
				if (!StringUtils.isBlank(role)) {
					userObject.put("role", role);
				}

				ldapLogger = transformer.getUser(userObject, managerObject);
				wrapper.setResponse(ldapLogger);
				wrapper.setMessage(null);
				wrapper.setSuccess(true);
			}
			
			else {
				wrapper.setResponse(null);
				wrapper.setSuccess(false);
				wrapper.setMessage("No records found.");

			}

		} catch (Throwable e) {
			wrapper.setSuccess(false);
			wrapper.setMessage("Internal System Error. Could not retrieve user login details.");
			wrapper.setResponse(null);
		}
		return wrapper;

	}

	public ResponseWrapper<?> createRequest(Request request) {
		Logger.info("BusinessService: createRequest" );
		ResponseWrapper wrapper = new ResponseWrapper();
		request=BusinessHelper.handleRequestPayload(request);
		
		String approverId=StringUtils.EMPTY;
		
		approverId = this.getRequestorApproverId(request);
		
		if (StringUtils.isBlank(request.getApproverId())) {
			request.setApproverId(approverId);
		}
		
		// Implement workflow
		boolean workflowProcessIdRequired = true;
		if (request.getId() != 0) {
			Request existingRequest = getRequestService().getRequestsById(
					request.getId());
			if (existingRequest.getId() != 0) {
				workflowProcessIdRequired = false;
			}
		}
		if (workflowProcessIdRequired) {
			String processUrl=StringUtils.EMPTY;
			
			String processInstanceId=StringUtils.EMPTY; 
			
			try {
				processInstanceId = activitiService.startProcess(request.getCreatedBy());
			} catch (Exception e) {
				Logger.error("Unable to start process for " + request.getCreatedBy()
						+ " Reason" + e.getMessage());
			}
			
			
		    if (StringUtils.isBlank(processInstanceId))
		    {
		    	request.setProcessId(0);
		    } else {
			int intProcessInstanceId = Integer.parseInt(processInstanceId);
			request.setProcessId(intProcessInstanceId);}
			//}
		}
	
		
		if (request.getProcessId() == 0) {
			List<Request> requestList = BusinessHelper
					.prepareRequestResponsePayload(Arrays.asList(request));

			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request. Business process failed to start.");
			return wrapper;
		}
		
		int id = getRequestService().createRequest(request);
		if (id > 0) {
			this.preserveHistory(request);
			wrapper = this.getRequestByID(id);
			wrapper.setSuccess(true);
			wrapper.setMessage(null); }
		else {
			List<Request> requestList = BusinessHelper.prepareRequestResponsePayload(Arrays
							.asList(request));
				
			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}

	public ResponseWrapper<?> getRequestByID(int id) {
		Logger.info("BusinessService: get Request by Id:"  +id);
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = getRequestService().getRequestsById(id);
		if (request != null) {
			List<Request> requestList = BusinessHelper
					.prepareRequestResponsePayload(Arrays.asList(request));

			wrapper.setResponse(requestList.get(0));
		}
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	/**
	 * Gets the request tracking by ID.
	 *
	 * @param id the id
	 * @return the request tracking by ID
	 */
	public ResponseWrapper<?> getRequestTrackingByID(int id) {
		Logger.info("BusinessService: get Audit Request by Id:"  +id);
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = getRequestService().getRequestsById(id);
		if (request != null) {
			List<RequestHistory> trackingList = getRequestService().getRequestHistoryById(id);
			if (trackingList != null && trackingList.size()==0)
			wrapper.setResponse(null);
			else {
			// transform trackingList
				wrapper.setResponse(this.getAuditTrailWithComments(trackingList));
			}
		}
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	public ResponseWrapper<?> getRequestsByUID(String uid) {
		Logger.info("BusinessService: get Request by User:"  +uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList =getRequestService().getRequestsByUID(uid);
		requestList= this.filterRequestList(requestList, uid);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		createResponseWrapper(wrapper);
		return wrapper;
	}

	/**
	 * Filter request list.
	 *
	 * @param requestList the request list
	 * @param uid the uid
	 * @return the list
	 */
	private List<Request> filterRequestList(List<Request> requestList,
			String uid) {
		List<Request> filteredRequestList = new ArrayList<Request>();

		for (Request request : requestList) {
			if ( !(!StringUtils.equalsIgnoreCase(request.getCreatedBy(), uid)
					&& StringUtils.equalsIgnoreCase(
							request.getStatusDescription(), "Draft"))) {
				filteredRequestList.add(request);
			}
		}
		return filteredRequestList;
	}
	
	public ResponseWrapper<?> getAllRequests() {
		Logger.info("BusinessService: get all Requests");
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList =getRequestService().getAllRequests();
		Logger.info("Got Requests count:" + requestList.size());
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		createResponseWrapper(wrapper);
		return wrapper;
	}

	public ResponseWrapper<?> getAllRequestsByStudyApproval(int studyApprovalId) {
		Logger.info("BusinessService: getAllRequestsByStudyApproval:"+studyApprovalId);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList =getRequestService()
				.getAllRequestsByStudyApproval(studyApprovalId);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		createResponseWrapper(wrapper);
		return wrapper;
	}

	
	public ResponseWrapper<?> getAllRequestsByStaus(String status) {
		Logger.info("BusinessService: get all Requests by Status:"+status);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList =getRequestService().getAllRequestsByStaus(status);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		createResponseWrapper(wrapper);
		return wrapper;
	}

	public ResponseWrapper<?> submitOrModify(Request request) {
		Logger.info("BusinessService: Sumbit/Modify Request");
		ResponseWrapper wrapper = new ResponseWrapper();
		wrapper= this.createRequest(request);
		return wrapper;
	}
	
	public  ResponseWrapper persist(int id, Request changeRequest,String operation) {
		Logger.info("BusinessService: Persisting: "	+ operation);
		ResponseWrapper wrapper = new ResponseWrapper();
		
		if (StringUtils.equalsIgnoreCase(operation, "Update")
				|| StringUtils.equalsIgnoreCase(operation, "Submitted")) {
			changeRequest = BusinessHelper.handleRequestPayload(changeRequest);
		}
		Request databaseRequest = requestService.getRequestsById(id);

		if (databaseRequest == null)
		{
			changeRequest.setId(id);
			wrapper.setResponse(changeRequest);
			wrapper.setMessage("No such request found in database");
			wrapper.setSuccess(false);
			return wrapper;
		}

		String status = StringUtils
				.trim(databaseRequest.getStatusDescription());
		
		if ( StringUtils.equalsIgnoreCase(status, "Rejected")) {
			wrapper.setResponse(changeRequest);
			wrapper.setMessage("You can't submit this request.");
			wrapper.setSuccess(false);
			return wrapper;
		}
		
				
		if (!StringUtils.equals(operation, "Update"))
		changeRequest.setStatusDescription(operation);

		if ( changeRequest.getProcessId() == 0)
			changeRequest.setProcessId(databaseRequest.getProcessId());

		if (changeRequest.getRequestType() == 0)
			changeRequest.setRequestType(databaseRequest.getRequestType());

		if (changeRequest.getAssociatedRequest1() == null || changeRequest.getAssociatedRequest1() == 0)
			changeRequest.setAssociatedRequest1(databaseRequest
					.getAssociatedRequest1());

		if (changeRequest.getAssociatedRequest2() == null || changeRequest.getAssociatedRequest2() == 0 )
			changeRequest.setAssociatedRequest2(databaseRequest
					.getAssociatedRequest2());

		if (changeRequest.getAssociatedRequest3() == null || changeRequest.getAssociatedRequest3() == 0)
			changeRequest.setAssociatedRequest3(databaseRequest
					.getAssociatedRequest3());

		if (changeRequest.getStudyApproval() == null || changeRequest.getStudyApproval() == 0)
			changeRequest.setStudyApproval(databaseRequest.getStudyApproval());

		if (changeRequest.getSource() == 0)
			changeRequest.setSource(databaseRequest.getSource());

		if (StringUtils.isBlank(changeRequest.getStatusDescription()))
			changeRequest.setStatusDescription(databaseRequest
					.getStatusDescription());

		if (StringUtils.isBlank(changeRequest.getTitle()))
			changeRequest.setTitle(databaseRequest.getTitle());

		if (StringUtils.isBlank(changeRequest.getDescription()))
			changeRequest.setDescription(databaseRequest.getDescription());
		

		changeRequest.setModifiedOn(new Date());
		changeRequest.setModifiedBy(changeRequest.getCreatedBy());
		
		changeRequest.setCreatedBy(databaseRequest.getCreatedBy());
		changeRequest.setCreatedOn(databaseRequest.getCreatedOn());
		changeRequest.setApproverId(databaseRequest.getApproverId());
		
		changeRequest.setId(id);
		
		String taskId = StringUtils.EMPTY;
		String submitter= StringUtils.EMPTY;
		submitter=changeRequest.getModifiedBy();
		if (StringUtils.equalsIgnoreCase(operation, "Submitted")){
			submitter=changeRequest.getCreatedBy();
		}
		
		try {
			taskId = activitiService.getprocessTaskId(String.valueOf(databaseRequest.getProcessId()), submitter);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		Logger.info("Action: " + operation + " Task id: " + taskId + " Process Id: " + databaseRequest.getProcessId() + "  Submitter:" +submitter);
		changeRequest.setTaskId(taskId);
		
		
		
		
		int submitOrModifyId = requestService.submitOrModify(changeRequest);
      
		if (submitOrModifyId > 0){
			wrapper = this.getRequestByID(submitOrModifyId);
			wrapper.setSuccess(true);
			wrapper.setMessage(null); 
			this.completeWorkflowProcess(submitOrModifyId);
			preserveHistory(requestService.getRequestsById(submitOrModifyId));
				if (!StringUtils.isBlank(changeRequest.getComments())){
					// Save Comments for declines in CommentHistory
					CommentHistory commentHistory = new CommentHistory();
					commentHistory.setComments(changeRequest.getComments());
					Request request = getRequestService().getRequestsById(submitOrModifyId);
					int statusId  = getLookUpService().getStatusId(changeRequest.getStatusDescription());
					
					WorkflowStatus workflowStatus = new WorkflowStatus();
					workflowStatus.setId(statusId);
					commentHistory.setWorkflowStatus(workflowStatus);
					commentHistory.setRequest(request);
					commentHistory.setCreatedBy(changeRequest.getModifiedBy());
					commentHistory.setType("action");
					commentHistory.setTaskId(taskId);
					getCommentHistoryService().submitOrModify(commentHistory);
				}
		    }
		else {
			List<Request> requestList = BusinessHelper
					.prepareRequestResponsePayload(Arrays.asList(changeRequest));

			wrapper.setResponse(requestList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}	
		
	/**
	 * Gets the requestor approver id.
	 *
	 * @param request the request
	 * @return the requestor approver id
	 */
	private String getRequestorApproverId(Request request) {
		String approverId = StringUtils.EMPTY;

		String requestor = request.getCreatedBy();
		String user = ldapService.getUserDetails("cn", requestor);
		Object object = this.populateUserDetails(user);
		JSONObject jsonUser = new JSONObject(object);
		JSONObject response = jsonUser.getJSONObject("response");
		approverId = (String) response.getJSONObject("datamanager").get("id");

		return approverId;

	}
	
	/**
	 * Complete workflow process.
	 *
	 * @param requestId the request id
	 * @return true, if successful
	 */
	private boolean completeWorkflowProcess(int requestId) {
		boolean success = true;
		String from = "";
		String to = "";
		String subject = "";
		String body = "";
		String requestorId = "";
		String approverId = "";
		String processUrl = StringUtils.EMPTY;

		Request request = getRequestService().getRequestsById(requestId);
		RequestType requestType = getRequestService().getRequestType(
				request.getRequestType());
		Source source = getRequestService().getSource(request.getSource());
		String user = ldapService.getUserDetails("cn", request.getCreatedBy());
		long processInstanceId = request.getProcessId();
		Object object = this.populateUserDetails(user);
		JSONObject jsonUser = new JSONObject(object);
		JSONObject response = jsonUser.getJSONObject("response");
		from = (String) response.get("email");
		to = (String) response.getJSONObject("datamanager").get("email");
		requestorId = (String) response.get("id");
		approverId = (String) response.getJSONObject("datamanager").get("id");
		String status = request.getStatusDescription();

		JSONObject json = new JSONObject();
		json.put("requestorId", requestorId);
		json.put("approverDecision", status);
		json.put("fulfillmentDecision", "");
		json.put("resultAcceptanceDecision", "");
		json.put("approverId", approverId);
		json.put("from_user", from);
		json.put("to_user", to);
		json.put("cc_users", from);// This would hold group of users. Currently,
									// from user
		String pid = String.valueOf(request.getProcessId());

		if (StringUtils.equals(status, REQUEST_SUBMIT)) {
			json.put("approverDecision", "");

			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getCreatedBy(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Submit " +  " Reason" + e.getMessage());
			}
		} else if (StringUtils.equals(status, REQUEST_RETURNED)) {

			json.put("approverDecision", "Return");
			json.put("from_user", to);
			json.put("to_user", from);
			json.put("cc_users", from);
			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Returned " +  " Reason" + e.getMessage());
			}
		} else if (StringUtils.equals(status, REQUEST_DENIED)) {
			json.put("approverDecision", "Reject");
			json.put("from_user", to);
			json.put("to_user", from);
			json.put("cc_users", from);
			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Denied " +  " Reason" + e.getMessage());
			}

		} else if (StringUtils.equals(status, REQUEST_SENT)) {
			json.put("approverDecision", "Sent");
			json.put("from_user", to);
			json.put("cc_users", to);
			json.put("to_user", "vinci.datasource@gmail.com");
			json.put("requestType", requestType);
			json.put("source", source);

			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Sent to VINCI " +  " Reason" + e.getMessage());
			}

		}

		else if (StringUtils.equals(status, REQUEST_NOT_ACCEPTED)) {
			json.put("fulfillmentDecision", "Unfulfilled");
			json.put("from_user", "vinci.datasource@gmail.com");
			json.put("to_user", to);
			json.put("cc_users", "vinci.datasource@gmail.com");
			
			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Not Accepted by VINCI " +  " Reason" + e.getMessage());
			}

		}

		else if (StringUtils.equals(status, REQUEST_ACCEPTED)) {
			json.put("fulfillmentDecision", "Fulfilled");
			json.put("from_user", "vinci.datasource@gmail.com");
			json.put("to_user", to);
			json.put("cc_users", "vinci.datasource@gmail.com");
			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Request Accepted by VINCI " +  " Reason" + e.getMessage());
			}

		}

		else if (StringUtils.equals(status, RESULTS_NOT_ACCEPTED)) {
			json.put("resultAcceptanceDecision", "Reject");
			json.put("from_user", to);
			json.put("to_user", from);
			json.put("cc_users", to);
			try {
				activitiService.claimAndCompleteHumanTask(pid,
						request.getApproverId(), request, json);
			} catch (Exception e) {
				Logger.error("Unable to Claim and Complete Task for Results Not Accepted. " +  " Reason" + e.getMessage());
			}

		}

		else if (StringUtils.equals(status, RESULTS_ACCEPTED)) {
			json.put("resultAcceptanceDecision", "Accept");
			json.put("from_user", to);
			json.put("to_user", from);
			json.put("cc_users", to);

		}
		try {
			activitiService.claimAndCompleteHumanTask(pid,
					request.getApproverId(), request, json);
		} catch (Exception e) {
			Logger.error("Unable to Claim and Complete Task for Results Accepted. " +  " Reason" + e.getMessage());
		}

		return success;
	}
	
	/**
	 * Preserve history.
	 *
	 * @param request the request
	 */
	private void preserveHistory(Request request) {

		RequestHistory requestHistory = new RequestHistory(0, 
				request,
				request.getProcessId(), 
				request.getTitle(),request.getDescription(), 
				request.getApprovalStatus(),
				request.getDetailesInternalStatusCode(),
				request.getStatusDescription(), 
				request.getRequestType(),
				request.getSource(), 
				request.getNotifyCollaboraters(),
				request.getAssociatedRequest1(),request.getAssociatedRequest2(),
				request.getAssociatedRequest3(),
				request.getStudyApproval(),
				request.getCreatedOn(), request.getModifiedOn(),
				request.getCreatedBy(), request.getModifiedBy(),
				request.getApproverId(),
				request.getTaskId()
				);

		int returnId = getRequestService().submitOrModify(requestHistory);
		if (returnId > 0) {

			Logger.info(requestHistory.getRequest().getId() + " with status "
					+ requestHistory.getStatusDescription()
					+ " added to the history successfully!!");
		} else {
			Logger.warn(requestHistory.getRequest().getId() + " with status "
					+ requestHistory.getStatusDescription()
					+ " could not be added to the history.");
		}

	}
	
	// Comment History
	public ResponseWrapper<?> createCommentHistory(CommentHistory commentHistory) {
		Logger.info("BusinessService:  createCommentHistory");
		Logger.info(commentHistory.toString());
		ResponseWrapper wrapper = new ResponseWrapper();
		int statusId  = getLookUpService().getStatusId(commentHistory.getStatus());
		WorkflowStatus workflowStatus = new WorkflowStatus();
		workflowStatus.setId(statusId);
		
		Request request = getRequestService().getRequestsById(commentHistory.getRequestId());
		commentHistory.setWorkflowStatus(workflowStatus);
		commentHistory.setRequest(request);
		
		commentHistory.setTaskId(request.getTaskId());
		
		int id = getCommentHistoryService().submitOrModify(commentHistory);
		
		if (id > 0) {
			//wrapper = this.getCommentHistoriesByID(id);
			CommentHistory returnCommentHistoryResp = getCommentHistoryService().getCommentHistoryByCommentId(id);
			returnCommentHistoryResp.setRequestId(commentHistory.getRequestId());
			returnCommentHistoryResp.setStatus(commentHistory.getStatus());
			wrapper.setResponse(returnCommentHistoryResp);
			
			wrapper.setSuccess(true);
			wrapper.setMessage(null); }
		else {
			List<CommentHistory> commentHistoryList = BusinessHelper
					.prepareCommentHistoryResponsePayload(Arrays
							.asList(commentHistory));
			wrapper.setResponse(commentHistoryList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Comment History");
		}
		return wrapper;
	}

	

	

	/*public ResponseWrapper<?> updateCommentHistory(CommentHistory commentHistory) {
		Logger.info("BusinessService: updateCommentHistory");
		ResponseWrapper wrapper = new ResponseWrapper();
		commentHistory = BusinessHelper.handleCommentHistoryPayload(commentHistory);
		int id = commentHistory.getCommentId();
		if (id > 0) {
			getCommentHistoryService().updateCommentHistory(commentHistory);
			wrapper = this.getCommentHistoriesByID(id);
		} else {
			List<CommentHistory> commentHistoryList = BusinessHelper.prepareCommentHistoryResponsePayload(Arrays.asList(commentHistory));
			wrapper.setResponse(commentHistoryList.get(0));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to save comments");
		}
		return wrapper;
	}
	
	public ResponseWrapper getCommentHistoriesByID(int id) {
		Logger.info("BusinessService:  getCommentHistoriesByID:"+id);
		ResponseWrapper wrapper = new ResponseWrapper();
		CommentHistory commentHistory = getCommentHistoryService()
				.getCommentHistoriesByID(id);
		if (commentHistory != null) {
			List<CommentHistory> commentHistoryList = BusinessHelper
					.prepareCommentHistoryResponsePayload(Arrays
							.asList(commentHistory));
			wrapper.setResponse(commentHistoryList.get(0));
		}
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	public ResponseWrapper getCommentHistoriesByUID(String uid) {
		Logger.info("BusinessService:  getCommentHistories by user:"+uid);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<CommentHistory> commentHistoryList = getCommentHistoryService()
				.getCommentHistoriesByUID(uid);
		commentHistoryList = BusinessHelper
				.prepareCommentHistoryResponsePayload(commentHistoryList);
		wrapper.setResponse(commentHistoryList);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	
	public ResponseWrapper<?> getAllRequestsByCommentHistory(int commentHistoryId) {
		Logger.info("BusinessService: getAllRequestsByCommentHistory:"+commentHistoryId);
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requestList =getRequestService()
				.getAllRequestsByCommentHistory(commentHistoryId);
		requestList = BusinessHelper.prepareRequestResponsePayload(requestList);
		wrapper.setResponse(requestList);
		createResponseWrapper(wrapper);
		return wrapper;
	}
	*/
	private List<gov.va.genisis2.vo.RequestHistory> getAuditTrailWithComments(
			List<RequestHistory> trackingList) {
		List<gov.va.genisis2.vo.RequestHistory> returnRequestHistory = new ArrayList<gov.va.genisis2.vo.RequestHistory>();
		for (RequestHistory requestHistory : trackingList) {
			int statusId = getLookUpService().getStatusId(
					requestHistory.getStatusDescription());
			List<CommentHistory> commentHistory = getCommentHistoryService()
					.getCommentHistoryByRequestIdStatusId(
							requestHistory.getRequest().getId(), requestHistory.getTaskId());
			gov.va.genisis2.vo.RequestHistory element = new gov.va.genisis2.vo.RequestHistory();
			element.setCommentHistory(commentHistory);
			element.setCreatedBy(requestHistory.getCreatedBy());
			element.setCreatedOn(requestHistory.getCreatedOn());
			element.setStatusDescription(requestHistory.getStatusDescription());
			element.setTitle(requestHistory.getTitle());
			element.setDescription(requestHistory.getDescription());
			element.setModifiedBy(requestHistory.getModifiedBy());
			element.setModifiedOn(requestHistory.getModifiedOn());
			
			returnRequestHistory.add(element);
			
		}
		return returnRequestHistory;
	}
}