'use strict';

angular.module('genisis').controller('requestsCtrl', ['$scope', 'request',
  '$location', 'user', '$rootScope', 'api', 'type',
  function ($scope, $requests, $location, $user, $rootScope, $api, $type) {

    $scope.requestFilter = '';
    $scope.orderBy = '-id';
    $scope.requests = [];
    $scope.type = $type;

    /**
     * Figure out what requests should be loaded for the user depending on their
     * current user permissions
     * @param string permission the user's permission level
     * @return object the configuration needed for the API call
     **/
    $scope.requestsConfig = function (permissions) {
      var config = {
        value: $user.id(),
        context: 'createdBy'
      };

      switch (permissions) {
        //see all of their requests and the submitted one they're manager for
      case 'manager':
        config = {
          value: $user.id(),
          context: 'dataManager'
        };
        break;
        //see all requests
      case 'admin':
        config = {
          value: null,
          context: null
        };
        break;
      }

      return config;
    };

    //load their requests
    $requests.load($scope.requestsConfig($user.permissions()))
      .then(function (loadedRequests) {
        $scope.requests = loadedRequests;
      });

    /**
     * Trying to create a new request, redirect to the create request page
     **/
    $scope.createRequest = function () {
      $location.path('/createRequest');
      return true;
    };

    /**
     * Trying to edit a request
     * @param int id the request id
     * @param boolean
     **/
    $scope.editRequest = function (id) {
      //make sure they have an ID and the request exists
      if (id && parseInt(id, 10)) {
        $location.path('/requests/' + id);
        return true;
      }

      return false;
    };

    /**
     * Checking to see whether or not they can edit a request
     *
     * @param string the status of the request
     * @return boolean
     **/
    $scope.canEdit = function (status) {
      switch (status) {
      case 'Draft':
      case 'Returned':
      case null:
        return true;
      }

      return false;
    };

    /**
     * Checking to see whether or not they can manage this request
     *
     * @param string the status of the request
     * @return boolean
     **/
    $scope.canManage = function (status) {
      switch (status) {
      case 'Submitted':
        if ($user.dataManager() || $user.admin()) {
          return true;
        }
      }

      return false;
    };

    /**
     * Checking to see whether or not they can approve or deny the data
     *
     * @param string the status of the request
     * @return boolean
     **/
    $scope.canData = function (status) {
      switch (status) {
      case 'Sent':
        if ($user.dataManager() || $user.admin()) {
          return true;
        }
      }

      return false;
    };

    /**
     * Checking to see whether or not they can approve or deny the data
     *
     * @param string the status of the request
     * @return boolean
     **/
    $scope.canResults = function (status) {
      switch (status) {
      case 'RequestAccepted':
        return true;
      }

      return false;
    };

    /**
     * Submitting a request to the data manager
     * @param Object request data
     * @return boolean
     **/
    $scope.submitRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id + '/submit', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {

          req.statusDescription = 'Submitted';

          //set a message that the request was submitted
          $rootScope.messages.push(
            'Your request has been submitted to your data manager.'
          );

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Submit request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to accept data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.acceptData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/acceptdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted.');

          req.statusDescription = 'RequestAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to reject data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.denyData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/denydata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request has been denied because the ' +
            'request cannot be fulfilled.');

          req.statusDescription = 'RequestNotAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Deny data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to accept data from the data supplier
     * @param Object request the request that's data is being confirmed for
     * @return boolean
     **/
    $scope.acceptResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/confirmdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted by the researcher.'
          );
          req.statusDescription = 'ResultsAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept results request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to reject the data from the data supplier
     * @param Object request the request that's data is being rejected for
     * @return boolean
     **/
    $scope.denyResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/rejectdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been rejected by the researcher.'
          );

          req.statusDescription = 'ResultsNotAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Deny results request API call failed.');
        }

        return false;
      });
    };

  }
]);