'use strict';

describe('requests/editRequest.js', function () {

  beforeEach(module('genisis'));

  var $controller, $scope, $user, $api, $rootScope, $sources, $studies,
    $requests, $types, $q, $httpBackend;

  beforeEach(
    inject(function (_$rootScope_, _$controller_, user, api,
      source, type, request, studyApproval, _$q_, _$httpBackend_) {
      $controller = _$controller_;
      $rootScope = _$rootScope_;
      $q = _$q_;
      $httpBackend = _$httpBackend_;
      $user = user;
      $sources = source;
      $requests = request;
      $studies = studyApproval;
      $types = type;
      $api = api;

      $httpBackend.when('GET').respond('test GET response');
      $httpBackend.when('PUT').respond('test PUT respoonse');

      $rootScope.config = {
        apiURL: '/api/'
      };

      $rootScope.errors = [];
      $rootScope.messages = [];
      $rootScope.warnings = [];

      $scope = $rootScope.$new();

      $controller('editRequestCtrl', {
        $scope: $scope,
        $requests: $requests,
        $rootScope: $rootScope,
        $sources: $sources,
        $types: $types,
        $studies: $studies,
        $api: $api,
        $user: $user,
        $routeParams: {
          id: 1
        }
      });
    })
  );

  it('sets $scope.requests to the $requests service', function () {
    expect($scope.requests).toEqual($requests);
  });

  it('sets $scope.request to an empty object', function () {
    expect($scope.request).toEqual({});
  });

  it('sets $scope.studyApprovals to the $studyApprovals service', function () {
    expect($scope.studyApprovals).toEqual($studies);
  });

  it('sets $scope.sources to the $sources service', function () {
    expect($scope.sources).toEqual($sources);
  });

  it('sets $scope.types to the $types service', function () {
    expect($scope.types).toEqual($types);
  });

  it('sets $scope.editable to false', function () {
    expect($scope.editable).toEqual(false);
  });

  it('sets $scope.submitted to false', function () {
    expect($scope.submitted).toEqual(false);
  });

  describe('$scope.loadRequest() function', function () {

    var fakeRequest1, fakeRequest2, fakeRequest3, fakeRequest4,
      fakeRequest5, fakeRequest6;

    beforeEach(function () {
      fakeRequest1 = {
        id: 1,
        statusDescription: 'Draft'
      };
      fakeRequest2 = {
        id: 2,
        statusDescription: 'Returned'
      };
      fakeRequest3 = {
        id: 3,
        statusDescription: 'Submitted'
      };
      fakeRequest4 = {
        id: 4,
        statusDescription: 'Unknown',
        reason: 'This is my reason'
      };
      fakeRequest5 = {
        id: 5,
        statusDescription: 'Sent'
      };
      fakeRequest6 = {
        id: 6,
        statusDescription: 'RequestAccepted'
      };

      spyOn($requests, 'load').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve(
            [fakeRequest1, fakeRequest2, fakeRequest3,
              fakeRequest4, fakeRequest5, fakeRequest6
            ]
          );
          return deferred.promise;
        }
      );
    });

    it('finds a draft request and sets $scope.editable = true',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest1;
          }
        );

        $scope.loadRequest(1);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest1);
        expect($scope.editable).toEqual(true);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      });

    it('finds a returned request and sets $scope.editable = true',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest2;
          }
        );

        $scope.loadRequest(2);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest2);
        expect($scope.editable).toEqual(true);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      });

    it('sets $scope.editable = false and $scope.submitted = false for ' +
      'regular users when $request.statusDescription = Submitted',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest3;
          }
        );

        spyOn($user, 'researcher').and.returnValue(true);
        spyOn($user, 'dataManager').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(false);

        $scope.loadRequest(3);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest3);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    it('sets $scope.submitted = true if the user is a dataManager',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest3;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);

        $scope.loadRequest(3);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest3);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(true);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    it('sets $scope.submitted = true if the user is an admin',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest3;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);

        $scope.loadRequest(3);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest3);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(true);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    it('sets $scope.sent = true if the user is a dataManager',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest5;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);

        $scope.loadRequest(5);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest5);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(true);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    it('sets $scope.sent = true if the user is an admin',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest5;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);

        $scope.loadRequest(5);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest5);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(true);
        expect($scope.accepted).toEqual(false);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    it('sets $scope.accepted = true if the user is a dataManager',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest6;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(true);
        spyOn($user, 'admin').and.returnValue(false);

        $scope.loadRequest(6);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest6);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(true);
        expect($rootScope.warnings).toEqual([]);
      }
    );


    it('sets $scope.accepted = true if the user is an admin',
      function () {

        spyOn($requests, 'find').and.callFake(
          function () {
            return fakeRequest6;
          }
        );

        spyOn($user, 'researcher').and.returnValue(false);
        spyOn($user, 'dataManager').and.returnValue(false);
        spyOn($user, 'admin').and.returnValue(true);

        $scope.loadRequest(6);
        $scope.$apply();
        expect($scope.request).toEqual(fakeRequest6);
        expect($scope.editable).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect($scope.sent).toEqual(false);
        expect($scope.accepted).toEqual(true);
        expect($rootScope.warnings).toEqual([]);
      }
    );

    // it(
    //   'pushes request.reason field to $rootScope.warnings if it exists',
    //   function () {

    //     spyOn($requests, 'find').and.callFake(
    //       function () {
    //         return fakeRequest4;
    //       }
    //     );

    //     $scope.loadRequest(4);
    //     $scope.$apply();
    //     expect($rootScope.warnings).toEqual(['This is my reason']);
    //   }
    // );
  });

  describe('$scope.loadLogs() function', function () {
    it('exists', function () {
      expect($scope.loadLogs).toBeDefined();
    });

    // it('loads log data', function () {

    //   var fakeLogs = [{commentHistory: []}, {commentHistory: []}];

    //   spyOn($requests, 'logs').and.callFake(
    //     function () {
    //       var deferred = $q.defer();
    //       deferred.resolve(fakeLogs);
    //       return deferred.promise;
    //     });

    //   $scope.loadLogs();
    //   $scope.$apply();
    //   expect($scope.logs).toEqual(fakeLogs);
    // });
  });

  describe('$scope.saveRequest() function', function () {

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.saveRequest(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.saveRequest(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('checks for a valid request title', function () {
      var testRequest = {};
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must enter a request title.'
      ]);
    });

    it('checks for a valid source', function () {
      var testRequest = {
        title: 'test title'
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must select a source.'
      ]);
    });

    it('checks for a valid study approval', function () {
      var testRequest = {
        title: 'test title',
        source: 1
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must select an approved study.'
      ]);
    });

    it('checks for a valid request type', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must select a request type.'
      ]);
    });

    it('checks for a valid notify collaborators', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual(
        [
          'You must select whether or not to notify collaborators.'
        ]
      );
    });

    it('checks for a valid notify collaborators', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual(
        [
          'You must select whether or not to notify collaborators.'
        ]
      );
    });

    it('checks for a valid request description', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true
      };
      expect($scope.saveRequest(testRequest)).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['You must enter a request description.']
      );
    });

    it('creates a request', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.saveRequest(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([
        'Your changes have been saved.'
      ]);
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.saveRequest(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.saveRequest(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([
        'Save request API call failed.'
      ]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.submitRequest() function', function () {
    it('exists', function () {
      expect($scope.submitRequest).toBeDefined();
    });

    it('clears out any existing $rootScope.errors', function () {
      var testError = 'this is a test error';

      $rootScope.errors.push(testError);

      $scope.submitRequest({});
      $scope.$apply();
      expect($rootScope.errors.indexOf(testError)).toEqual(-1);
    });

    it('clears out any existing $rootScope.messages', function () {
      var testMessage = 'this is a test error';

      $rootScope.messages.push(testMessage);

      $scope.submitRequest({});
      $scope.$apply();
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('returns $rootScope.errors if the request has no title',
      function () {

        var fakeRequest = {};

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual([
          'You must enter a request title.'
        ]);
      });

    it('returns $rootScope.errors if the request has no source',
      function () {

        var fakeRequest = {
          title: 'test title'
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual([
          'You must select a source.'
        ]);
      });

    it('returns $rootScope.errors if the request has no study approval',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual(
          ['You must select an approved study.']
        );
      }
    );

    it('returns $rootScope.errors if the request has no request type',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual([
          'You must select a request type.'
        ]);
      }
    );

    it('returns $rootScope.errors if the request has no notify value',
      function () {
        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2,
          type: 3
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual(
          [
            'You must select whether or not to notify collaborators.'
          ]
        );
      }
    );

    it('returns $rootScope.errors if the request has no description',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2,
          type: 3,
          notify: true
        };

        expect($scope.submitRequest(fakeRequest)).toEqual(false);
        expect($rootScope.errors).toEqual(
          ['You must enter a request description.']
        );
      }
    );

    it('submits the request by calling $api.update() function',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2,
          type: 3,
          notify: true,
          description: 'test description'
        };

        spyOn($api, 'update').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        });

        var promise = $scope.submitRequest(fakeRequest);

        promise.then(function (data) {
          expect(data).toEqual(true);
          expect($scope.editable).toEqual(false);
          expect(fakeRequest.statusDescription).toEqual(
            'Submitted');
          expect($rootScope.messages).toEqual(
            [
              'Your request has been submitted to your data manager.'
            ]
          );
          expect($rootScope.errors).toEqual([]);
        });

        $scope.$apply();
        expect($api.update.calls.count()).toEqual(1);
      });

    it('returns $rootScope.errors if the api call fails', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          message: 'api error',
          response: {}
        });
        return deferred.promise;
      });

      var promise = $scope.submitRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });

    it('returns $rootScope.errors if the api errors', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({});
        return deferred.promise;
      });

      var promise = $scope.submitRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual([
          'Submit request API call failed.'
        ]);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });
  });

  describe('$scope.approveRequest() function', function () {
    it('exists', function () {
      expect($scope.approveRequest).toBeDefined();
    });

    it('clears out any existing $rootScope.errors', function () {
      var testError = 'this is a test error';

      $rootScope.errors.push(testError);

      $scope.approveRequest({});
      $scope.$apply();
      expect($rootScope.errors.indexOf(testError)).toEqual(-1);
    });

    it('clears out any existing $rootScope.messages', function () {
      var testMessage = 'this is a test error';

      $rootScope.messages.push(testMessage);

      $scope.approveRequest({});
      $scope.$apply();
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('approves the request by calling $api.update() function',
      function () {

        var fakeRequest = {
          title: 'test title',
          source: 1,
          studyApproval: 2,
          type: 3,
          notify: true,
          description: 'test description',
          statusDescription: 'Submitted'
        };

        spyOn($api, 'update').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        });

        var promise = $scope.approveRequest(fakeRequest);

        promise.then(function (data) {
          expect(data).toEqual(true);
          expect($scope.editable).toEqual(false);
          expect(fakeRequest.statusDescription).toEqual('Sent');
          expect($rootScope.messages).toEqual(
            ['The request has been approved.']
          );
          expect($rootScope.errors).toEqual([]);
        });

        $scope.$apply();
        expect($api.update.calls.count()).toEqual(1);
      });

    it('returns $rootScope.errors if the api call fails', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          message: 'api error',
          response: {}
        });
        return deferred.promise;
      });

      var promise = $scope.approveRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });

    it('returns $rootScope.errors if the api errors', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({});
        return deferred.promise;
      });

      var promise = $scope.approveRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual([
          'Approve request API call failed.'
        ]);
      });

      $scope.$apply();
      expect($api.update.calls.count()).toEqual(1);
    });
  });

  describe('$scope.returnRequest() function', function () {

    it('exists', function () {
      expect($scope.returnRequest).toBeDefined();
    });

    it('clears out any existing $rootScope.errors', function () {
      var testError = 'this is a test error';

      $rootScope.errors.push(testError);

      $scope.returnRequest('test', {
        id: 1
      });
      $scope.$apply();
      expect($rootScope.errors.indexOf(testError)).toEqual(-1);
    });

    it('clears out any existing $rootScope.messages', function () {
      var testMessage = 'this is a test error';

      $rootScope.messages.push(testMessage);

      $scope.approveRequest('test');
      $scope.$apply();
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('returns false if no message parameter is provided', function () {
      var results = $scope.returnRequest(null);
      expect(results).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        ]
      );
    });

    it('rejects a request', function () {

      var fakeRequest = {
        id: 'fake',
        statusDescription: 'Submitted'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      var promise = $scope.returnRequest('test mesage', fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($scope.submitted).toEqual(false);
        expect(fakeRequest.statusDescription).toEqual(
          'Returned');
        expect($rootScope.messages).toEqual(
          [
            'This request has been returned to the owner.'
          ]
        );
      });

      $scope.$apply();
    });

    it('returns $rootScope.errors if the API call fails', function () {

      var fakeRequest = {
        id: 'fake',
        statusDescription: 'Submitted'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error',
            response: {}
          });
          return deferred.promise;
        }
      );

      var promise = $scope.returnRequest('test mesage', fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect(fakeRequest.statusDescription).toEqual(
          'Submitted');
        expect($rootScope.messages).toEqual([]);
        expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
    });

    it('returns $rootScope.errors if the API does not respond',
      function () {

        var fakeRequest = {
          id: 'fake',
          statusDescription: 'Submitted'
        };

        spyOn($api, 'update').and.callFake(
          function () {
            var deferred = $q.defer();
            deferred.resolve({});
            return deferred.promise;
          }
        );

        var promise = $scope.returnRequest('test mesage', fakeRequest);

        promise.then(function (data) {
          expect(data).toEqual(false);
          expect($scope.submitted).toEqual(false);
          expect(fakeRequest.statusDescription).toEqual(
            'Submitted');
          expect($rootScope.messages).toEqual([]);
          expect($rootScope.errors).toEqual([
            'Reject request API call failed.'
          ]);
        });

        $scope.$apply();
      }
    );
  });

  describe('$scope.denyRequest() function', function () {

    it('exists', function () {
      expect($scope.denyRequest).toBeDefined();
    });

    it('clears out any existing $rootScope.errors', function () {
      var testError = 'this is a test error';

      $rootScope.errors.push(testError);

      $scope.denyRequest('test', {
        id: 1
      });
      $scope.$apply();
      expect($rootScope.errors.indexOf(testError)).toEqual(-1);
    });

    it('clears out any existing $rootScope.messages', function () {
      var testMessage = 'this is a test error';

      $rootScope.messages.push(testMessage);

      $scope.denyRequest('test', {
        id: 1
      });
      $scope.$apply();
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('returns false if no message parameter is provided', function () {
      var results = $scope.denyRequest(null);
      expect(results).toEqual(false);
      expect($rootScope.errors).toEqual(
        ['Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        ]
      );
    });

    it('denies a request', function () {

      var fakeRequest = {
        id: 'fake',
        statusDescription: 'Submitted'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      var promise = $scope.denyRequest('test mesage', fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($scope.submitted).toEqual(false);
        expect($scope.editable).toEqual(false);
        expect(fakeRequest.statusDescription).toEqual('Denied');
        expect($rootScope.messages).toEqual(
          [
            'This request has been denied.'
          ]
        );
      });

      $scope.$apply();
    });

    it('returns $rootScope.errors if the API call fails', function () {

      var fakeRequest = {
        id: 'fake',
        statusDescription: 'Submitted'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error',
            response: {}
          });
          return deferred.promise;
        }
      );

      var promise = $scope.denyRequest('test mesage', fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($scope.submitted).toEqual(false);
        expect(fakeRequest.statusDescription).toEqual(
          'Submitted');
        expect($rootScope.messages).toEqual([]);
        expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
    });

    it('returns $rootScope.errors if the API does not respond',
      function () {

        var fakeRequest = {
          id: 'fake',
          statusDescription: 'Submitted'
        };

        spyOn($api, 'update').and.callFake(
          function () {
            var deferred = $q.defer();
            deferred.resolve({});
            return deferred.promise;
          }
        );

        var promise = $scope.denyRequest('test mesage', fakeRequest);

        promise.then(function (data) {
          expect(data).toEqual(false);
          expect($scope.submitted).toEqual(false);
          expect(fakeRequest.statusDescription).toEqual(
            'Submitted');
          expect($rootScope.messages).toEqual([]);
          expect($rootScope.errors).toEqual([
            'Deny request API call failed.'
          ]);
        });

        $scope.$apply();
      }
    );
  });

  describe('$scope.acceptData() function', function () {
    it('exists', function () {
      expect($scope.acceptData).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.acceptData(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.acceptData(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('approves the requests data', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.acceptData(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([
        'This request\'s data has been accepted.'
      ]);
      expect($scope.editable).toEqual(false);
      expect($scope.submitted).toEqual(false);
      expect($scope.sent).toEqual(false);
      expect($scope.accepted).toEqual(true);
      expect(testRequest.statusDescription).toEqual(
        'RequestAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.acceptData(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.acceptData(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([
        'Accept data request API call failed.'
      ]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.denyData() function', function () {
    it('exists', function () {
      expect($scope.denyData).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.denyData(testMessage, testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.denyData(testMessage, testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('Denies the requests data', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.denyData('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([
        'This request has been denied because the request cannot be fulfilled.'
      ]);
      expect($scope.editable).toEqual(false);
      expect($scope.submitted).toEqual(false);
      expect($scope.sent).toEqual(false);
      expect($scope.accepted).toEqual(false);
      expect(testRequest.statusDescription).toEqual(
        'RequestNotAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.denyData('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.denyData('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([
        'Deny data request API call failed.'
      ]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.acceptResults() function', function () {
    it('exists', function () {
      expect($scope.acceptResults).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.acceptResults(testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.acceptResults(testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('accepts the requests results', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.acceptResults(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([
        'This request\'s data has been accepted by the researcher.'
      ]);
      expect($scope.editable).toEqual(false);
      expect($scope.submitted).toEqual(false);
      expect($scope.sent).toEqual(false);
      expect($scope.accepted).toEqual(false);
      expect(testRequest.statusDescription).toEqual(
        'ResultsAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.acceptResults(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.acceptResults(testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([
        'Accept results request API call failed.'
      ]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });
  });

  describe('$scope.denyResults() function', function () {
    it('exists', function () {
      expect($scope.denyResults).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.errors.push(testMessage);
      $scope.denyResults(testMessage, testRequest);
      expect($rootScope.errors.indexOf(testMessage)).toEqual(-1);
    });

    it('clears out existing $rootScope.messages', function () {
      var testRequest = {};
      var testMessage = 'this is a test';
      $rootScope.messages.push(testMessage);
      $scope.denyResults(testMessage, testRequest);
      expect($rootScope.messages.indexOf(testMessage)).toEqual(-1);
    });

    it('Denies the requests results', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        status: 'Sent',
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: {}
          });
          return deferred.promise;
        }
      );

      $scope.denyResults('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([
        'This request\'s data has been rejected by the researcher.'
      ]);
      expect($scope.editable).toEqual(false);
      expect($scope.submitted).toEqual(false);
      expect($scope.sent).toEqual(false);
      expect($scope.accepted).toEqual(false);
      expect(testRequest.statusDescription).toEqual(
        'ResultsNotAccepted');
    });

    it('returns an error if the API fails', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error'
          });
          return deferred.promise;
        });

      $scope.denyResults('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual(['api error']);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });

    it('handles no api response', function () {
      var testRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($api, 'update').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        });

      $scope.denyResults('test', testRequest);
      $scope.$apply();

      expect($rootScope.errors).toEqual([
        'Deny results request API call failed.'
      ]);
      expect($api.update.calls.count()).toEqual(1);
      expect($rootScope.messages).toEqual([]);
    });
  });
});