'use strict';

angular.module('genisis').controller('editRequestCtrl', ['$scope', 'request',
  '$rootScope', 'source', 'type', 'studyApproval', 'api', 'user',
  '$location', '$routeParams',
  function ($scope, $requests, $rootScope, $sources, $types,
    $studyApprovals, $api, $user, $location, $routeParams) {

    $scope.requests = $requests;
    $scope.request = {};
    $scope.studies = {};
    $scope.studyApprovals = $studyApprovals;
    $scope.sources = $sources;
    $scope.types = $types;
    $scope.editable = false;
    $scope.submitted = false;
    $scope.commentable = false;
    $scope.sent = false;
    $scope.accepted = false;
    $scope.logs = [];
    $scope.loadingRequest = true;
    $scope.loadingLogs = true;

    $scope.visibleComments = [{
        comment: 'Please check that you have the correct study selected,' +
          'it seems like this will not work with the current ' +
          'study you selected.',
        user: 'DataManager1',
        date: '12:50:30 03/07/2017'
      },
      {
        comment: 'Sure, will do!',
        user: 'Researcher1',
        date: '01:30:05 03/07/2017'
      },
    ];

    $studyApprovals.load().then(function (loadedApprovals) {
      $scope.studies = loadedApprovals;
    });

    /**
     * Loads data for a specific request
     * @param int requestID the ID of the request to load
     * @return boolean
     **/
    $scope.loadRequest = function (requestID) {

      //load the request matching the given id
      $requests.load().then(function ( /*request*/ ) {
        $scope.request = $requests.find(requestID);

        switch ($scope.request.statusDescription) {
        case 'Draft':
          $scope.editable = true;
          $scope.commentable = false;
          break;
        case 'Returned':
          $scope.editable = true;
          $scope.commentable = true;
          break;
        case null:
          $scope.editable = false;
          break;
        case 'Submitted':
          if ($user.dataManager() || $user.admin()) {
            $scope.submitted = true;
          }
          $scope.commentable = true;
          break;
        case 'Sent':
          if ($user.dataManager() || $user.admin()) {
            $scope.sent = true;
          }
          $scope.commentable = true;
          break;
        case 'RequestAccepted':
          //if ($user.dataManager() || $user.admin()) {
          $scope.accepted = true;
          //}
          $scope.commentable = true;
          break;
        }

        if ($location.search().created === 'true') {
          $rootScope.messages.push('Request created successfully.');
        }

        // if ($scope.request.reason !== undefined &&
        //   $scope.request.reason !== null) {
        //   $rootScope.warnings.push($scope.request.reason);
        // }

        return true;
      });
    };

    $scope.resetForm = function () {
      $scope.request.comments = null;
    };

    $scope.loadRequestByID = function (requestID) {
      $requests.findByID(requestID).then(function (request) {
        $scope.request = request;

        switch ($scope.request.statusDescription) {
        case 'Draft':
          $scope.editable = true;
          $scope.commentable = false;
          break;
        case 'Returned':
          $scope.editable = true;
          $scope.commentable = true;
          break;
        case null:
          $scope.editable = false;
          break;
        case 'Submitted':
          if ($user.dataManager() || $user.admin()) {
            $scope.submitted = true;
          }
          $scope.commentable = true;
          break;
        case 'Sent':
          if ($user.dataManager() || $user.admin()) {
            $scope.sent = true;
          }
          $scope.commentable = true;
          break;
        case 'RequestAccepted':
          //if ($user.dataManager() || $user.admin()) {
          $scope.accepted = true;
          //}
          $scope.commentable = true;
          break;
        }

        if ($location.search().created === 'true') {
          $rootScope.messages.push('Request created successfully.');
        }

        // if ($scope.request.reason !== undefined &&
        //   $scope.request.reason !== null) {
        //   $rootScope.warnings.push($scope.request.reason);
        // }

        $scope.loadingRequest = false;

        $scope.setCommentability(request);

        return true;
      });
    };

    $scope.loadLogs = function (requestID) {

      //load the logs for this request
      $requests.logs(requestID).then(function (logData) {
        $scope.logs = $requests.parseLogs(logData);
        if ($scope.logs) {
          $scope.loadingLogs = false;
        }

      });
    };

    //load the request matching the given id
    $scope.studies = $studyApprovals.load();

    /**
     * Save updates made to a request
     * @param Object request
     * @return boolean
     **/
    $scope.saveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id, {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('Your changes have been saved.');
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Save request API call failed.');
        }

        return false;
      });
    };

    /**
     * Submitting a request to the data manager
     * @param Object request data
     * @return boolean
     **/
    $scope.submitRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id + '/submit', {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {

          //this is no longer editable
          $scope.editable = false;
          req.statusDescription = 'Submitted';

          //set a message that the request was submitted
          $rootScope.messages.push(
            'Your request has been submitted to your data manager.'
          );

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Submit request API call failed.');
        }

        return false;
      });
    };

    /**
     * Approve a request
     * @param Object request
     * @return boolean
     **/
    $scope.approveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      var data = {
        createdBy: $user.id()
      };

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/sent', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('The request has been approved.');
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = true;
          req.statusDescription = 'Sent';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Approve request API call failed.');
        }

        return false;
      });
    };

    /**
     * Save updates made to a request
     * @param string message why this request is being rejected
     * @param Object request the request that's being returned
     * @return boolean
     **/
    $scope.returnRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.message = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/return', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request has been returned to the owner.'
          );
          $scope.submitted = false;
          req.statusDescription = 'Returned';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Reject request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers and admins to deny submitted requests
     * @param string message why this request is being denied
     * @param Object request the request that's being denied
     * @return boolean
     **/
    $scope.denyRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/deny', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('This request has been denied.');
          $scope.editable = false;
          $scope.submitted = false;
          req.statusDescription = 'Denied';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Deny request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to accept data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.acceptData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      var data = {
        createdBy: $user.id()
      };

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/acceptdata', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted.');
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = true;
          req.statusDescription = 'RequestAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to reject data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.denyData = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/denydata', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request has been denied because the ' +
            'request cannot be fulfilled.');

          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;

          req.statusDescription = 'RequestNotAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Deny data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to accept data from the data supplier
     * @param Object request the request that's data is being confirmed for
     * @return boolean
     **/
    $scope.acceptResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      var data = {
        createdBy: $user.id()
      };

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/confirmdata', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted by the researcher.'
          );
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;
          req.statusDescription = 'ResultsAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept results request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to reject the data from the data supplier
     * @param Object request the request that's data is being rejected for
     * @return boolean
     **/
    $scope.denyResults = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/rejectdata', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been rejected by the researcher.'
          );

          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;

          req.statusDescription = 'ResultsNotAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Deny results request API call failed.');
        }

        return false;
      });
    };

    $scope.submitComment = function (request) {
      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!request.comments) {
        $rootScope.errors.push(
          'Please enter a comment and try again.'
        );

        return false;
      }

      return $api.create('comment', {
        data: {
          requestId: request.id,
          status: request.statusDescription,
          comments: request.comments,
          createdBy: $user.id()
        }
      }).then(function (comment) {
        if (comment.success && comment.response) {
          $rootScope.messages.push(
            'Comment posted.'
          );

          $scope.loadLogs(comment.response.requestId);
          $scope.resetForm();

          return true;
        } else if (comment && comment.message) {
          $rootScope.errors.push(comment.message);
        } else {
          $rootScope.errors.push(
            'Failed to post comment, please try again.'
          );
        }

        return false;
      });
    };

    $scope.requestHistory = function (request) {
      //make sure they have an ID and the request exists
      if (request.id && parseInt(request.id, 10)) {
        $location.path('/requestHistory/' + request.id);
        return true;
      }

      return false;
    };

    $scope.setCommentability = function (request) {
      // Is user a Requestor or not?
      var currentStatus = request.statusDescription;
      if ($user.researcher()) {
        if (currentStatus === 'Returned') {
          $scope.commentable = true;
        } else {
          $scope.commentable = false;
        }
      } else {
        if (currentStatus === 'Draft' ||
          currentStatus === 'Denied' ||
          currentStatus === 'ResultsAccepted' ||
          currentStatus === 'ResultsNotAccepted') {
          $scope.commentable = false;
        } else {
          $scope.commentable = true;
        }
      }
    };

    // $scope.loadRequest($routeParams.id);
    $scope.loadRequestByID($routeParams.id);
    $scope.loadLogs($routeParams.id);

  }
]);