/**
 * 
 */
package gov.va.genisis2.controller;

import gov.va.genisis2.bo.BusinessService;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;

import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author 586338
 *
 */

@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/services")
public class Genisis2Controller {

	private final org.slf4j.Logger Logger = LoggerFactory
			.getLogger(Genisis2Controller.class);

	private final static String REQUEST_SENT = "Sent";
	private final static String REQUEST_DENIED = "Denied";
	private final static String REQUEST_RETURNED = "Returned";
	private final static String REQUEST_UPDATE = "Update";
	private final static String REQUEST_SUBMIT = "Submitted";

	private final static String REQUEST_ACCEPTED = "RequestAccepted";
	private final static String REQUEST_NOT_ACCEPTED = "RequestNotAccepted";
	private final static String RESULTS_ACCEPTED = "ResultsAccepted";
	private final static String RESULTS_NOT_ACCEPTED = "ResultsNotAccepted";

	private BusinessService businessService;

	/* L D A P R E S T C A L L S */

	@RequestMapping(value = "/users/{email}/email", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getLoginDetailsByEmail(@PathVariable String email) {
		Logger.info(" Get ldap user by email.");
		ResponseWrapper wrapper = businessService
				.ldapGetLoginDetailsByEmail(email);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/users/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getLoginDetailsById(@PathVariable String id) {
		Logger.info(" Get ldap user by Id.");
		ResponseWrapper wrapper = businessService.ldapGetLoginDetailsById(id);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	/* R E Q U E S T R E S T C A L L S */

	@RequestMapping(value = "/requests", method = RequestMethod.POST, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	public ResponseEntity createRequest(@RequestBody Request request) {
		Logger.info("Create Request ");
		ResponseWrapper wrapper = businessService.createRequest(request);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{studyApprovalId}/studyApproval", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequestsByStudyApproval(
			@PathVariable int studyApprovalId) {
		Logger.info("getAllRequestsByStudyApproval ");
		ResponseWrapper wrapper = businessService
				.getAllRequestsByStudyApproval(studyApprovalId);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsById(@PathVariable int id) {
		Logger.info("Get Request  by Id ");
		ResponseWrapper wrapper = businessService.getRequestByID(id);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{uid}/createdBy", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsByUser(@PathVariable String uid) {
		Logger.info("Get Requests  by user ");
		ResponseWrapper wrapper = businessService.getRequestsByUID(uid);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequests() {
		Logger.info("Get All Requests ");
		ResponseWrapper wrapper = businessService.getAllRequests();
		Logger.info("Get All Requests ");
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}/dataManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByDataManager(@PathVariable String id) {
		Logger.info("Request by DataManger  ");
		ResponseWrapper wrapper = businessService.getRequestsByUID(id);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{status}/status", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByStatus(@PathVariable String status) {
		Logger.info("Requests by Status /approved/status");
		ResponseWrapper wrapper = businessService.getAllRequestsByStaus(status);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Request Update");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_UPDATE);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}/submit", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity submitRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Request Submit ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_SUBMIT);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);

	}

	@RequestMapping(value = "/requests/{id}/sent", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity sentRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Request Sent/Approved ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_SENT);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);

	}

	@RequestMapping(value = "/requests/{id}/deny", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity denyRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Request Deny  ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_DENIED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);

	}

	@RequestMapping(value = "/requests/{id}/return", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity returnRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info(" Request Returned  ");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_RETURNED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);

	}

	@RequestMapping(value = "/requests/{id}/acceptdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity fulfillRequest(@PathVariable int id,
			@RequestBody Request request) {
	Logger.info("Accept Data");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_ACCEPTED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}/denydata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity unFulfillRequest(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Reject Data");
		ResponseWrapper wrapper = businessService.persist(id, request,
				REQUEST_NOT_ACCEPTED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}/rejectdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity rejectData(@PathVariable int id,
			@RequestBody Request request) {
	   Logger.info("Reject Results");
		ResponseWrapper wrapper = businessService.persist(id, request,
				RESULTS_NOT_ACCEPTED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/requests/{id}/confirmdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity acceptData(@PathVariable int id,
			@RequestBody Request request) {
		Logger.info("Accept Results");
		ResponseWrapper wrapper = businessService.persist(id, request,
				RESULTS_ACCEPTED);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	/* S T U D Y A P P R O V A L R E S T C A L L S */

	@RequestMapping(value = "/studyApprovals", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity createStudyApproval(
			@RequestBody StudyApproval studyApproval) {
		Logger.info("Create Study Approval");
		ResponseWrapper wrapper = businessService
				.createStudyApproval(studyApproval);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateStudyApproval(@PathVariable int id,
			@RequestBody StudyApproval studyApproval) {
		if (id > 0) {
			studyApproval.setId(id);
			ResponseWrapper wrapper = businessService
					.updateStudyApproval(studyApproval);
			Logger.info("Updated Study Approval");
			return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
		} else {
			ResponseWrapper wrapper = new ResponseWrapper();
			Logger.info("No ID Found");
			wrapper.setSuccess(false);
			wrapper.setMessage("No ID Found");
			return new ResponseEntity<ResponseWrapper>(wrapper,
					HttpStatus.NOT_MODIFIED);
		}
	}

	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalById(@PathVariable int id) {
		Logger.info("Study Approval by Id");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByID(id);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/studyApprovals/{uid}/{user}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalByUser(@PathVariable String uid) {
		Logger.info("Study Approval by User");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByUID(uid);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	@RequestMapping(value = "/studyApprovals", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllStudyApprovals() {
		Logger.info("All Study Approvals");
		ResponseWrapper wrapper = businessService.getStudyApprovals();
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	/* JBPM R E S T C A L L S */
	@RequestMapping(value = "/requests/{id}/log", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestTrackingById(@PathVariable int id) {
		Logger.info("Get Request Tracking by Id ");
		ResponseWrapper wrapper = businessService.getRequestTrackingByID(id);
		return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.OK);
	}

	public BusinessService getBusinessService() {
		return businessService;
	}

	@Autowired
	public void setBusinessService(BusinessService businessService) {
		this.businessService = businessService;
	}

}
