'use strict';

describe('services/request.js', function () {

  var $request, $q, $scope, $httpBackend, $rootScope, $api = null;

  beforeEach(module('genisis'));

  beforeEach(function () {
    inject(function (request, api, _$q_, _$rootScope_, _$httpBackend_) {
      $request = request;
      $api = api;
      $q = _$q_;
      $rootScope = _$rootScope_;
      $scope = $rootScope.$new();
      $httpBackend = _$httpBackend_;
      $rootScope.errors = [];

      $httpBackend.when('GET').respond('sample GET response');
    });
  });

  it('service exists', function () {
    expect($request).not.toBeNull();
  });

  describe('request.toName() function', function () {

    it('returns None when no id is given', function () {
      expect($request.toName()).toEqual('None');
    });

    it('returns the name of the request given the id', function () {

      var testValue = {
        title: 'Test Request'
      };

      spyOn($request, 'find').and.callFake(function () {
        return testValue;
      });

      expect($request.toName(1)).toEqual(testValue.title);
    });

    it('returns the name of the request given the id', function () {

      spyOn($request, 'find').and.callFake(function () {
        return;
      });

      expect($request.toName(-1)).toEqual('None');
    });
  });

  describe('request.load() function()', function () {
    it('returns a list of loaded requests', function () {

      var request1 = {
        id: 1,
        title: 'test'
      };
      var request2 = {
        id: 2,
        title: 'fake'
      };

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: [request1, request2]
        });
        return deferred.promise;
      });

      var promise = $request.load();

      promise.then(function (data) {
        expect(data).toEqual([request1, request2]);
      });

      $scope.$apply();
      expect($request.get()).toEqual([request1, request2]);
    });

    it('returns an array of requests even if there is only one',
      function () {

        var request1 = {
          id: 1,
          title: 'test'
        };

        spyOn($api, 'retrieve').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: request1
          });
          return deferred.promise;
        });

        var promise = $request.load();

        promise.then(function (data) {
          expect(data).toEqual([request1]);
        });

        $scope.$apply();
        expect($request.get()).toEqual([request1]);
      });

    it('returns an empty array if there is no requests', function () {

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          response: null
        });
        return deferred.promise;
      });

      var promise = $request.load();

      promise.then(function (data) {
        expect(data).toEqual([]);
      });

      $scope.$apply();
      expect($request.get()).toEqual([]);
    });
  });

  describe('$request.find() function', function () {

    it('exists', function () {
      expect($request.find).toBeDefined();
    });

    it('returns null if no id is given', function () {
      $request.load();
      expect($request.find()).toBeNull();
    });

    it('returns null if the id is not found', function () {
      $request.load();
      expect($request.find('doestExist')).toBeNull();
    });

    it('returns the request if the id exists', function () {
      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: [{
            id: 1,
            name: 'test'
          }]
        });
        return deferred.promise;
      });

      var promise = $request.load();

      promise.then(function () {
        expect($request.find('1')).toEqual({
          id: 1,
          name: 'test'
        });
      });

      $scope.$apply();
    });
  });

  describe('$request.get() function', function () {
    it('exists', function () {
      expect($request.get).toBeDefined();
    });

    it('returns an empty array if no requests are loaded', function () {
      expect($request.get()).toEqual([]);
    });

    it('returns requests if they are loaded', function () {

      var request1 = {
        id: 'test1'
      };
      var request2 = {
        id: 'test2'
      };

      spyOn($api, 'retrieve').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: [request1, request2]
        });
        return deferred.promise;
      });

      var promise = $request.load();

      promise.then(function () {
        expect($request.get()).toEqual([request1, request2]);
      });

      $scope.$apply();
    });
  });

  describe('$request.valid() function', function () {
    it('exists', function () {
      expect($request.valid).toBeDefined();
    });

    it('returns a $rootScope.error if there is no title', function () {
      var fakeRequest = {};
      expect($request.valid(fakeRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must enter a request title.'
      ]);
    });

    it('returns a $rootScope.error if there is no source', function () {
      var fakeRequest = {
        title: 'test'
      };
      expect($request.valid(fakeRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must select a source.'
      ]);
    });
  });

  describe('$request.validDraft() function', function () {
    it('exists', function () {
      expect($request.validDraft).toBeDefined();
    });

    it('returns a $rootScope.error if there is no title', function () {
      var fakeRequest = {};
      expect($request.validDraft(fakeRequest)).toEqual(false);
      expect($rootScope.errors).toEqual([
        'You must enter a request title.'
      ]);
    });

    it('returns true if there is a title', function () {
      var fakeRequest = {
        title: 'testing 123'
      };
      expect($request.validDraft(fakeRequest)).toEqual(true);
      expect($rootScope.errors).toEqual([]);
    });
  });
});