'use strict';

angular.module('genisis').service('request', ['api', '$rootScope',
  function ($api, $rootScope) {
    var _requests = [];

    /**
     * Return the name of the request given the id
     * @param int id the id number of the source
     * @return string source name for that id
     **/
    this.toName = function (id) {
      if (!id) {
        return 'None';
      }

      var request = this.find(id);
      return (request) ? request.title : 'None';
    };

    /**
     * Load all requests or specific requests for a user
     * @param int uid user id
     * @return array{} list of requests that were loaded
     **/
    this.load = function (payload) {
      return $api.retrieve('requests', payload).then(
        function (request) {
          if (request.success) {

            if (request.response instanceof Array) {
              _requests = request.response;
            } else {
              _requests = [request.response];
            }

            return _requests;
          }

          return [];
        }
      );
    };

    /**
     * Load the logs for a specific request
     * @param int request id
     * @return array{} list of requests that were loaded
     **/
    this.logs = function (requestID) {

      var logs = [];

      return $api.retrieve('requests', {
        value: requestID,
        context: 'log'
      }).then(
        function (request) {
          if (request.success) {

            if (request.response instanceof Array) {
              logs = request.response;
            } else {
              logs = [request.response];
            }

            return logs;
          }

          return [];
        }
      );
    };

    /**
     * Find a specific request in the list
     * Note: requests must have already been loaded
     * @param int id the request id
     * @return Object|null
     **/
    this.find = function (id) {

      if (!id) {
        return null;
      }

      //make sure it's always an integer
      id = parseInt(id, 10);

      var req = null;

      angular.forEach(_requests, function (request) {
        if (request.id === id) {
          req = request;
        }
      });

      return req;
    };

    /**
     * Return the array of all loaded requests
     * @return array{}
     **/
    this.get = function () {
      return _requests;
    };

    /**
     * Validate the request
     * @param Object the request that needs validated
     * @return boolean
     **/
    this.valid = function (request) {

      if (!request.title) {
        $rootScope.errors.push('You must enter a request title.');
        return false;
      }

      if (!request.source) {
        $rootScope.errors.push('You must select a source.');
        return false;
      }

      if (!request.studyApproval) {
        $rootScope.errors.push('You must select an approved study.');
        return false;
      }

      if (!request.type) {
        $rootScope.errors.push(
          'You must select a request type.');

        return false;
      }

      if (typeof request.notify !== 'boolean') {
        $rootScope.errors.push(
          'You must select whether or not to notify collaborators.');

        return false;
      }

      if (!request.description) {
        $rootScope.errors.push(
          'You must enter a request description.');

        return false;
      }

      return true;
    };


    /**
     * Validate the draft request (before submitted)
     * @param Object the request that needs validated
     * @return boolean
     **/
    this.validDraft = function (request) {

      if (!request.title) {
        $rootScope.errors.push('You must enter a request title.');
        return false;
      }

      return true;
    };
  }
]);