'use strict';

angular.module('genisis').service('api', ['$rootScope', '$http', 'ENV',
  function ($rootScope, $http, ENV) {

    /**
     * Default configuration for making an API request
     **/
    var defaultConfig = {
      endpoint: null,
      options: null,
      headers: {
        'Content-Type': 'application/json'
      },
      payload: {
        value: '',
        context: '',
        data: ''
      },
      method: 'GET'
    };

    /**
     * Generate the configration object depending on the CRUD operation they
     * are trying to use
     * @param string method GET|POST|PUT|DELETE
     *   POST - create
     *   GET - retrieve
     *   PUT - update
     *   DELETE - duh...
     * @param string endpoint the endpoint to hit on the API
     * @param object payload an object with the value, context and data to send
     *  to the API
     * @param object options angular $http configuration options
     * @param object headers the headers to send with the request
     *   You shouldn't need this is most cases but there are some exceptions
     *   that make come up like file uploads
     * @return Object proper $http configuration settings
     **/
    this.generateConfig = function (method, endpoint, payload, options,
      headers) {

      var value = '';
      var context = '';
      var data = null;

      if (payload && payload.value) {
        value = '/' + payload.value;
      }

      if (payload && payload.context) {
        context = '/' + payload.context;
      }

      if (payload && payload.data) {
        data = payload.data;
      }

      return {
        method: method,
        endpoint: endpoint + value + context,
        data: data,
        options: options || defaultConfig.options,
        headers: headers || defaultConfig.headers
      };
    };

    /**
     * Process the request coming back from an $http call
     * @param Object the request object
     * @return Object the formatted request
     **/
    this.formatRequest = function (request) {
      if (request.status === 200 && request.data) {
        return request.data;
      } else {
        return {
          success: false,
          message: 'API request error',
          apiResponse: request
        };
      }
    };

    /**
     * Makes the $http request given the proper configuration options
     * @param Object configruation for the $http request
     * @return Object $http promise that resolves to request payload or error
     **/
    this.fetch = function (config) {
      var t = this;

      return $http({
        method: config.method,
        url: ENV.apiURL + config.endpoint,
        headers: config.headers,
        data: config.data
      }).then(
        function (request) {
          return t.formatRequest(request);
        }
      );
    };

    /**
     * Retrieve payload through the API
     * @param string endpoint what api service/value/context to call
     * @param object payload an object with the value, context and data to send
     *  to the API
     * @param object options $http angular configuration options
     * @param object headers the request headers to send
     * @return object $http promise that resolves the request payload
     **/
    this.retrieve = function (endpoint, payload, options, headers) {

      return this.fetch(this.generateConfig('GET', endpoint, payload,
        options, headers));
    };

    /**
     * Update a record through the API
     * @param string endpoint what api service/value/context to call
     * @param object payload an object with the value, context and data to send
     *  to the API
     * @param object options $http angular configuration options
     * @param object headers the request headers to send
     * @return object $http promise that resolves the request payload
     **/
    this.update = function (endpoint, payload, options, headers) {
      return this.fetch(this.generateConfig('PUT', endpoint, payload,
        options, headers));
    };

    /**
     * Create a record through the API
     * @param string endpoint what api service/value/context to call
     * @param object payload an object with the value, context and data to send
     *  to the API
     * @param object options $http angular configuration options
     * @param object headers the request headers to send
     * @return object $http promise that resolves the request payload
     **/
    this.create = function (url, payload, options, headers) {
      return this.fetch(this.generateConfig('POST', url, payload, options,
        headers));
    };

    /**
     * Delete a record through the API
     * @param string endpoint what api service/value/context to call
     * @param object payload an object with the value, context and data to send
     *  to the API
     * @param object options $http angular configuration options
     * @param object headers the request headers to send
     * @return object $http promise that resolves the request payload
     **/
    this.delete = function (url, payload, options, headers) {
      return this.fetch(this.generateConfig('DELETE', url, payload,
        options, headers));
    };
  }
]);