'use strict';

angular.module('genisis').controller('editRequestCtrl', ['$scope', 'request',
  '$rootScope', 'source', 'type', 'studyApproval', 'api', 'user',
  '$routeParams',
  function ($scope, $requests, $rootScope, $sources, $types,
    $studyApprovals, $api, $user, $routeParams) {

    $scope.requests = $requests;
    $scope.request = {};
    $scope.studies = {};
    $scope.studyApprovals = $studyApprovals;
    $scope.sources = $sources;
    $scope.types = $types;
    $scope.editable = false;
    $scope.submitted = false;
    $scope.sent = false;
    $scope.accepted = false;
    $scope.logs = [];

    /**
     * Loads data for a specific request
     * @param int requestID the ID of the request to load
     * @return boolean
     **/
    $scope.loadRequest = function (requestID) {

      //load the request matching the given id
      $requests.load().then(function ( /*request*/ ) {
        $scope.request = $requests.find(requestID);

        switch ($scope.request.statusDescription) {
        case 'Draft':
        case 'Returned':
        case null:
          $scope.editable = true;
          break;
        case 'Submitted':
          if ($user.dataManager() || $user.admin()) {
            $scope.submitted = true;
          }
          break;
        case 'Sent':
          if ($user.dataManager() || $user.admin()) {
            $scope.sent = true;
          }
          break;
        case 'RequestAccepted':
          //if ($user.dataManager() || $user.admin()) {
          $scope.accepted = true;
          //}
          break;
        }

        if ($scope.request.reason !== undefined &&
          $scope.request.reason !== null) {
          $rootScope.warnings.push($scope.request.reason);
        }

        return true;
      });
    };

    $scope.loadLogs = function (requestID) {

      //load the logs for this request
      $requests.logs(requestID).then(function (logData) {
        $scope.logs = logData;
      });
    };

    //load the request matching the given id
    $scope.studies = $studyApprovals.load();

    /**
     * Save updates made to a request
     * @param Object request
     * @return boolean
     **/
    $scope.saveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id, {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('Your changes have been saved.');
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Save request API call failed.');
        }

        return false;
      });
    };

    /**
     * Submitting a request to the data manager
     * @param Object request data
     * @return boolean
     **/
    $scope.submitRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id + '/submit', {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {

          //this is no longer editable
          $scope.editable = false;
          req.statusDescription = 'Submitted';

          //set a message that the request was submitted
          $rootScope.messages.push(
            'Your request has been submitted to your data manager.'
          );

          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Submit request API call failed.');
        }

        return false;
      });
    };

    /**
     * Approve a request
     * @param Object request
     * @return boolean
     **/
    $scope.approveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/sent', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('The request has been approved.');
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = true;
          req.statusDescription = 'Sent';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Approve request API call failed.');
        }

        return false;
      });
    };

    /**
     * Save updates made to a request
     * @param string message why this request is being rejected
     * @param Object request the request that's being returned
     * @return boolean
     **/
    $scope.returnRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.message = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re returning this request.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/return', {
        data: {
          reason: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request has been returned to the owner.'
          );
          $scope.submitted = false;
          req.statusDescription = 'Returned';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Reject request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers and admins to deny submitted requests
     * @param string message why this request is being denied
     * @param Object request the request that's being denied
     * @return boolean
     **/
    $scope.denyRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if (!message) {
        $rootScope.errors.push(
          'Enter a reason why you\'re denying this request.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/deny', {
        data: {
          reason: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push('This request has been denied.');
          $scope.editable = false;
          $scope.submitted = false;
          req.statusDescription = 'Denied';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Deny request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to accept data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.acceptData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/acceptdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted.');
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = true;
          req.statusDescription = 'RequestAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow data managers to reject data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.denyData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/denydata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request has been denied because the ' +
            'request cannot be fulfilled.');

          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;

          req.statusDescription = 'RequestNotAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push('Deny data request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to accept data from the data supplier
     * @param Object request the request that's data is being confirmed for
     * @return boolean
     **/
    $scope.acceptResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/confirmdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been accepted by the researcher.'
          );
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;
          req.statusDescription = 'ResultsAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Accept results request API call failed.');
        }

        return false;
      });
    };

    /**
     * Allow researcher to reject the data from the data supplier
     * @param Object request the request that's data is being rejected for
     * @return boolean
     **/
    $scope.denyResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      return $api.update('requests/' + req.id + '/rejectdata', {
        data: {
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $rootScope.messages.push(
            'This request\'s data has been rejected by the researcher.'
          );

          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;

          req.statusDescription = 'ResultsNotAccepted';
          return true;
        } else if (request && request.message) {
          $rootScope.errors.push(request.message);
        } else {
          $rootScope.errors.push(
            'Deny results request API call failed.');
        }

        return false;
      });
    };

    $scope.loadRequest($routeParams.id);
    $scope.loadLogs($routeParams.id);
  }
]);