'use strict';

describe('requests/createRequest.js', function () {

  beforeEach(module('genisis'));

  var $controller, $scope, $user, $location, $rootScope, $sources,
    $studyApprovals, $requests, $types, $httpBackend, $q, $api;

  beforeEach(
    inject(function (_$rootScope_, _$controller_, user, _$location_,
      source, type, request, studyApproval, _$httpBackend_, _$q_, api) {
      $controller = _$controller_;
      $rootScope = _$rootScope_;
      $httpBackend = _$httpBackend_;
      $q = _$q_;
      $location = _$location_;
      $user = user;
      $sources = source;
      $requests = request;
      $studyApprovals = studyApproval;
      $types = type;
      $api = api;
      $scope = $rootScope.$new();

      $httpBackend.when('GET').respond('test GET call');
      $httpBackend.when('POST').respond('test POST call');

      $controller('createRequestCtrl', {
        $scope: $scope,
        $sources: $sources,
        $studyApprovals: $studyApprovals,
        $types: $types,
        $rootScope: $rootScope,
        $location: $location,
        $requests: $requests,
        $user: $user,
        $api: $api
      });
    })
  );

  it('sets $scope.associatedRequests to empty object', function () {
    expect($scope.associatedRequests).toEqual([]);
  });

  it('sets $scope.selectedRequest to null', function () {
    expect($scope.selectedRequest).toEqual(null);
  });

  it('loads $scope.studies', function () {

    var study1 = {
      id: 'study1'
    };
    var study2 = {
      id: 'study2'
    };

    spyOn($studyApprovals, 'load').and.callFake(function () {
      var deferred = $q.defer();
      deferred.resolve([study1, study2]);
      return deferred.promise;
    });

    $scope.studies = $studyApprovals.load();
    $scope.$apply();
    expect($studyApprovals.load.calls.count()).toEqual(1);
  });

  it('loads $scope.sources()', function () {
    expect($scope.sources.get().length).toEqual(3);
  });

  it('loads $scope.allRequests', function () {

    var request1 = {
      id: 'request1'
    };

    var request2 = {
      id: 'request2'
    };

    spyOn($requests, 'load').and.callFake(function () {
      var deferred = $q.defer();
      deferred.resolve([request1, request2]);
      return deferred.promise;
    });

    $scope.allRequests = $requests.load();
    $scope.$apply();
    expect($requests.load.calls.count()).toEqual(1);
  });

  describe('$scope.addAssociation() function', function () {

    it('exists', function () {
      expect($scope.addAssociation).toBeDefined();
    });

    it('adds an association', function () {
      var request = {};

      expect($scope.associatedRequests.length).toEqual(0);
      expect($scope.addAssociation(request)).toEqual(true);
      expect($scope.associatedRequests.length).toEqual(1);
    });

    it('doesn\'t allow duplicate associations', function () {
      var request = {
        id: 1
      };
      var request2 = {
        id: 2
      };

      expect($scope.associatedRequests.length).toEqual(0);

      expect($scope.addAssociation(request)).toEqual(true);
      expect($scope.associatedRequests.length).toEqual(1);

      expect($scope.addAssociation(request)).toEqual(false);
      expect($scope.associatedRequests.length).toEqual(1);

      expect($scope.addAssociation(request2)).toEqual(true);
      expect($scope.associatedRequests.length).toEqual(2);
    });
  });

  describe('$scope.removeAssociation() function', function () {

    it('exists', function () {
      expect($scope.removeAssociation).toBeDefined();
    });

    it(
      'removes an association if it exists in $scope.associatedRequests',
      function () {
        var request1 = {
          id: 1
        };
        var request2 = {
          id: 2
        };

        $scope.addAssociation(request1);
        $scope.addAssociation(request2);

        expect($scope.associatedRequests.length).toEqual(2);

        expect($scope.removeAssociation(request1)).toEqual(true);
        expect($scope.associatedRequests.length).toEqual(1);

        expect($scope.removeAssociation(request1)).toEqual(false);
        expect($scope.associatedRequests.length).toEqual(1);

        expect($scope.removeAssociation(request2)).toEqual(true);
        expect($scope.associatedRequests.length).toEqual(0);

        expect($scope.removeAssociation(request2)).toEqual(false);
      }
    );
  });

  describe('$scope.setRequest() function', function () {
    it('exists', function () {
      expect($scope.setRequest).toBeDefined();
    });

    it('sets $scope.selectedRequest', function () {
      var request = {};
      $scope.setRequest(request);
      expect($scope.selectedRequest).toEqual(request);
    });
  });

  describe('$scope.createRequest() function', function () {

    it('exists', function () {
      expect($scope.createRequest).toBeDefined();
    });

    it('clears out existing $rootScope.errors', function () {
      var someError = ['test1'];
      $rootScope.errors = someError;
      expect($rootScope.errors.length).toEqual(1);
      expect($scope.createRequest({})).toEqual(false);
      expect($rootScope.errors.indexOf(someError)).toEqual(-1);
    });

    it('adds associatedRequests', function () {
      var request = {
        associatedRequests: []
      };
      var testAssociation = {
        id: 'test'
      };

      expect($scope.associatedRequests.length).toEqual(0);

      $scope.associatedRequests = [testAssociation];

      expect($scope.createRequest(request)).toEqual(false);
      expect(request.associatedRequests).toEqual([testAssociation.id]);
    });

    it('returns $rootScope.errors if there is no request title',
      function () {
        var invalidRequest = {};
        expect($scope.createRequest(invalidRequest)).toEqual(false);
        expect($rootScope.errors).toEqual([
          'You must enter a request title.'
        ]);
      });

    it('creates a request', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      var fakeResponse = {
        id: 1
      };

      spyOn($location, 'path').and.returnValue('/');

      spyOn($api, 'create').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: fakeResponse
          });
          return deferred.promise;
        }
      );

      var promise = $scope.createRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($location.path).toHaveBeenCalled();
      });

      $scope.$apply();
    });

    it('returns an error if the request fails', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      var fakeResponse = {
        id: 1
      };

      spyOn($location, 'path').and.returnValue('/');

      spyOn($api, 'create').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: false,
            message: 'api error',
            response: fakeResponse
          });
          return deferred.promise;
        }
      );

      var promise = $scope.createRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual(['api error']);
      });

      $scope.$apply();
    });

    it('returns an error if the request fails', function () {

      var fakeRequest = {
        title: 'test title',
        source: 1,
        studyApproval: 2,
        type: 3,
        notify: true,
        description: 'test description'
      };

      spyOn($location, 'path').and.returnValue('/');

      spyOn($api, 'create').and.callFake(
        function () {
          var deferred = $q.defer();
          deferred.resolve({});
          return deferred.promise;
        }
      );

      var promise = $scope.createRequest(fakeRequest);

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual([
          'Create request API call failed.'
        ]);
      });

      $scope.$apply();
    });
  });
});