# return true if the given form has any field with a
# 'ratingCalculator' attribute equal to the given symptom
def rating_calculator_field(form, symptom)
  form.fields.each do |name, field|
    if field.attrs["ratingCalculator"] == symptom
      return true
    elsif ["select", "radio"].include? field.type
      return true if field.options.any? { |o| o["ratingCalculator"] == symptom }
    end
  end

  false
end

RSpec::Matchers.define :have_a_symptom do |symptom|
  match do |exam|
    rating_calculator_field(exam, symptom)
  end

  failure_message do |exam|
    "expected that `#{symptom}` would be present in the `#{exam.title}` exam"
  end
end

def ratingCalculatorFields(exam)
  exam.fields.select{ |_, field| field.attrs.key? "ratingCalculator" }
end

RSpec::Matchers.define :only_contain_symptoms_from do |iepd_values|
  attr_reader :failed_symptom

  match do |exam|
    ratingCalculatorFields(exam).each do |_, field|
      symptom = field.attrs['ratingCalculator']
      @failed_symptom = symptom unless iepd_values.include? symptom
      return false if @failed_symptom
    end
  end

  failure_message do |exam|
    "Unexpected symptom #{@failed_symptom} in exam #{exam.title}"
  end
end

# Custom matcher that takes a list of symbolized keys, and a hash with symbolized keys
# and ActiveRecord values.
# The values of lookup_hash is expected to contain all elements referenced by
# keys_included, and only those values referenced by keys_included.
RSpec::Matchers.define :include_only_values_with_listed_keys do |keys_included, lookup_hash|
  # attributes used to construct failure message, if test fails
  attr_reader :should_include, :current_key

  match do |results|
    lookup_hash.each do |key, record_value|
      @current_key = key
      if keys_included.include? key
        @should_include = true
        return false unless results.include?(record_value)
      else
        @should_include = false
        return false unless results.exclude?(record_value)
      end
    end
    true
  end

  failure_message do |results|
    if @should_include
      "#{results} does not contains expected value corresponding to key '#{@current_key}'."
    else
      "#{results} contains value corresponding to key '#{@current_key}', which was supposed to be excluded."
    end
  end

end


# Custom matcher that takes an array of excluded values, and ensures that no value
# matches from the expected array.
RSpec::Matchers.define :include_no_values_from_list do |excluded_list|
  # attributes used to construct failure message, if test fails
  attr_reader :current_value

  match do |results|
    excluded_list.each do |value_to_exclude|
      @current_value = value_to_exclude
      return false unless results.exclude?(value_to_exclude)
    end
    true
  end

  failure_message do |results|
    "#{results} contains value '#{@current_value}' which was supposed to be excluded."
  end

end


# Custom matcher that takes an array-of-hashes as complete_list, cherry-picks then
# complete list for hashes where hash[field_name] is found in the list of field values,
# and ensures that all field values are contained in order within the cherry-picked list.
RSpec::Matchers.define :include_elements_with_field_values_in_order do |complete_list|

  match do |field_name, field_values|
    filtered_list = complete_list.select {|value| field_values.include? value[field_name] }
    field_values.each_with_index do |field_value, fv_index|
      return false unless filtered_list[field_name] == field_value
    end
    return true
  end

  failure_message do |results|
    "#{complete_list} was expected to contain all values #{results} in order."
  end

end
