require 'rails_helper'

require 'via_api.rb'

RSpec.describe 'VIA_API::FindPatientService', cpp: true do

  before do
    login_via_response = Rails.root.join("spec/support/via_api/login_via_response.xml").read
    select_response = Rails.root.join("spec/support/via_api/select_response.xml").read
    match_response = Rails.root.join("spec/support/via_api/match_response.xml").read
    url  = VIA_API::VIA_EMR_ENDPOINT_URL
    login_url = VIA_API::VIA_EMR_ENDPOINT_URL
    stub_request(:post, login_url).with { |request| request.body.include?("loginVIA") }.to_return(status: 200, body: login_via_response)
    stub_request(:post, url).with { |request| request.body.include?("match") }.to_return(status: 200, body: match_response)
    stub_request(:post, url).with { |request| request.body.include?("select") }.to_return(status: 200, body: select_response)
    params = { access_code: "access", verify_code: "verify", site_code: "site" }
    @vista_session = { vista: VIA_API.authenticate(params)}
  end

  describe 'methods', cpp: true do

    before :context do
      @veteran = VIA_API::Models::Veteran.new( {:first_name=>"FAITH",
                                               :middle_name=>"K",
                                               :last_name=>"AARSVOLD",
                                               :date_of_birth=>"07/16/1942",
                                               :address=>"712 S OAK",
                                               :city=>"KIMBALL",
                                               :county=>"65",
                                               :state=>"NEBRASKA",
                                               :zip=>"69145",
                                               :phone=>"(444)-444-4444",
                                               :ssn=>"101012294",
                                               :gender=>"F",
                                               :age=>"74",
                                               :marital_status=>"MARRIED",
                                               :patient_name=>"AARSVOLD,FAITH K",
                                               :mpi_pid=>"1009609054",
                                               :mpi_checksum=>"897943",
                                               :local_pid=>"7177262",
                                               :location=>{:id=>"80", :name=>"C SURGERY", :specialty=>{:text=>"4"}, :room=>"2112", :bed=>"1", :ask_for_check_in=>false},
                                               :cwad=>"AD",
                                               :restricted=>false,
                                               :admit_timestamp=>"20160609.165547",
                                               :service_connected=>false,
                                               :sc_percent=>"0",
                                               :inpatient=>false,
                                               :confidentiality=>{:tag=>"0"},
                                               :needs_means_test=>false,
                                               :patient_flags=>{:count=>"0"},
                                               :cmor_site_id=>"442",
                                               :active_insurance=>
                                                "Pt Insur^Patient has active Insurance^\nCompany ID:                   7168777.868\nCompany Name:                 BLUE CROSS/BS WY\nCompany Address:              PO BOX 2266\n                              CHEYENNE, WYOMING 82003-2266\nPhone Number:                 634-1393\nCoordination of Benefits:     PRIMARY insurer\nPolicy Name:                  6033 - KIESOW\nPolicy Reimbursable:          YES\nEffective Date:               Jan 01, 1995\nExpiration Date:              \nSubscriber Relationship to Patient:     SPOUSE\nSubscriber Name:                        AARSVOLD,ELI\nSubscriber ID:                          R026770010\nPharmacy Coverage:            NO\nOutpatient Coverage:          YES\nInpatient Coverage:           YES\nGroup Number:                 GRP NUM 6033\nPatient Relationship to Subscriber:     SPOUSE",
                                               :is_test_patient=>false,
                                               :has_insurance=>false,
                                               :patient_type=>"NON-VETERAN (OTHER)",
                                               :is_veteran=>false,
                                               :is_locally_assigned_mpi_pid=>false,
                                               :sites=>{:count=>"1", :sites=>{:site_to=>{:sitecode=>"516", :name=>"Bay Pines, FL"}}},
                                               :team=>{:attending_name=>"AUDELHUK,CECILIA C", :name=>"GREEN TEAM"}} )
    end

    let(:ssn) {"123456789"}

    it 'Performs the match and select operations' do
      match_veteran = VIA_API::EmrService.match(ssn, @vista_session[:vista]).first
      veteran = VIA_API::EmrService.select(match_veteran, @vista_session[:vista])
      expect(veteran).to eq @veteran
    end

    it "should return all portions of the address", cpp: true do
      expect(@veteran.address_attrs[:address]).to eq "712 S OAK"
      expect(@veteran.address_attrs[:city]).to eq "KIMBALL"
      expect(@veteran.address_attrs[:state]).to eq "NEBRASKA"
      expect(@veteran.address_attrs[:zip]).to eq "69145"
    end

    it "should return the full name", cpp: true do
      expect(@veteran.full_name).to eq "FAITH K AARSVOLD"
    end

    it "should return a formatted ssn", cpp: true do
      expect(@veteran.formatted_ssn).to eq "101-01-2294"
    end

    # FIXME:
    # it "should return the status of other health insurance", cpp: true do

    # end

    it "should return the first name,middle name & last name in a array", cpp:true do
      expect(VIA_API::Models::Veteran.split_name(@veteran[:patient_name])).to eq [@veteran.first_name, @veteran.middle_name, @veteran.last_name]
    end

    # FIXME: Update the test cases for filtered_veterans method
    # it "should return all veterans with blank filter", cpp: true do

    # end

    # it "should filter veterans by ssn", cpp: true do

    # end

    # it "should filter veterans by first name", cpp: true do

    # end

    # it "should filter veterans by last name", cpp: true do

    # end

    # it "should filter veterans by date", cpp: true do

    # end
  end
end
