require 'rails_helper'
require 'mvi.rb'
require 'mvi_helper'

RSpec.configure do |c|
  c.include MVIHelper
end

# the actual unit testing spec
RSpec.describe 'MVI Retrieve person', cui: true do

  describe 'can create 1305 retrieve person request', cui:true do

    xit 'creates a valid 1305 for PID=12937191' do
      pid = '12937191'
      request_id = "MCI-CUI-" + SecureRandom.hex(7)
      test_request = File.read(Rails.root.join("spec/support/mvi/retrieve_person_pid_12937191.xml"))

      request = MVI::MVI_CLIENT.build_request(:prpa_in201305_uv02, attributes: MVI::MSG_TAG_ATTRIBUTES, message: MVI.retrieve_person_request(pid, request_id))
      request_xml = MVI.add_default_header(request.body)
      req_xml = MVI.remove_namespaces(request_xml)

      updated_test_request = substitute_attributes req_xml, test_request
      expect(req_xml).to eql(updated_test_request)
    end

    xit 'creates a valid 1305 for PID=103027964' do
      pid = '103027964'
      request_id = "CUI" + SecureRandom.base64(10)
      test_request = File.read(Rails.root.join("spec/support/mvi/retrieve_person_pid_103027964.xml"))

      request = MVI::MVI_CLIENT.build_request(:prpa_in201305_uv02, attributes: MVI::MSG_TAG_ATTRIBUTES, message: MVI.retrieve_person_request(pid, request_id))
      request_xml = MVI.add_default_header(request.body)
      req_xml = MVI.remove_namespaces(request_xml)

      updated_test_request = substitute_attributes req_xml, test_request
      expect(req_xml).to eql(updated_test_request)
    end
  end


  describe 'can retrieve a person', cui: true do
    before do
      url = MVI::MVI_ENDPOINT_URL
      retrieve_person_success_response = Rails.root.join("spec/support/mvi/retrieve_person_success_response.xml").read
      stub_request(:post, url).to_return(status: 200, body: retrieve_person_success_response)
    end
    it 'with a valid pid' do
      request_id = "MCI-CUI-" + SecureRandom.hex(7)
      person_data = MVI.retrieve_person("32326950", request_id)
      
      person_firstname = person_data.dig(:patient, :patient_person, :name, :given)
      person_familyname = person_data.dig(:patient, :patient_person, :name, :family)
      
      expect(person_firstname).to eq("BOB")
      expect(person_familyname).to eq("ADELSON")
    end
  end

  describe 'will provie an error response', cui: true do  
    before do 
      url = MVI::MVI_ENDPOINT_URL
      retrieve_person_not_found_response = Rails.root.join("spec/support/mvi/retrieve_person_not_found_response.xml").read
      stub_request(:post, url).to_return(status: 200, body: retrieve_person_not_found_response)
    end
    it 'with an invalid pid' do
      request_id = "MCI-CUI-" + SecureRandom.hex(7)
      expect{ MVI.retrieve_person("123456", request_id) }.to raise_error(MVI::MviPersonNotFoundError)
    end
  end

  describe 'when system is down', cui:true do
    before do 
      url = MVI::MVI_ENDPOINT_URL
      retrieve_person_system_down_response = Rails.root.join("spec/support/mvi/system_downtime_response.xml").read
      stub_request(:post, url).to_return(status: 500, body: retrieve_person_system_down_response)
    end
    it 'provides an appropriate response' do
      request_id = "MCI-CUI-" + SecureRandom.hex(7)
      expect{ MVI.retrieve_person("123456", request_id) }.to raise_error(MVI::MviApiFaultError, "System is unavailable. Request cannot be completed at this time.")
    end
  end

  describe 'when database is down', cui:true do
    before do 
      url = MVI::MVI_ENDPOINT_URL
      retrieve_person_database_down_response = Rails.root.join("spec/support/mvi/database_down_response.xml").read
      stub_request(:post, url).to_return(status: 200, body: retrieve_person_database_down_response)
    end
    it 'provides an appropriate response' do
      request_id = "MCI-CUI-" + SecureRandom.hex(7)
      expect{ MVI.retrieve_person("123456", request_id) }.to raise_error(MVI::MviApplicationReject, "Database is down. Request cannot be completed at this time. Please try again later.")
    end
  end
end
