require 'rails_helper'
require 'mvi.rb'

require 'mvi_helper'

RSpec.configure do |c|
  c.include MVIHelper
end

# the actual unit testing spec
RSpec.describe 'MVI Proxy Add to Vista', cui: true do

  before do
    @veteran_hash = {
      :retrieve_person_request_id => "MCI-CUIA-mmEvkiI2GefL6w==",
      :patient_icn => "1008713016V776783",
      :station_number => "523",
      :fname => "BOB",
      :lname => "ADELSON",
      :telecom_value => "571-123-1234",
      :telecom_usage => "HP",
      :gender => "M",
      :birth_date => "19500101",
      :addr_usage => "HP",
      :addr => {
        :street => "123 Main Street",
        :city => "Charleston",
        :state => "SC",
        :country => "USA",
        :zipcode => "29492"
      },
      :ssn => "555550100",
      :clinic_name => "Good Health Clinic",
      :clinic_phone => "tel:+1-342-555-8394"
    }
  end

  describe 'can add a veteran' do
    before do
      url = MVI::MVI_ENDPOINT_URL
      proxy_add_success_response = Rails.root.join("spec/support/mvi/proxy_add_to_vista_response.xml").read
      stub_request(:post, url).to_return(status: 200, body: proxy_add_success_response)
    end

    it 'to a valid vista site' do

      ack_message = MVI.proxy_add_to_vista @veteran_hash
      
      expect(ack_message).to eq("Correlation Proxy Added to VistA")
    end
  end

  describe 'cant add a veteran' do
    before do 
      url = MVI::MVI_ENDPOINT_URL
      proxy_add_wrong_station_response = Rails.root.join("spec/support/mvi/proxy_add_incorrect_station_response.xml").read
      stub_request(:post, url).to_return(status: 200, body: proxy_add_wrong_station_response)
    end

    it 'to an invalid vista site' do
      
      expect{ MVI.proxy_add_to_vista @veteran_hash }.to raise_error("Patient could not be added to site. Communication Failure with requested VistA Station")
    end
  end

   describe 'generates correct 1301 request for PID=12937191' do
     let!(:site) { create(:orig_manchester_va) }

     before do
       url = MVI::MVI_ENDPOINT_URL
        response = File.read(Rails.root.join("spec/support/mvi/1306_pid_12937191.xml"))
        stub_request(:post, url).to_return(status: 200, body: response)

        request_id = "CUI/4o4lEHsfNRy0Q=="
        person_data = MVI.retrieve_person("12937191", request_id)
        prpr = MVI::ProcessRetrievePersonResponse.new(person_data, "Samantha Smith", site)
        @vh = prpr.create_veteran_hash(request_id, site.site_station_number)
      end

      xit 'will generate the 1301 request ' do
        request = MVI::MVI_CLIENT.build_request(:prpa_in201301_uv02, attributes: MVI::MSG_TAG_ATTRIBUTES, message: MVI.proxy_add_request(@vh))
        request_xml = MVI.add_default_header(request.body)
        req_xml = MVI.remove_namespaces(request_xml)
        doc = Nokogiri::XML(req_xml)
        new_creation_time = get_attribute_value doc, "creationTime", "value"

        test_request = File.read(Rails.root.join("spec/support/mvi/1301_pid_12937191.xml"))
        test_doc = Nokogiri::XML(test_request)
        test_doc.at_xpath("//href:creationTime", "href" => "urn:hl7-org:v3")["value"] = new_creation_time
        updated_test_req = test_doc.to_s

        expect(req_xml).to eql(updated_test_req)
      end

   end

  describe 'generates correct 1301 request for PID=103027964' do
     let!(:site) { create(:manchester_va) }

     before do
        url = MVI::MVI_ENDPOINT_URL
        response = File.read(Rails.root.join("spec/support/mvi/1306_pid_103027964.xml"))
        stub_request(:post, url).to_return(status: 200, body: response)

        request_id = "CUI2m8WiPTdjMFMaQ=="
        person_data = MVI.retrieve_person("103027964", request_id)
        prpr = MVI::ProcessRetrievePersonResponse.new(person_data, "Samantha Smith", site)
        @vh = prpr.create_veteran_hash(request_id, site.site_station_number)
      end

      xit 'will generate the correct 1301 request ' do
        request = MVI::MVI_CLIENT.build_request(:prpa_in201301_uv02, attributes: MVI::MSG_TAG_ATTRIBUTES, message: MVI.proxy_add_request(@vh))
        request_xml = MVI.add_default_header(request.body)
        req_xml = MVI.remove_namespaces(request_xml)
        doc = Nokogiri::XML(req_xml)
        new_creation_time = get_attribute_value doc, "creationTime", "value"

        test_request = File.read(Rails.root.join("spec/support/mvi/1301_pid_103027964.xml"))
        test_doc = Nokogiri::XML(test_request)
        test_doc.at_xpath("//href:creationTime", "href" => "urn:hl7-org:v3")["value"] = new_creation_time
        updated_test_req = test_doc.to_s

        expect(req_xml).to eql(updated_test_req)
      end
   end

end
