#!/bin/env ruby
# encoding: utf-8
require 'rails_helper'

RSpec.describe "Contentions", type: :request do
  before do
    @user = create(:examiner)
    @user_2 = create(:user, first_name: "Ms", last_name: "Green", email: "doctor2@va.gov")
    login_as @user
    @evaluation_spec = create(:evaluation_spec)
    allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
    @spec_1 = create(:short_spec01)
    @spec_2 = create(:short_spec02)
    @spec_3 = create(:short_spec03)
  end

  describe "GET /claims/:claim_id/contentions/new" do
    before do
      @claim = create :claim
      @claim.contentions << create(:contention)
    end

    context "when the user is not a medical assistant" do
      it "should not show the user a link to create a new contention or let them visit the new contention page"
    end

    context "when the user is a medical assistant" do
      before do
        @user.roles = ["medical_assistant"]
      end

      it "should create the contention and allow the user to view it"

      it "when no name is provided it shows an error message"

      context "when the claim has one or more evaluations" do
        before do
          @user2 = create("examiner")
        end

         it "should allow an owner medical assistant to delete the evaluation"

         it "should allow a non-owner medical assistant to delete the evaluation"
      end
    end
  end

  describe "GET /claims/:claim_id/contentions/:id" do
    context "when a claim is unresolved and has resolved and unresolved contentions" do
      before do
        @claim = create :claim
        @claim.contentions << create(:contention)
      end

      context "when a user makes a change" do
        before do
          visit claim_contention_path(@claim, @claim.contentions.first)
        end

        xit "changes the evaluation last update time", js: true do
          last_update = find("#contention-updated").text
          fill_in "Describe the history of the Veteran’s condition", with: "History goes here"
          choose 'contention_claim_folder_reviewed_false'
          sleep 62
          expect(find("#contention-updated")).to_not have_content last_update
        end
      end

      context "when the contention has at least one evaluation" do
        before do
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @evaluation_spec, claim: @claim, user: @user)
        end

        context "when viewing the unresolved contention" do
          before do
            visit claim_contention_path(@claim, @claim.contentions.first)
          end

          it "should show the name and contention information, and link to the evaluation spec"

          context "when the evaluation has not yet been completed" do
            it "should link to an editable evaluation" do
              expect(page).to have_link "Simple Yes/No Evaluation", href: edit_claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
              expect(page).to have_content "Draft"
            end
          end

          context "when the evaluation has been completed" do
            before do
              @claim.contentions.first.evaluations.each{|evaluation| evaluation.complete! @user }
              visit claim_contention_path(@claim, @claim.contentions.first)
            end

            it "should link to a non-editable page, show status as Completed and provide a link to a PDF and XML version" do
              expect(page).to have_link 'Simple Yes/No Evaluation', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first)
              expect(page).to have_content "Completed"
              expect(page).to have_link 'PDF', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first, format: :pdf)
              expect(page).to have_link 'XML', href: claim_evaluation_path(@claim, @claim.contentions.first.evaluations.first, format: :xml)
              expect(page).to have_link 'Spec', href: admin_evaluation_spec_path(@evaluation_spec)
            end
          end

          context "when the contention shows Veterans VA claim folder reviewed" do
            before do
              @claim.contentions.first.update_attributes(claim_folder_reviewed: true)
            end

            it "should not display the records table" do
              visit claim_contention_path(@claim, @claim.contentions.first)
              expect(page).to have_selector('#records_table', visible: false)
            end

            context "when the Veterans VA claim folder option is changed and updated" do
              it "should save the change and then display accordingly"
            end
          end

          context "when a contention shows Veterans VA claim folder not reviewed" do
            before do
              @claim.contentions.first.update_attributes(claim_folder_reviewed: false)
            end

            it "should  display the records table" do
              visit claim_contention_path(@claim, @claim.contentions.first)
              expect(page).to have_selector('#records_table', visible: true)
            end

            context "when the Veterans VA claim folder option is changed and updated" do
              it "should save the change and then display accordingly"
            end
          end
        end

        context "when viewing a contention that is ready to be resolved" do
          context "when the contention indicates no diagnosis" do
            before do
              @claim.contentions.first.update_attributes(no_diagnosis: true)
              visit claim_contention_path(@claim, @claim.contentions.first)
            end

            it "should show the details of the contention and prompt for resolution"
          end

          context "when the contention has all evaluations marked as completed" do
            before do
              @claim.contentions.first.evaluations.each{|evaluation| evaluation.complete! @user }
              visit claim_contention_path(@claim, @claim.contentions.first)
            end

            it "should show all the details of the contention and prompt for resolution"
          end
        end

        context "when a logged in user views a contention that they resolved and is ready for review" do
          before do
            @claim.contentions.first.update_attributes(no_diagnosis: true, history: "Patient History",
                                                       claim_folder_reviewed: true)
            @claim.contentions.first.resolve!(@user)
            visit claim_contention_path(@claim, @claim.contentions.first)
          end

          it "should show that the contention was sent to an administrator" do
            expect(page).to have_content "Contention complete"
            expect(page).to have_no_content "Send to an Administrator for review"
            expect(page).to have_content "Sent to Administrator for review"
            expect(page).to have_no_link "Add evaluation"
          end

          it "should not display a submit button " do
            expect(page).to have_no_selector(".button primary")
          end
        end

        context "when a supervisor views a resolved contention" do
          let(:no_diagnosis) { false }
          let(:claim_folder_reviewed) { false }
          before do
            @claim.contentions.first.update(
              no_diagnosis: no_diagnosis,
              claim_folder_reviewed: claim_folder_reviewed,
              history: "some history here"
            )

            @claim.contentions.first.resolve!(@user)

            @supervisor = create(:supervisor)
            login_as @supervisor
            visit claim_contention_path(@claim, @claim.contentions.first)
          end

          it "should show diagnosis and records text" do
            expect(page).to have_content("Did you review the Veteran’s VA claim folder?")
            expect(page).to have_content("no")
          end

          it "should not display a submit button " do
            expect(page).to_not have_button(".primary")
          end

          context "when the veteran has the diagnosis" do
            it "doesn't show unchecked record fields" do
              expect(page).to have_content("The Veteran does not have a current diagnosis")
            end
          end

          context "when the veteran does not have the diagnosis" do
            let(:no_diagnosis) { true }
            let(:claim_folder_reviewed) { true }
            it "shows all unchecked record fields" do
              expect(find("#contention_no_diagnosis")).to be_checked
            end
          end

          context "when the claim folder has been reviewed" do
            let(:claim_folder_reviewed) { true }
            it "doesn't show unchecked record fields" do
              Contention::RECORD_FIELD_LABEL_MAP.values.each {|fieldname|
                expect(page).not_to have_content(fieldname)
              }
            end
          end

          context "when the claim folder has not been reviewed" do
            it "shows all unchecked record fields" do
              Contention::RECORD_FIELD_LABEL_MAP.values.each {|fieldname|
                expect(page).to have_content(fieldname)
              }
            end
          end

          context "when there is another contention for this claim" do
            before do
              @claim.contentions << create(:contention)
              @claim.contentions.last.update_attributes(name: 'Contention 2', history: "Patient History")
              visit claim_contention_path(@claim, @claim.contentions.first)
            end

#            it "should should allow the user to select the other contention from a select drop down", js: true do
#              expect(page).to have_select 'contention_name', with_options: [@claim.contentions.first.name, @claim.contentions.last.name]
#              select("Contention 2", from: "contention_name")
#              expect(page).to have_content "Contention: Contention 2"
#              expect(page).to have_select 'contention_name', with_options: [@claim.contentions.first.name]
#            end
          end
        end

        context "when viewing a contention that has been reviewed" do
          before do
            @claim.contentions.first.update(
              no_diagnosis: false,
              claim_folder_reviewed: true,
              history: "some history here"
            )
            @claim.contentions.first.resolve!(@user)
          end

          context "when there's no user that reviewed the contention" do
            before do
              @claim.contentions.first.review!
            end

            it "should show the contention" do
              visit claim_contention_path(@claim, @claim.contentions.first)
            end
          end

          context "when there's a user that reviewed the contention" do
            before do
              @supervisor = create(:supervisor)
              @claim.contentions.first.review!(@supervisor.id)
            end

            it "should show the contention and who it was reviewed by" do
              visit claim_contention_path(@claim, @claim.contentions.first)
              expect(page).to have_content "Approved by: #{@supervisor.last_name}"
            end
          end
        end
      end

      context "when the contention has at least two evaluations" do
        before do
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_1, claim: @claim, user: @user)
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_3, claim: @claim, user: @user)
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_2, claim: @claim, user: @user)
        end

        it "should list the evaluations in ascending order by creation date/time" do
          visit claim_contention_path(@claim, @claim.contentions.first)
          expect(page.text).to match(/Spec01.*Spec03.*Spec02/)
          visit claim_path(@claim)
          expect(page.text).to match(/Spec01.*Spec03.*Spec02/)
        end
      end

      context "when the contention has mutliple evaluations assigned to different users" do
        before do
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_1, claim: @claim, user: @user)
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_3, claim: @claim, user: @user)
          @claim.contentions.first.evaluations.last.complete! @user
          @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @spec_2, claim: @claim, user: @user_2)
        end

        it "should only allow the examiner user that owns the evaluation to delete uncompleted evaluations"
      end
    end
  end

  describe "PUT /claims/:claim_id/contentions/:id" do
    before do
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "Patient History")
    end

    it "should update the contention's attributes"

    context "when a contention does not have any evaluations associated with it" do
      before do
        @user_2.roles = ["examiner"]
        @user_2.save
      end

#      it "should allow an examiner to add an evaluation to a contention", js: true do
#        visit claim_contention_path(@claim, @claim.contentions.first)
#        expect(page).to have_content "Evaluations"
#        click_link 'Evaluations'
#        click_link 'Add evaluation'
#        select('Green, Ms', from: 'evaluation_user_id')
#        find('a span', text: '--Select a new evaluation').click
#        find('li', text: 'Simple Yes/No Evaluation').click
#        click_button 'Add evaluation'
#        expect(page).to have_content "Added evaluation to contention"
#        click_link 'Evaluations'
#        expect(page).to have_content "Simple Yes/No Evaluation"
#        page.assert_selector('span', visible: false, text: "Assigner: #{@user.id}")
#
#      end

      it "should not allow an evaluation to be added without an evaluation specification"

 #     it "should allow searching of evaluations by keywords", js: true do
#        visit new_claim_evaluation_path(@claim, contention_id: @claim.contentions.first.id)
#        find('a span', text: '--Select a new evaluation').click
#        expect(page).to have_selector('.active-result', count: 5)
#        fill_in 'chosen-search', :with => 'testing'
#        expect(page).to have_selector('.active-result', count: 1)
#      end

      it "should not blow up if the user tries to link a blank evaluation"
    end

    context "when a contention has some evaluations, as do other contentions for the same claim" do
      before do
        @claim.contentions << create(:contention, name: "My other arm hurts")
        @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @evaluation_spec, claim: @claim, user: @user)
      end

      it "should allow an examiner to link an existing evaluation for that claim to a contention"
    end
  end

  describe "POST /claims/:claim_id/contentions/:contention_id/resolve" do
    before do
      @claim = create :claim
      @claim.contentions << create(:contention, history: "Patient History",
                                   no_diagnosis: true, claim_folder_reviewed: true)
    end

    context "when a contention's claim folder is not reviewed but at least one document has been reviewed" do
#      it "successfully resolves the contention", js: true do
#        contention = @claim.contentions.first
#        visit claim_contention_path(@claim, @claim.contentions.first)
#        click_link "Records"
#        choose "contention_claim_folder_reviewed_false"
#        check "contention_reviewed_military_service_personnel_records"
#        click_button 'contention-submit'
#        click_link "Send to Administrator for review"
#        expect(page).to have_content "Contention has been resolved and sent to administrator for review."
#        expect(page).to have_content "Resolved"
#      end
    end

    context "when a contention's claim folder is not reviewed and no documents have been reviewed" do
      it "displays an error"
    end

    context "when a contention has a blank history" do
      it "displays an error" do
        contention = @claim.contentions.first
        contention.update_attributes(history: nil)
        visit claim_contention_path(@claim, @claim.contentions.first)
        click_link "Send"
        expect(page).to have_content "Contention history cannot be blank."
      end
    end

    context "when a contention has not picked whether a claim folder has been reviewed" do
      it "displays an error" do
        contention = @claim.contentions.first
        contention.update_attributes(claim_folder_reviewed: nil)
        visit claim_contention_path(@claim, @claim.contentions.first)
        click_link "Send"
        expect(page).to have_content "A contention claim folder reviewed option must be picked."
      end
    end

    context "when a contention has a blank history and no claim folder reviewed option picked" do
      it "displays an error" do
        contention = @claim.contentions.first
        contention.update_attributes(claim_folder_reviewed: nil, history: nil)
        visit claim_contention_path(@claim, @claim.contentions.first)
        click_link "Send"
        expect(page).to have_content "Contention history cannot be blank. A contention claim folder reviewed option must be picked."
      end
    end

    context "when a contention has a no diagnosis indicator, even if all evaluations are not completed" do
      before do
        @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @evaluation_spec, claim: @claim, user: @user)
        @claim.contentions.first.update_attributes(no_diagnosis: true, history: "Patient History", claim_folder_reviewed: true)
      end

      it "should enable the examiner to resolve the contention"
    end

    context "when a contention shows Veterans VA claim folder reviewed" do
      before do
        @claim.contentions.first.update_attributes(claim_folder_reviewed: true, history: "Patient History")
      end

      it "should not display the records table" do
        visit claim_contention_path(@claim, @claim.contentions.first)
        expect(page).to have_selector('#records_table', visible: false)
      end
    end

    context "when a contention shows Veterans VA claim folder not reviewed" do
      before do
        @claim.contentions.first.update_attributes(claim_folder_reviewed: false)
      end

      it "should not display the records table" do
        visit claim_contention_path(@claim, @claim.contentions.first)
        expect(page).to have_selector('#records_table', visible: true)
      end
    end

    context "when a contention has multiple evaluations" do
      before do
        @phase_1_spec = @evaluation_spec
        @phase_2_spec = create(:diabetes_spec)
        @phase_1_evaluation = Evaluation.new(claim: @claim, evaluation_spec: @phase_1_spec, user: @user)
        @phase_2_evaluation = Evaluation.new(claim: @claim, evaluation_spec: @phase_2_spec, user: @user)
        @claim.contentions.first.evaluations << @phase_1_evaluation
        @claim.contentions.first.evaluations << @phase_2_evaluation
      end

      context "when a contention has all evaluations completed" do
        before do
          @claim.contentions.first.evaluations.each{|evaluation| evaluation.complete! @user }
        end

        it "should enable the examiner to resolve the contention"

        context "when the user completing the contention is under review" do
          before do
            @user.update_attributes(is_under_review: true)
          end

          it "should not send the evaluation to the backend"
        end

        context "for a contention with a phase 1 and phase 2 evaluation do" do
          context "when Phase 2 is not enabled" do
            before do
              ENV['ENABLE_PHASE_2'] = "false"
            end

            after do
              ENV.delete "ENABLE_PHASE_2"
            end

            it "should resolve the contention and send all exams to VBMS"
          end

          context "when Phase 2 is enabled" do
            before do
              ENV['ENABLE_PHASE_2'] = "true"
              ENV["PHASE_2_BODY_SYSTEMS"] = '["Endocrine"]'
            end

            after do
              ENV.delete "ENABLE_PHASE_2"
              ENV.delete "PHASE_2_BODY_SYSTEMS"
            end

            it "should resolve the contention and send all evaluations to VBMS and VLER-DAS dependeing on body system"
          end
        end
      end
    end
  end

  describe "PUT /claims/:claim_id/contentions/:contention_id/accept" do
    context "when the logged in user is a supervisor" do
      before do
        @supervisor = create(:supervisor)
        login_as @supervisor
      end

      context "when a contention has been resolved" do
        before do
          @claim = create(:claim)
          @contention = create(:contention, history: "Patient History", claim_folder_reviewed: true)
          @claim.contentions << @contention
          @evaluation_spec = create(:diabetes_spec)
          @evaluation = Evaluation.new(claim: @claim, evaluation_spec: @evaluation_spec, user: @user)
          @claim.contentions.first.evaluations << @evaluation
          @evaluation.complete! @user
          @contention.resolve!(@user)
        end

        it "displays the reviewed at date"

        it "should allow the user to accept the contention, which send the contention's evaluations to the backend"
      end
    end
  end

  describe "PUT /claims/:claim_id/contentions/:contention_id/reject" do
    context "when the logged in user is a supervisor" do
      before do
        @supervisor = create(:supervisor)
        login_as @supervisor
      end

      context "when a contention has been resolved" do
        before do
          @claim = create(:claim)
          @contention = create(:contention, history: "Patient History", claim_folder_reviewed: true)
          @claim.contentions << @contention
          @evaluation_spec = create(:diabetes_spec)
          @evaluation = Evaluation.new(
            claim: @claim,
            evaluation_spec: @evaluation_spec,
            doc: build(:diabetes_doc).doc,
            user: @user)
          @claim.contentions.first.evaluations << @evaluation
          @evaluation.complete! @supervisor
          @contention.resolve!(@supervisor)
        end

        it "should allow the user to accept the contention, which send the contention's evaluations to the backend" do
          visit claim_contention_path(@claim, @contention)
          expect(page).to have_link 'Reject', href: new_claim_contention_rejection_path(@claim, @contention)
          click_link 'Reject'
          expect(page.current_path).to eq new_claim_contention_rejection_path(@claim, @contention)
        end
      end
    end
  end
end
