require 'rails_helper'

RSpec.shared_examples "permitted resolver" do |action_allowed|
  before do
    allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
  end

  prefix = action_allowed ? 'should' : 'should not'

  context "when a contention has a no diagnosis indicator, even if all evaluations are not completed" do
    before do
      @claim.contentions.first.evaluations << Evaluation.new(evaluation_spec: @evaluation_spec, claim: @claim, user: user)
      @claim.contentions.first.update_attributes(no_diagnosis: true)
    end

    it "#{prefix} enable the user to resolve the contention" do
      visit claim_contention_path(@claim, @claim.contentions.first)
      if action_allowed
        expect(VBMSJob).to receive(:perform_later).once
        expect(page).to have_content "Contention complete"
        expect(page).to have_no_content "Sent to Administrator for review"
        expect(page).to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
        click_link "Send to Administrator for review"
        expect(page).to have_content "Contention has been resolved and sent to administrator for review."
        expect(page).to have_content "Resolved"
      else
        expect(page).not_to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
      end
    end
  end

  context "when a contention has multiple evaluations" do
    before do
      @phase_1_spec = @evaluation_spec
      @phase_2_spec = create(:diabetes_spec)
      @phase_1_evaluation = Evaluation.new(claim: @claim, evaluation_spec: @phase_1_spec, user: user)
      @phase_2_evaluation = Evaluation.new(claim: @claim, evaluation_spec: @phase_2_spec, user: user)
      @claim.contentions.first.evaluations = [@phase_1_evaluation,  @phase_2_evaluation]
    end

    context "when a contention has all evaluations completed" do
      before do
        @claim.contentions.first.evaluations.each{|evaluation| evaluation.complete! user }
      end

      it "#{prefix} enable the examiner to resolve the contention" do

        visit claim_contention_path(@claim, @claim.contentions.first)
        if action_allowed
          expect(VBMSJob).to receive(:perform_later).twice
          expect(page).to have_content "Contention complete"
          expect(page).to have_no_content "Sent to Administrator for review"
          expect(page).to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
          click_link "Send to Administrator for review"
          expect(page).to have_content "Contention has been resolved and sent to administrator for review."
          expect(page).to have_content "Resolved"
        else
          expect(page).not_to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
        end
      end

      context "when the user completing the contention is under review" do
        before do
          user.update_attributes(is_under_review: true)
        end

        it "#{prefix} not send the evaluation to the backend" do
          visit claim_contention_path(@claim, @claim.contentions.first)
          if action_allowed
            expect(VBMSJob).not_to receive(:perform_later)
            click_link "Send to Administrator for review"
            expect(page).to have_content "Contention has been resolved and sent to administrator for review."
            expect(page).to have_content "Resolved"
          else
            expect(page).not_to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
          end
        end
      end

      context "for a contention with a phase 1 and phase 2 evaluation do" do
        context "when Phase 2 is not enabled" do
          before do
            ENV['ENABLE_PHASE_2'] = "false"
          end

          after do
            ENV.delete "ENABLE_PHASE_2"
          end

          it "#{prefix} resolve the contention and send all exams to VBMS" do
            visit claim_contention_path(@claim, @claim.contentions.first)
            if action_allowed
              expect(VBMSJob).to receive(:perform_later).twice
              click_link "Send to Administrator for review"
              expect(page).to have_content "Contention has been resolved and sent to administrator for review."
              expect(page).to have_content "Resolved"
            else
              expect(page).not_to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
            end
          end
        end

        context "when Phase 2 is enabled" do
          before do
            ENV['ENABLE_PHASE_2'] = "true"
            ENV["PHASE_2_BODY_SYSTEMS"] = '["Endocrine"]'
          end

          after do
            ENV.delete "ENABLE_PHASE_2"
            ENV.delete "PHASE_2_BODY_SYSTEMS"
          end

          it "#{prefix} resolve the contention and send all evaluations to VBMS and VLER-DAS dependeing on body system" do
            visit claim_contention_path(@claim, @claim.contentions.first)
            if action_allowed
              expect(VlerDASJob).to receive(:perform_later).with(@phase_2_evaluation)
              expect(VBMSJob).to receive(:perform_later).with(@phase_1_evaluation)
              click_link "Send to Administrator for review"
              expect(page).to have_content "Contention has been resolved and sent to administrator for review."
              expect(page).to have_content "Resolved"
            else
              expect(page).not_to have_link "Send to Administrator for review", href: resolve_claim_contention_path(@claim, @claim.contentions.first)
            end
          end
        end
      end
    end
  end
end


RSpec.describe "Contentions", :type => :request do
  before do
    @evaluation_spec = create(:evaluation_spec)
    ENV['ENABLE_PHASE_2'] = nil
    allow_any_instance_of(VlerDASJob).to receive(:perform) { "fake" }
    allow_any_instance_of(VBMSJob).to receive(:perform) { "fake" }
  end

  after do
    ENV.delete("ENABLE_PHASE_2")
  end

  describe "POST /claims/:claim_id/contentions/:contention_id/resolve" do
    let(:contention) { create(:contention, history: "this is a history", claim_folder_reviewed: true) }

    before do
      @claim = create :claim
      @claim.contentions << contention
      login_as user
    end

    context "when the logged in user is a supervisor"

    context "when the logged in user is a medical assistant" do
      let(:user) { create(:medical_assistant) }
      it_behaves_like 'permitted resolver', false
    end

    context "when the logged in user is an examiner"
  end
end
