require 'rails_helper'

RSpec.describe MusculoskeletalThoracolumbarSpineEvaluationPresenter do
  before do
    schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    @schema = XML::Schema.document(schema_xml)
    @claim = create(:claim, date_of_birth: Date.new(1900))
    @contention = create(:contention)
    @claim.contentions << @contention
    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }
    @spec_json = File.read("spec/evaluations/back-thoracolumbar-spine-evaluation.json")

    @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/back-thoracolumbar-spine-evaluation.yml"))
  end

  context "for an exam without arthritis or IVDS" do
    before do
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
      @evaluation.doc = @spec_json
      @evaluation_xml = @evaluation.to_xml
      @doc = XML::Document.string(@evaluation_xml)
    end
    
    # it "should generate valid XML for a back spine exam" do
    #   @doc.validate_schema(@schema)
    # end
  
    it "should output joint evidence and other diagnoses" do
      expect(@doc.find("//jf:jointsObjectiveEvidence/jf:value").first.content).to eq "Fatigue"
    end
    
    it "should output the diagnoses selected" do
      expect(@doc.find("//spf:spineDiagnoses/spf:value").first.content).to eq "Segmental instability (5239)"
    end
    
    it "should not show IVDS if hasIVDS is false" do
      expect(@evaluation_xml).not_to match "spineIVDS"
    end
    
    it "should show the arthritis diagnoses" do
      expect(@evaluation_xml).not_to match "Arthritis, gonorrheal"
    end
  end
  
  context "for an exam with arthiritis or IVDS" do
    before do
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
      @evaluation.doc = File.read("spec/evaluations/back-thoracolumbar-spine-with-ivds-and-arthritis.json")
      @evaluation_xml = @evaluation.to_xml
      @doc = XML::Document.string(@evaluation_xml)
    end
    
    # it "should validate" do
    #   @doc.validate_schema(@schema)
    # end
    
    it "should have IVDS and arthritis information" do
      expect(@doc.find("//jf:jointsObjectiveEvidence/jf:value[text() = 'Painful motion']").size).to eq 1
      expect(@doc.find("//jf:jointsOtherDiagnoses/jf:value").first.content).to eq "Arthritis, gonorrheal"
      expect(@doc.find("//spf:spineIVDS/spf:value").first.content).to eq "With incapacitating episodes having a total duration of at least six weeks during the past 12 months"
    end
  end

  context "when the eval is secondary, we output its parent as well" do
    before do
      user = create(:user)
      claim = create(:claim)
      claim.contentions << create(:contention, history: "This or That")

      dependent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, user: user)
      dependent_eval.doc = @spec_json
      claim.contentions.first.evaluations << dependent_eval

      parent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, completed_at: Time.now, user: user)
      parent_eval.doc = @spec_json
      claim.contentions.first.evaluations << parent_eval

      dependent_eval.primary_evaluation_id = parent_eval.id
      dependent_eval.save

      @evaluation_xml = dependent_eval.to_xml
    end

    # it "should include the parent eval" do
    #   xml = XML::Document.string(@evaluation_xml)
    #   facts = xml.find("//spf:spineFacts")
    #
    #   expect(facts.length).to eq 2
    #
    #   schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    #   schema = XML::Schema.document(schema_xml)
    #   xml.validate_schema(schema)
    # end
  end
end
