# == Schema Information
#
# Table name: veterans
#
#  id                          :integer          not null, primary key
#  ssn                         :string(10)       not null
#  first_name                  :string
#  middle_name                 :string
#  last_name                   :string
#  other_health_insurance_name :string
#  content                     :json
#  created_at                  :datetime         not null
#  updated_at                  :datetime         not null
#  other_health_insurance_id   :integer
#

require 'rails_helper'

RSpec.describe Cpp::Local::Veteran, :type => :model, cpp: true do
  context "associations" do
    it "should belong to a other_health_insurance" do
      veteran = Cpp::Local::Veteran.reflect_on_association(:other_health_insurance)
      expect(veteran.macro).to eq :belongs_to
    end

    # RWP TEMP: is the has_many side of the (1 veteran -> (n) consultations) association
    # important from a business perspective, or do we only care about looking
    # up veterans froma  consultation?

    # it "should have many consultations" do
    #   veteran = Cpp::Local::Veteran.reflect_on_association(:consultations)
    #   expect(veteran.macro).to eq :has_many
    # end
    #
    # it "should have many referrals through consultations" do
    #   veteran = Cpp::Local::Veteran.reflect_on_association(:referrals)
    #   expect(veteran.macro).to eq :has_many
    # end
  end

  context "methods" do
    before :context do
      address = ["1235 Any Street Blvd.", "P.O. Box 1235"]
      city = "Columbia"
      state = "MD"
      zip = "21046"
      content = {address: address, city: city, state: state, zip: zip}

      first_name = "John"
      middle_name = "Mark"
      last_name = "Doe"

      ssn = "123454321"

      created_at = "2017-01-01 00:00:00"

      @veteran = create(:veteran, first_name: first_name, middle_name: middle_name, last_name: last_name, ssn: ssn, content: content, created_at: created_at)
      @veteran2 = create(:veteran, first_name: "James", last_name: "Smith", ssn: "100101234", created_at: "2017-02-01 00:00:00")
    end

    it "should return all portions of the address" do
      expect(@veteran.address_attrs[:address][0]).to eq "1235 Any Street Blvd."
      expect(@veteran.address_attrs[:address][1]).to eq "P.O. Box 1235"
      expect(@veteran.address_attrs[:city]).to eq "Columbia"
      expect(@veteran.address_attrs[:state]).to eq "MD"
      expect(@veteran.address_attrs[:zip]).to eq "21046"
    end

    it "should return the full name" do
      expect(@veteran.full_name).to eq "John Mark Doe"
    end

    it "should return a formatted ssn" do
      expect(@veteran.formatted_ssn).to eq "123-45-4321"
    end

    it "should return a masked ssn" do
      expect(@veteran.hidden_formatted_ssn).to eq "XXX-XX-4321"
    end

    it "should return true if there is other health insurance" do
      @veteran.other_health_insurance_id = OtherHealthInsurance.where(title: 'Yes').first.id
      expect(@veteran.has_other_health_insurance?).to eq true
    end

    it "should return false if there is no other health insurance" do
      @veteran.other_health_insurance_id = OtherHealthInsurance.where(title: 'No').first.id
      expect(@veteran.has_other_health_insurance?).to eq false
    end

    it "should raise error if status of other health insurance is unknown" do
      @veteran.other_health_insurance_id = OtherHealthInsurance.where(title: 'Unknown').first.id
      expect{@veteran.has_other_health_insurance?}.to raise_error(RuntimeError, "Unknown")
    end

    it "should return all veterans with blank filter" do
      params = {
        :ssn => "",
        :first_name => "",
        :last_name => "",
        :from_date => "",
        :to_date => "",
      }
      filtered_veterans = Cpp::Local::Veteran.filtered_veterans(params)
      expect(filtered_veterans.count).to eq Cpp::Local::Veteran.count
    end

    it "should filter veterans by ssn" do
      params = {
        :ssn => @veteran.ssn,
        :first_name => "",
        :last_name => "",
        :from_date => "",
        :to_date => "",
      }
      filtered_veterans = Cpp::Local::Veteran.filtered_veterans(params)
      expect(filtered_veterans.count).to eq 1
      expect(filtered_veterans.first).to eq @veteran
    end

    it "should filter veterans by first name" do
      params = {
        :ssn => "",
        :first_name => @veteran.first_name,
        :last_name => "",
        :from_date => "",
        :to_date => "",
      }
      filtered_veterans = Cpp::Local::Veteran.filtered_veterans(params)
      expect(filtered_veterans.count).to eq 1
      expect(filtered_veterans.first).to eq @veteran
    end

    it "should filter veterans by last name" do
      params = {
        :ssn => "",
        :first_name => "",
        :last_name => @veteran.last_name,
        :from_date => "",
        :to_date => "",
      }
      filtered_veterans = Cpp::Local::Veteran.filtered_veterans(params)
      expect(filtered_veterans.count).to eq 1
      expect(filtered_veterans.first).to eq @veteran
    end

    it "should filter veterans by date" do
      params = {
        :ssn => "",
        :first_name => "",
        :last_name => "",
        :from_date => @veteran.created_at.strftime('%m/%d/%Y'),
        :to_date => @veteran.created_at.strftime('%m/%d/%Y'),
      }
      filtered_veterans = Cpp::Local::Veteran.filtered_veterans(params)
      expect(filtered_veterans.count).to eq 1
      expect(filtered_veterans.first).to eq @veteran
    end

  end
end
