# == Schema Information
#
# Table name: sites
#
#  id              :integer          not null, primary key
#  name            :string
#  address         :string
#  city            :string
#  state           :string
#  country         :string
#  created_at      :datetime         not null
#  updated_at      :datetime         not null
#  access_requests :string           default([]), is an Array
#  role_requests   :string           default([]), is an Array
#  site_station_number     :string
#  zip_code        :string
#  deleted_at      :datetime
#  clinics_loaded_state     :integer
#

require 'rails_helper'

RSpec.describe Site, :type => :model, cui: true do
  context "Associations" do
    it { should have_many(:site_role_sets) }
    it { should have_many(:examinations) }
    it { should have_many(:exam_requests) }
    it { should have_many(:clinics).dependent(:destroy) }
    it { should have_many(:users).through(:site_role_sets) }
  end

  context "Validations" do
    it { should validate_presence_of(:name) }
    it { should validate_presence_of(:address) }
    it { should validate_presence_of(:city) }
    it { should validate_presence_of(:state) }
    it { should validate_presence_of(:country) }
  end

  context "clinics_loaded_state" do
    it "has three valid states" do
      subject = create(:site)

      expect(subject.clinics_loaded_state).to eq("initial")
      expect(subject.loaded?).to be false

      subject.update!(clinics_loaded_state: :loading)
      expect(subject.clinics_loaded_state).to eq("loading")
      expect(subject.loaded?).to be false

      subject.update!(clinics_loaded_state: :loaded)
      expect(subject.clinics_loaded_state).to eq("loaded")
      expect(subject.loaded?).to be true

      expect { subject.update!(clinics_loaded_state: :testing) }.to raise_error(ArgumentError)
    end
  end

  context "#clinics_loaded?" do
    context "based on the ENV variable" do
      it "valid cached clinics" do
        stub_const("VISTA_CLINIC_CACHE_AGE_IN_DAYS", 7)
        subject = create(:site)

        expect(subject.clinics_loaded?).to be false

        subject.clinics << create(:clinic, created_at: 6.days.ago)
        subject.clinics_loaded_state = :loaded

        expect(subject.clinics_loaded?).to be true
      end

      it "invalid cached clinics" do
        stub_const("VISTA_CLINIC_CACHE_AGE_IN_DAYS", 7)
        subject = create(:site)

        expect(subject.clinics_loaded?).to be false

        subject.clinics << create(:clinic, created_at: 8.days.ago)
        subject.clinics_loaded_state = :loaded

        expect(subject.clinics_loaded?).to be false
      end
    end
  end
end
