# == Schema Information
#
# Table name: examinations
#
#  id                                              :integer          not null, primary key
#  title                                           :string
#  exam_date                                       :datetime
#  state                                           :integer
#  exam_id                                         :integer
#  clinician                                       :integer
#  created_at                                      :datetime         not null
#  updated_at                                      :datetime         not null
#  site_id                                         :integer
#  examination_state_id                            :integer
#  active                                          :boolean
#  examfile                                        :string
#  exam_request_id                                 :integer
#  purpose                                         :string
#  evaluation_id                                   :integer
#  examination_state_start_date                    :datetime
#  claim_folder_reviewed                           :boolean
#  reviewed_not_indicated                          :boolean
#  reviewed_military_service_treatment_records     :boolean
#  reviewed_military_service_personnel_records     :boolean
#  reviewed_military_enlistment_examination        :boolean
#  reviewed_military_separation_examination        :boolean
#  reviewed_military_post_deployment_questionnaire :boolean
#  reviewed_dod_form_214_separation_documents      :boolean
#  reviewed_vha_treatment_records                  :boolean
#  reviewed_civilian_medical_records               :boolean
#  reviewed_interviews_with_collateral_witnesses   :boolean
#  reviewed_at                                     :datetime
#  reviewed_by                                     :integer
#  interview                                       :text
#  plan                                            :text
#  examfiles                                       :string           default([]), is an Array
#  claim_id                                        :integer
#  cancellation_reason                             :text
#  identifier                                      :string
#  acknowledged                                    :boolean
#  clinician_acknowledged                          :boolean
#

require 'rails_helper'

RSpec.describe Examination, :type => :model, cui: true do

  context "Associations" do
    it { should belong_to(:examination_state) }
    it { should belong_to(:site) }
    it { should belong_to(:claim) }
    it { should belong_to(:parent).
         class_name('ExamRequest').
         with_foreign_key('exam_request_id')
    }
    it { should belong_to(:clinician_user).
         class_name('User').
         with_foreign_key('clinician')
    }
    it { should belong_to(:exam_request) }

    it { should have_one(:full_body_evaluation).
         dependent(:destroy)
    }
    it { should have_one(:examination_review_questionnaire) }

    it { should have_and_belong_to_many(:contentions) }

    it { should have_many(:examination_schedules) }
    it { should have_many(:examination_notes) }
  end

  context "Validations" do
    # FIXME: This model is coupled very tightly to Contentions and FullBodyEvaluation
    # needs to be refactored.
    # it "verifies that the total size of all examfiles does not exceed 125 megabytes"
  end

  context "Methods" do

    it "#clinician_name" do
      subject = Examination.new

      clinician = create(:user, first_name: "first", last_name: "last")
      subject.assign_a_clinician(clinician.id)

      expect(subject.clinician_name).to eq("first last")
    end

    context "methods that depend on the examination being saved/save-able" do

      let(:subject) { create(:examination) }

      before do
        # This model has many dependencies on the state of the world...
        create(:fbe_spec)
        create(:evaluation_template)
      end

      context "reviewing" do
        it "without a supplied reviewer" do
          expect(subject.reviewed?).to be false
          subject.review!
          expect(subject.reviewed?).to be true
          expect(subject.reviewed_by).to be nil
        end

        it "with a supplied reviewer" do
          user = create(:user)
          subject.review!(user.id)

          expect(subject.reviewed?).to be true
          expect(subject.reviewed_by).to eq(user.id)
        end

        it "can only be reviewed once" do
          user1 = create(:user)
          user2 = create(:user)

          subject.review!(user1.id)
          subject.review!(user2.id)

          expect(subject.reviewed?).to be true
          expect(subject.reviewed_by).to eq(user1.id)
        end
      end

      it "#most_recent_active_appointment" do
        expect(subject.most_recent_active_appointment).to be nil

        examination_schedule_1 = create(:examination_schedule_active)
        examination_schedule_2 = create(:examination_schedule_active)
        examination_schedule_3 = create(:examination_schedule_not_active)

        subject.examination_schedules << examination_schedule_1
        subject.examination_schedules << examination_schedule_2
        subject.examination_schedules << examination_schedule_3

        expect(subject.most_recent_active_appointment).to eq(examination_schedule_2)
      end

      it "#set_the_creating_user when a full_body_evaluation is present" do
        user = create(:user)
        subject.set_the_creating_user(user.id)

        expect(subject.full_body_evaluation[:assigner_id]).to eq(user.id)
      end

      # This test fails because of a foreign key violation
      xit "#set_the_creating_user when a full_body_evaluation is not present" do
        user = create(:user)
        subject.full_body_evaluation.destroy
        subject.set_the_creating_user(user.id)

        expect(subject.full_body_evaluation).to be nil
      end
    end
  end
end
