# == Schema Information
#
# Table name: exam_requests
#
#  id                        :integer          not null, primary key
#  xml                       :text
#  claim_id                  :integer
#  error_hash                :text
#  created_at                :datetime
#  updated_at                :datetime
#  site_id                   :integer
#  identifier                :string
#  exam_request_state_id     :integer
#  cancellation_reason       :text
#  cancellation_acknowledged :boolean
#

require 'rails_helper'

RSpec.describe ExamRequest, :type => :model, cui: true do

  context "Associations" do
    it { should belong_to(:claim) }
    it { should belong_to(:site) }
    it { should belong_to(:exam_request_state) }
    it { should belong_to(:exam_requester) }

    it { should have_many(:contentions) }
    it { should have_many(:examinations) }
  end

  context "Methods" do

    let(:claim) {create(:claim)}
    let(:site) {create(:site)}
    let(:exam_requester) {create(:exam_requester)}
    let(:exam_request) {create(:exam_request, claim: claim, site: site, exam_requester: exam_requester)}
    let(:exam_request2) {create(:exam_request, claim: claim, site: site, exam_requester: nil)}
    let(:contention) {create(:contention, exam_request: exam_request, claim: claim)}
    let(:contention2) {create(:contention, exam_request: exam_request2, claim: claim)}
    let(:dbq_information) {create(:dbq_information)}
    let(:examination_state_ip) {create(:examination_state_in_progress)}
    let(:examination_state_cp) {create(:examination_state_completed)}
    let!(:evaluation_spec) {create(:fbe_spec)}
    let!(:eval_template) {create(:evaluation_template)}

    it "responds to exams_requested" do
      contention.dbq_informations << dbq_information
      expect(exam_request.exams_requested).to eq "DBQ PSYCH Mental Disorders"
    end

    it "responds to requester_name_and_org" do
      expect(exam_request.requester_name_and_org).to eq "Alice Smith:VBA"
      expect(exam_request2.requester_name_and_org).to eq "Not Supplied"
    end

    it "reponds to total_number_of_exams_pending" do
      exam_test = Examination.new(claim: claim, site: site, examination_state: examination_state_ip, exam_request: exam_request)
      exam_test.contentions << contention
      exam_test.save

      exam_test2 = Examination.new(claim: claim, site: site, examination_state: examination_state_cp, exam_request: exam_request2)
      exam_test2.contentions << contention2
      exam_test2.save

      expect(exam_request.total_number_of_exams_pending).to eq 1
      expect(exam_request2.total_number_of_exams_pending).to eq 0
    end

    it "responds to requester_info" do
      expect(exam_request.requester_info(:full_name)).to eq("Alice Smith")
      expect(exam_request.requester_info(:organization)).to eq("VBA")
      expect(exam_request.requester_info(:email_address)).to eq("alice@smith.com")
      expect(exam_request2.requester_info(:email_address)).to eq("Not Supplied")
    end

    it "responds to first_name" do
      expect(exam_request.first_name).to eq("Joe")
    end

    it "responds to last_name" do
      expect(exam_request.last_name).to eq("Veteran")
    end

    it "responds to date_of_birth" do
      expect(exam_request.date_of_birth).to eq(DateTime.new(1990,1,1))
    end
  end
end
