# == Schema Information
#
# Table name: contentions
#
#  id                                              :integer          not null, primary key
#  claim_id                                        :integer
#  resolved_at                                     :datetime
#  vba_diagnostic_code                             :string
#  created_at                                      :datetime
#  updated_at                                      :datetime
#  name                                            :text
#  history                                         :text
#  no_diagnosis                                    :boolean
#  claim_folder_reviewed                           :boolean
#  reviewed_military_service_treatment_records     :boolean
#  reviewed_military_service_personnel_records     :boolean
#  reviewed_military_enlistment_examination        :boolean
#  reviewed_military_separation_examination        :boolean
#  reviewed_military_post_deployment_questionnaire :boolean
#  reviewed_dod_form_214_separation_documents      :boolean
#  reviewed_vha_treatment_records                  :boolean
#  reviewed_civilian_medical_records               :boolean
#  reviewed_interviews_with_collateral_witnesses   :boolean
#  reviewed_at                                     :datetime
#  reviewed_by                                     :integer
#  assigner_id                                     :integer
#  reviewed_not_indicated                          :boolean
#  exam_related_contention_id                      :string
#  clarification_requested                         :boolean
#  is_contention_cancelled                         :boolean
#  reason_for_cancellation                         :text
#  cancellation_details                            :text
#  identifier                                      :string
#

require 'rails_helper'

RSpec.describe Contention, :type => :model, cui: true do

  context "Associations" do
    it { should belong_to(:claim) }
    it { should belong_to(:exam_request) }
    it { should belong_to(:assigner).class_name('User') }
    it { should belong_to(:reviewer).class_name('User').
         with_foreign_key(:reviewed_by)
    }

    it { should have_and_belong_to_many(:evaluations) }
    it { should have_and_belong_to_many(:examinations) }

    it { should have_many(:rejections) }
    it { should have_many(:dbq_informations).
         through(:contentions_dbq_informations).
         dependent(:destroy)
    }
    it { should have_many(:contentions_dbq_informations) }
    it { should have_many(:contention_details) }
    it { should have_many(:clarification_details) }
    it { should have_many(:exam_prioritization_special_issues) }
    it { should have_many(:rework_reasons) }
    it { should have_many(:rework_reason_free_texts) }
  end

  describe "#resolved?" do
    context "when the contention has been resolved" do
      before do
        @contention = Contention.new(resolved_at: Time.now)
      end

      it "should return true" do
        expect(@contention.resolved?).to eq true
      end
    end

    context "when the contention has not been resolved" do
      before do
        @contention = Contention.new
      end

      it "should return false" do
        expect(@contention.resolved?).to eq false
      end
    end
  end

  describe "resolve!" do
    context "a contention without history or valid review" do
      before do
        @user = User.create
        @contention = Contention.new
        @result = @contention.resolve!(@user)
      end

      it "should return false" do
        expect(@result).to eq false
      end

      it "should not be resolved" do
        expect(@contention.resolved?).to eq false
      end
    end

    context "a contention with history and a valid review" do
      before do
        @user = User.create
        @contention = Contention.new(history: "something", claim_folder_reviewed: true)
        @contention.resolve!(@user)
      end

      it "should mark it resolved" do
        expect(@contention.resolved?).to eq true
      end
    end

    context "a contention with evaluations" do
      before do
        @user = User.create
        @contention = Contention.new(history: "history", claim_folder_reviewed: true)
        @evaluation_spec = create(:diabetes_spec)
        @evaluation = Evaluation.new(evaluation_spec: @evaluation_spec,
                                     doc: build(:diabetes_doc).doc,
                                     completed_at: Time.now,
                                     user: @user)
        @evaluation2 = Evaluation.new(evaluation_spec: @evaluation_spec,
                                     doc: build(:diabetes_doc).doc,
                                     completed_at: Time.now,
                                     user: @user)

        @contention.evaluations << @evaluation
        @contention.evaluations << @evaluation2
        @contention.resolve!(@user)
      end

      it "should generate EvaluationLogs for each contention" do
        expect(EvaluationLog.where(message: "Evaluation accepted by #{ @user.name }").length).to eq 2
      end
    end
  end

  describe "create" do
    context "when a contention is created with required attributes" do
      before do
        @contention = create(:contention, name: "Ickyitis")
      end
      it "should save properly with initialized records and claim_folder_reviewed attributes" do
        @contention = Contention.first
        expect(@contention.name).to eq "Ickyitis"
        expect(@contention.id).not_to be_nil
        expect(@contention.claim_folder_reviewed).to be_nil
      end
    end
  end

  describe "unresolve!" do
    before do
      @user = User.create
      @contention = create(:contention, name: "Ickyitis", resolved_at: Time.now)
      @evaluation_spec = create(:diabetes_spec)
      @evaluation = Evaluation.new(evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @evaluation2 = Evaluation.new(evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)

      @contention.evaluations << @evaluation
      @contention.evaluations << @evaluation2
    end

    it "should mark the contention as unresolved" do
      expect(@contention.resolved_at).not_to eq nil
      @contention.unresolve! @user
      expect(@contention.resolved_at).to eq nil
    end

    it "should unresolve all evaluations" do
      expect(@evaluation.completed_at).not_to eq nil
      expect(@evaluation2.completed_at).not_to eq nil
      @contention.unresolve! @user
      expect(@evaluation.completed_at).to eq nil
      expect(@evaluation2.completed_at).to eq nil
    end
  end

  describe "rejection" do
    before do
      @contention = create(:contention, name: "Ickyitis")
    end

    it "should return the newest rejection" do
      @contention.rejections << create(:rejection)
      @contention.rejections << create(:rejection, reason: "newest rejection")
      expect(@contention.rejection.reason).to eq "newest rejection"
    end

    it "should return nil if there are no rejections" do
      expect(@contention.rejection).to eq nil
    end
  end

  describe "rejections" do
    before do
      @contention = create(:contention, name: "Ickyitis")
      @contention.rejections << create(:rejection, reason: "oldest rejection")
      @contention.rejections << create(:rejection, reason: "newest rejection")
    end

    it "should have all rejections" do
      expect(@contention.rejections.length).to eq 2
    end

    it "should order them in ascending order" do
      expect(@contention.rejections[0].reason).to eq "oldest rejection"
      expect(@contention.rejections[1].reason).to eq "newest rejection"
    end
  end
end
