# == Schema Information
#
# Table name: consultations
#
#  id                     :integer          not null, primary key
#  consultation_number    :string           not null
#  ordering_provider_id   :integer
#  valid_from             :datetime
#  valid_to               :datetime
#  content                :json
#  created_at             :datetime         not null
#  updated_at             :datetime         not null
#  care_category_id       :integer
#  consultation_order_id  :integer
#  consultation_status_id :integer
#  consultation_type_id   :integer
#  veteran_id             :integer
#

require 'rails_helper'

RSpec.describe Consultation, :type => :model, cpp: true do

  context "Associations" do
    it { should belong_to(:care_category) }
    it { should belong_to(:consultation_order) }
    it { should belong_to(:consultation_status) }
    it { should belong_to(:consultation_type) }
    it { should belong_to(:ordering_provider).class_name('Provider') }

    it { should have_many(:referrals) }
    it { should have_many(:facilities).
         through(:referrals)
    }
    it { should have_many(:visns).
         through(:referrals)
    }
    it { should have_many(:comments).class_name('ConsultationComment') }
  end

  context "Validations" do
    subject { build(:consultation) }

    it { should validate_presence_of(:care_category_id) }
    it { should validate_presence_of(:veteran_ssn) }
    it { should validate_presence_of(:consultation_number) }
    it { should validate_presence_of(:valid_to) }
    it { should validate_presence_of(:valid_from) }

    # NOTE: consultation_number is a string '01-0000' that is why we have to use the case_insensitive option
    it { should validate_uniqueness_of(:consultation_number).case_insensitive }

    it ".urgent_priority has to be either true or false" do
      consultation = Consultation.new
      consultation.valid?
      expect(consultation.errors.messages[:base]).to include("Request Priority cannot be blank")

      consultation.urgent_priority = true
      consultation.valid?
      expect(consultation.errors.messages[:base]).to be_nil

      consultation.urgent_priority = false
      consultation.valid?
      expect(consultation.errors.messages[:base]).to be_nil
    end

    it "validates requesting_provider_telephone if one is present" do
      consultation = build_stubbed(:consultation)
      expect(consultation).to be_valid

      consultation.requesting_provider_telephone = "123-456-7890"
      expect(consultation).to be_valid

      consultation.requesting_provider_telephone = "1234567890"
      expect(consultation).not_to be_valid

      consultation.requesting_provider_telephone = "123.456.7890"
      expect(consultation).not_to be_valid

      consultation.requesting_provider_telephone = "aaa-bbb-cccc"
      expect(consultation).not_to be_valid
    end

    it "validates requesting_provider_fax if one is present" do
      consultation = build_stubbed(:consultation)
      expect(consultation).to be_valid

      consultation.requesting_provider_fax = "123-456-7890"
      expect(consultation).to be_valid

      consultation.requesting_provider_fax = "1234567890"
      expect(consultation).not_to be_valid

      consultation.requesting_provider_fax = "123.456.7890"
      expect(consultation).not_to be_valid

      consultation.requesting_provider_fax = "aaa-bbb-cccc"
      expect(consultation).not_to be_valid
    end

  end

  context "#human_readable_request_priority" do
    it "routine" do
      subject = build_stubbed(:consultation, urgent_priority: false)
      expect(subject.human_readable_request_priority).to eq("Routine")
    end

    it "urgent" do
      subject = build_stubbed(:consultation, urgent_priority: true)
      expect(subject.human_readable_request_priority).to eq("Stat")
    end
  end

  context "#veteran_full_name" do
    it "with first, middle, and last names" do
      subject = build_stubbed(:consultation, veteran_first_name: "first", veteran_middle_name: "middle", veteran_last_name: "last")

      expect(subject.veteran_full_name).to eq("first middle last")
    end

    it "with first and last names" do
      subject = build_stubbed(:consultation, veteran_first_name: "first", veteran_middle_name: nil, veteran_last_name: "last")

      expect(subject.veteran_full_name).to eq("first  last")
    end

    it "with no names" do
      subject = build_stubbed(:consultation, veteran_first_name: nil, veteran_middle_name: nil, veteran_last_name: nil)

      expect(subject.veteran_full_name).to eq("  ")
    end
  end

  context "ssn related behavior" do
    subject { build_stubbed(:consultation, veteran_ssn: 123456789) }

    context "#veteran_formatted_ssn" do
      it "with a ssn of proper length" do
        expect(subject.veteran_formatted_ssn).to eq("123-45-6789")
      end
    end

    context "#hidden_formatted_ssn" do
      it "with a ssn of proper length" do
        expect(subject.hidden_formatted_ssn).to eq("XXX-XX-6789")
      end
    end

    context "#formatted_ssn" do
      it "with a ssn of proper length" do
        expect(subject.formatted_ssn).to eq("6789")
      end
    end
  end

  it "#care_category_title" do
    subject = build_stubbed(:consultation, care_category: CareCategory.new(title: "title"))

    expect(subject.care_category_title).to eq("title")
  end

  it "#status_name" do
    subject = build_stubbed(:consultation, consultation_status: ConsultationStatus.new(title: "title"))

    expect(subject.status_name).to eq("title")
  end

  it "#order_name" do
    subject = build_stubbed(:consultation, consultation_order: ConsultationOrder.new(title: "title"))

    expect(subject.order_name).to eq("title")
  end

  it "#facility_name" do
    subject = build_stubbed(:consultation, ordering_provider: build(:provider, facility: build(:facility, name: "facility name")))

    expect(subject.facility_name).to eq("facility name")
  end

  context "when creating associations" do

    before :context do
      @veteran = create(:veteran)
    end

    it "should have an associated veteran" do
      consultation = build_stubbed(:consultation, veteran: @veteran)
      expect(consultation.veteran(nil).full_name).to eq @veteran.full_name
    end

    it "should display consultation status name" do
      consultation = build_stubbed(:consultation, :open_consultation, veteran: @veteran)
      expect(consultation.status_name).to eq 'Open'

      consultation = build_stubbed(:consultation, :complete_consultation, veteran: @veteran)
      expect(consultation.status_name).to eq 'Complete'
    end

  end # context "when creating associations"

  context "when filtering using queues" do

    before :context do
      @status_queue_consultations = {
        with_stat: create(:consultation, :with_stat_priority),
        expired_with_incomplete: create(:consultation, :expired_with_referral_status_incomplete),
        expired_with_complete: create(:consultation, :expired_with_referral_status_complete),
        with_new_appointments: create(:consultation, :with_new_referral_appointments),
        with_cprs_appointments: create(:consultation, :with_appointments_added_to_cprs),
        with_appointments_before_3_days_and_no_status_update: create(:consultation, :with_medical_record_letters),
        with_appointments_before_3_days_and_updated_status: create(:consultation, :without_medical_record_letters)
      }
    end

    it "should return all stat consultations", cpp:true do
      results = Consultation.filter({"stat_data" => "true"})
      expect(results).to include(@status_queue_consultations[:with_stat])
    end

    it "should return all consultations with expired referral appointments", cpp:true do
      results = Consultation.filter({"expired_data" => "true"})
      expect(results).to include(@status_queue_consultations[:expired_with_incomplete])
      expect(results).not_to include(@status_queue_consultations[:expired_with_complete])
    end

    it "should return all consultations with new referral appointments", cpp:true do
      results = Consultation.filter({"new_appointments" => "true"})
      expect(results).to include(@status_queue_consultations[:with_new_appointments])
      expect(results).not_to include(@status_queue_consultations[:with_cprs_appointments])
    end

    it "should return all consultations with medical record letters", cpp:true do
      results = Consultation.filter({"medical_record_letters" => "true"})
      expect(results).to include(@status_queue_consultations[:with_appointments_before_3_days_and_no_status_update])
      expect(results).not_to include(@status_queue_consultations[:with_appointments_before_3_days_and_updated_status])
    end
  end

  context "when performing filtering" do

    # creating test data used for filtering tests
    before :context do
      @test_visn = Visn.where(name: 'VA Southeast Network').first
      @test_care_category = CareCategory.where(title: 'Echocardiogram').first
      @test_vha_user = create(:vha_user)
      @default_veteran = create(:veteran)
      @veteran_with_testname = create(:veteran, first_name: 'Homer', last_name: 'Simpson')
      @veteran_with_ssn = create(:veteran, first_name: 'TestSSN', last_name: 'UserSSN', ssn: '333669999')
      @veteran_with_dob = create(:veteran, first_name: 'TestDOB', last_name: 'UserDOB', date_of_birth: '11/30/1960')
      @consultations = {
        with_testname:         create(:consultation, veteran: @veteran_with_testname),
        with_test_ssn:         create(:consultation, veteran: @veteran_with_ssn),
        with_test_dob:         create(:consultation, veteran: @veteran_with_dob),
        with_care_category:    create(:consultation, :with_care_category,
                                      veteran: @default_veteran, care_category_title: @test_care_category.title),
        with_visn:             create(:consultation, :with_referral_having_visn,
                                      veteran: @default_veteran, visn: @test_visn),
        with_consult_number:   create(:consultation, veteran: @default_veteran, consultation_number: 'ABC-123'),
        with_auth_number:      create(:consultation, :with_referral_having_authorization_number,
                                      veteran: @default_veteran, authorization_number: 'ABC-123-456'),
        invalid_date_range:    create(:consultation, veteran: @default_veteran, valid_from: 1.year.ago, valid_to: 9.months.ago),
        with_coordinator:      create(:consultation, :with_referral_having_coordinator,
                                      veteran: @default_veteran, coordinator: @test_vha_user),
        open_consultation:     create(:consultation, :open_consultation, veteran: @default_veteran),
        complete_consultation: create(:consultation, :complete_consultation, veteran: @default_veteran),
        }
    end

    it "should return all consultations in collection when no filter params are passed in" do
      results = Consultation.filter
      expect(results.count).to be >= @consultations.values.count
      @consultations.values.each do |created_consultation|
        expect(results).to include(created_consultation)
      end
    end

    it "should filter based on veteran name" do
      results = Consultation.filter('first_name' => @veteran_with_testname.first_name)
      expect(results.count).to eq 1
      expect(results.first.veteran(nil)).to eq @veteran_with_testname
    end

    it "should return no records when filtering on name not in database" do
      results = Consultation.filter('first_name' => 'asdfrgasgasghadsfghafsg')
      expect(results).to be_empty
    end

    it "should filter based on veteran SSN" do
      results = Consultation.filter('ssn' => '333669999')
      expect(results.count).to eq 1
      expect(results.first.veteran(nil).first_name).to eq 'TestSSN'
      expect(results.first.veteran(nil).ssn).to eq '333669999'
    end

    it "should return no records when filtering on SSN not in database" do
      results = Consultation.filter('ssn' => '333668888')
      expect(results).to be_empty
    end

    it "should filter based on veteran DOB" do
      results = Consultation.filter('dob' => '11/30/1960')
      expect(results.count).to eq 1
      expect(results.first.veteran(nil).first_name).to eq 'TestDOB'
      expect(results.first.veteran(nil).date_of_birth).to eq '11/30/1960'
    end

    it "should return no records when filtering on DOB not in database" do
      results = Consultation.filter('dob' => '11/30/1900')
      expect(results).to be_empty
    end

    it "should filter based on authorization number" do
      results = Consultation.filter('authorization_number' => 'ABC-123-456')
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_auth_number]
    end

    it "should return no records when filtering on authorization number not in database" do
      results = Consultation.filter('authorization_number' => 'NOT-VALID-AUTHNUM')
      expect(results).to be_empty
    end

    it "should filter based on consultation number" do
      results = Consultation.filter('consultation_number' => 'ABC-123')
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_consult_number]
    end

    it "should return no records when filtering on consultation number not in database" do
      results = Consultation.filter('consultation_number' => 'NOT-VALID-CONSULT_NUM')
      expect(results).to be_empty
    end

    it "should filter based on VISN id" do
      results = Consultation.filter('visn_id' => @test_visn.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_visn]
    end

    it "should return no records when filtering on VISN without consultations" do
      visn_not_found = Visn.where(name: 'Sierra Pacific Network').first
      expect(visn_not_found).not_to be_nil

      results = Consultation.filter('visn_id' => visn_not_found.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on care category id" do
      results = Consultation.filter('care_category_id' => @test_care_category.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_care_category]
    end

    it "should return no records when filtering on care category id not found" do
      care_category_not_found = CareCategory.where(title: 'Diabetes').first
      expect(care_category_not_found).not_to be_nil

      results = Consultation.filter('care_category_id' => care_category_not_found.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on coordinator id" do
      results = Consultation.filter('coordinator_id' => @test_vha_user.id.to_s)
      expect(results.count).to eq 1
      expect(results.first).to eq @consultations[:with_coordinator]
    end

    it "should return no records when filtering on coordinator not in database" do
      not_a_coordinator = create(:examiner)
      results = Consultation.filter('coordinator_id' => not_a_coordinator.id.to_s)
      expect(results).to be_empty
    end

    it "should filter based on referral type" do
      results = Consultation.filter('referral_type' => 'MD Office Visit')
      expect(results).to_not be_empty
    end

    it "should return no records when filtering on referral type not in database" do
      not_valid_referral_type = ReferralType.where(title: 'NotValid').first
      results = Consultation.filter('referral_type' => not_valid_referral_type)
      results = results.first['referral_type_id']
      expect(results).to be_nil
    end

    it "should filter based on a single selected status" do
      open_status = ConsultationStatus.where(title: 'Open').first
      open_results = Consultation.filter('status' => [open_status.id])
      expect(open_results.count).to eq 1
      expect(open_results.first).to eq @consultations[:open_consultation]
      expect(open_results).not_to include(@consultations[:complete_consultation])

      complete_status = ConsultationStatus.where(title: 'Complete').first
      complete_results = Consultation.filter('status' => [complete_status.id])
      expect(complete_results.count).to eq 1
      expect(complete_results.first).to eq @consultations[:complete_consultation]
      expect(complete_results).not_to include(@consultations[:open_consultation])
    end

    it "should filter based on multiple statuses by returning consultations with any of the included statuses" do
      open_status = ConsultationStatus.where(title: 'Open').first
      complete_status = ConsultationStatus.where(title: 'Complete').first
      results = Consultation.filter('status' => [open_status.id, complete_status.id])
      expect(results.count).to eq 2
      expect(results).to include(@consultations[:open_consultation])
      expect(results).to include(@consultations[:complete_consultation])
    end


    context "including a record with all filter fields" do

      # creating test data used for filtering tests
      before :example do
        @consultation_with_all_fields =
          create(:consultation, :with_care_category, :open_consultation,
                 :with_single_referral_having_multiple_fields,
                 veteran: @veteran_with_ssn,
                 consultation_number: 'ABC-999',
                 authorization_number: 'ABC-123-999',
                 care_category_title: @test_care_category.title,
                 visn: @test_visn,
                 coordinator: @test_vha_user)
      end

      it "should filter based on veteran name" do
        results = Consultation.filter('first_name' => @veteran_with_ssn.first_name)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_test_ssn])
      end

      it "should filter based on veteran SSN" do
        results = Consultation.filter('ssn' => '333669999')
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_test_ssn])
      end

      it "should filter based on VISN id" do
        results = Consultation.filter('visn_id' => @test_visn.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_visn])
      end

      it "should filter based on care category id" do
        results = Consultation.filter('care_category_id' => @test_care_category.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_care_category])
      end

      it "should filter based on coordinator id" do
        results = Consultation.filter('coordinator_id' => @test_vha_user.id.to_s)
        expect(results.count).to eq 2
        expect(results).to include(@consultation_with_all_fields)
        expect(results).to include(@consultations[:with_coordinator])
      end

      it "should filter based on all filterable consultation/veteran record fields plus coordinator" do
        results = Consultation.filter(
          'first_name' => @veteran_with_ssn.first_name,
          'last_name' => @veteran_with_ssn.last_name,
          'ssn' => @veteran_with_ssn.ssn,
          'visn_id' => @test_visn.id.to_s,
          'care_category_id' => @test_care_category.id.to_s,
          'coordinator_id' => @test_vha_user.id.to_s,
          'consultation_number' => 'ABC-999'
        )
        expect(results.count).to eq 1
        expect(results).to include(@consultation_with_all_fields)
      end

      it "should filter based on all filterable consultation/veteran record fields plus authorization number" do
        results = Consultation.filter(
          'first_name' => @veteran_with_ssn.first_name,
          'last_name' => @veteran_with_ssn.last_name,
          'ssn' => @veteran_with_ssn.ssn,
          'visn_id' => @test_visn.id.to_s,
          'care_category_id' => @test_care_category.id.to_s,
          'authorization_number' => 'ABC-123-999',
          'consultation_number' => 'ABC-999'
        )
        expect(results.count).to eq 1
        expect(results).to include(@consultation_with_all_fields)
      end

    end # context "including a record with all filter fields"


    context "and generating CSV documents based on filtered results" do

      before :context do
        @veteran_names = Cpp::Local::Veteran.all.map &:first_name
        @find_any_name = -> (data_row) do
          @veteran_names.any? {|name| data_row.include? name }
        end
      end

      it "should return all records in CSV file" do
        csv_rows = Consultation.to_csv({}).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq (Consultation.count + 1)

        csv_header = csv_rows.first
        csv_rows_without_header = csv_rows.drop 1

        expect(@find_any_name.call(csv_header)).to eq false
        csv_rows_without_header.each do |csv_row|
          expect(@find_any_name.call(csv_row)).to eq true
        end
      end

      it "should return one record in CSV file for one-record filter" do
        csv_rows = Consultation.to_csv('first_name' => @veteran_with_testname.first_name).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq 2

        # header line should not contain veteran names, and first line should
        # contain veteran whose name was passed into to_csv filter parameter.
        expect(@find_any_name.call(csv_rows[0])).to eq false
        expect(@find_any_name.call(csv_rows[1])).to eq true
        expect(csv_rows[1]).to include(@veteran_with_testname.first_name)
      end

    end # context "and generating CSV documents based on filtered results"

    context "" do
      before :context do
          @referral_status_search = {
          new_referral_status:                 create(:consultation, :new_referral_status, veteran: @default_veteran),
          prep_referral_status:                create(:consultation, :prep_referral_status, veteran: @default_veteran),
          assigned_referral_status:            create(:consultation, :assigned_referral_status, veteran: @default_veteran),
          accepted_referral_status:            create(:consultation, :accepted_referral_status, veteran: @default_veteran),
          review_pending_referral_status:      create(:consultation, :review_pending_referral_status, veteran: @default_veteran),
          information_needed_referral_status:  create(:consultation, :information_needed_referral_status,
                                               veteran: @default_veteran),
          complete_referral_status:            create(:consultation, :complete_referral_status, veteran: @default_veteran)
        }
      end

      it "should filter based on referral status New" do
        New_status = ReferralStatus.where(name: 'New').first
        New_results = Consultation.filter('referral_status' => [New_status.id])
        expect(New_results).to_not be_empty
      end

      it "should filter based on referral status Prep" do
        Prep_status = ReferralStatus.where(name: 'Prep').first
        Prep_results = Consultation.filter('referral_status' => [Prep_status.id])
        expect(Prep_results).to_not be_empty
      end

      it "should filter based on referral status Assigned" do
        Assigned_status = ReferralStatus.where(name: 'Assigned').first
        Assigned_results = Consultation.filter('referral_status' => [Assigned_status.id])
        expect(Assigned_results).to_not be_empty
      end

      it "should filter based on referral status Accepted" do
        Accepted_status = ReferralStatus.where(name: 'Accepted').first
        Accepted_results = Consultation.filter('referral_status' => [Accepted_status.id])
        expect(Accepted_results).to_not be_empty
      end

      it "should filter based on referral status Review Pending" do
        Review_pending_status = ReferralStatus.where(name: 'Review Pending').first
        Review_pending_results = Consultation.filter('referral_status' => [Review_pending_status.id])
        expect(Review_pending_results).to_not be_empty
      end

      it "should filter based on referral status Information Needed" do
        Information_needed_status = ReferralStatus.where(name: 'Information Needed').first
        Information_needed_results = Consultation.filter('referral_status' => [Information_needed_status.id])
        expect(Information_needed_results).to_not be_empty
      end

      it "should filter based on referral status Complete" do
        Complete_status = ReferralStatus.where(name: 'Complete').first
        Complete_results = Consultation.filter('referral_status' => [Complete_status.id])
        expect(Complete_results).to_not be_empty
    end

    end

  end # context "when performing filtering"

end
