require "rails_helper"

RSpec.describe ExaminationAssignClinician, cui:true do
  # Not able to use instance doubles on ActiveRecord classes because of their dynamic methods
  let(:examination_history_klass) { double("ExaminationHistory") }
  let(:exam_management_mailer_klass) { instance_double("ExamManagementMailer") }
  let(:user_klass) { double("User") }

  context "if the clinician is the same" do
    it "nothing happens" do
      clinician_id = 777
      clinician = double(id: clinician_id)

      subject = ExaminationAssignClinician.new(examination_history_klass, exam_management_mailer_klass, user_klass)

      examination = double("examination", clinician: clinician_id)
      current_user = double

      result = subject.run!(examination, clinician, current_user)
      expect(result).to be_nil
    end
  end

  context "if the clinician is different" do
    let(:clinician_1_id) { 111 }
    let(:clinician) { double(id: clinician_1_id, name: "current_clinician") }
    let(:current_user) { double(name: "name") }
    let(:examination_history) { double }
    let(:exam_mailer) { double }

    before do
      allow(examination_history).to receive(:save)
      allow(exam_management_mailer_klass).to receive(:examination_clinician_assigned).with(clinician).and_return(exam_mailer)
      allow(exam_mailer).to receive(:deliver_now)
    end

    it "and examination did not have a previously assigned clinician" do
      subject = ExaminationAssignClinician.new(examination_history_klass, exam_management_mailer_klass, user_klass)

      examination = double("examination", clinician: nil, id: 1337)
      examination_history_details = {
        :examination_id=>1337,
        :notes=>"#{current_user.name} assigned #{clinician.name} as the clinician."
      }

      allow(examination_history_klass).to receive(:new).with(examination_history_details).and_return(examination_history)
      allow(examination).to receive(:assign_a_clinician).with(clinician.id)
      allow(examination).to receive(:save)

      result = subject.run!(examination, clinician, current_user)
      expect(result).to eq("Examination has been successfully updated.")

      expect(examination_history_klass).to have_received(:new).with(examination_history_details)
      expect(examination_history).to have_received(:save)

      expect(examination).to have_received(:assign_a_clinician).with(clinician.id)
      expect(examination).to have_received(:save)

    end

    it "and examination had a previously assigned clinician" do
      subject = ExaminationAssignClinician.new(examination_history_klass, exam_management_mailer_klass, user_klass)

      previous_clinician = double("examination_clinician", name: "examination_clinician")
      examination = double("examination", clinician: 777, id: 1337)
      examination_history_details = {
        :examination_id=>1337,
        :notes=>"#{current_user.name} changed the clinician from #{previous_clinician.name} to #{clinician.name}."
      }

      allow(user_klass).to receive(:find_by_id).with(examination.clinician).and_return(previous_clinician)
      allow(examination_history_klass).to receive(:new).with(examination_history_details).and_return(examination_history)
      allow(examination).to receive(:assign_a_clinician).with(clinician.id)
      allow(examination).to receive(:save)

      result = subject.run!(examination, clinician, current_user)
      expect(result).to eq("Examination has been successfully updated.")

      expect(examination_history_klass).to have_received(:new).with(examination_history_details)
      expect(examination_history).to have_received(:save)

      expect(examination).to have_received(:assign_a_clinician).with(clinician.id)
      expect(examination).to have_received(:save)

    end
  end
end
