# == Schema Information
#
# Table name: consultations
#
#  id                     :integer          not null, primary key
#  consultation_number    :string           not null
#  ordering_provider_id   :integer
#  valid_from             :datetime
#  valid_to               :datetime
#  content                :json
#  created_at             :datetime         not null
#  updated_at             :datetime         not null
#  care_category_id       :integer
#  consultation_order_id  :integer
#  consultation_status_id :integer
#  consultation_type_id   :integer
#  veteran_id             :integer
#

FactoryBot.define do

  get_content = -> (n) do
    fourdigit = (n+2345) % 10000
    {
      npi: "",
      urgent_priority: false,
      ordering_physician_name: "Test#{n} OrderPhys",
      requesting_provider_telephone: "727-555-#{fourdigit}",
      requesting_provider_fax: "727-999-#{fourdigit}",
      requesting_provider_contact_name: "TEST#{n} PHYS CONTACT"
   }
  end

  # Default consultation: use default (first) veteran and default consultation
  # status, category of care, and consultation order.  The default value text is
  # explicitly designated for readability purposes -- do not use these as explicit
  # values when performing searches or queries.
  # ConsultationStatus, ConsultationOrder, and CareCategory get preloaded
  # when running `rake db:seed`.
  factory :consultation do
    veteran  { Cpp::Local::Veteran.first || FactoryBot.create(:veteran) }
    veteran_mpi_pid       { "123456789"}
    veteran_local_pid     { "456789" }
    consultation_status   { ConsultationStatus.where(title: 'New').first }
    care_category         { CareCategory.where(title: 'Chiropractic').first }
    consultation_order    { ConsultationOrder.where(title: 'Evaluate and Recommend').first }
    sequence(:consultation_number) {|n| sprintf("%02d-%04d", n, (n*37) % 10_000) }
    sequence(:content)             {|n| get_content.call(n) }
    valid_from  { 1.month.ago }
    valid_to    { 1.month.from_now }

    trait :open_consultation do
      consultation_status { ConsultationStatus.where(title: 'Open').first }
    end

    trait :complete_consultation do
      consultation_status { ConsultationStatus.where(title: 'Complete').first }
    end

    trait :urgent do
      urgent_priority true
    end

    trait :recently_no_longer_valid do
      valid_from  { 6.weeks.ago }
      valid_to    { 2.weeks.ago }
    end

    trait :no_longer_valid do
      valid_from  { 12.weeks.ago }
      valid_to    { 8.weeks.ago }
    end

    trait :with_care_category do
      transient do
        care_category_title nil
      end
      care_category { CareCategory.where(title: care_category_title).first }
    end

    trait :with_referral do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation)
      end
    end

    trait :with_referral_having_authorization_number do
      transient do
        authorization_number nil
      end
      after(:create) do |saved_consultation, evaluator|
        FactoryBot.create(:referral, consultation: saved_consultation,
                                      authorization_number: evaluator.authorization_number)
      end
    end

    trait :with_referral_having_coordinator do
      transient do
        coordinator nil
      end
      after(:create) do |saved_consultation, evaluator|
         FactoryBot.create(:referral, consultation: saved_consultation,
                                       coordinator: evaluator.coordinator)
      end
    end

    trait :with_referral_having_visn do
      transient do
        visn nil
      end
      after(:create) do |saved_consultation, evaluator|
        facility = FactoryBot.create(:facility, visn: evaluator.visn)
        provider = FactoryBot.create(:provider, facility: facility)
        FactoryBot.create(:referral, consultation: saved_consultation,
                                      provider: provider)
      end
    end

    # Creates a single referral field with data fields equal to all the
    # transient values passed in.  The other traits (with_referral_having_authorization_number,
    # with_referral_having_coordinator, etc.) each create a separate referral.
    trait :with_single_referral_having_multiple_fields do
      transient do
        authorization_number nil
        coordinator nil
        visn nil
        referral_status nil
      end
      after(:create) do |saved_consultation, evaluator|
        facility = FactoryBot.create(:facility, visn: evaluator.visn)
        provider = FactoryBot.create(:provider, facility: facility)
        FactoryBot.create(:referral,
          consultation: saved_consultation,
          authorization_number: evaluator.authorization_number,
          coordinator: evaluator.coordinator,
          provider: provider,
          referral_status: ReferralStatus.where(name: 'Accepted').first,
          referral_type: ReferralType.where(title: 'MD Office Visit').first
        )
      end
    end

    trait :with_stat_priority do
      urgent_priority true
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Assigned').first)
      end
    end

    trait :expired_with_referral_status_complete do
      valid_from  { 6.weeks.ago }
      valid_to    { 2.weeks.ago }
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Complete').first)
      end
    end

    trait :expired_with_referral_status_incomplete do
      valid_from  { 6.weeks.ago }
      valid_to    { 2.weeks.ago }
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Assigned').first)
      end
    end

    trait :with_new_referral_appointments do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, :with_new_appointments, consultation: saved_consultation, provider: create(:provider))
      end
    end

    trait :with_appointments_added_to_cprs do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, :with_appointments_added_to_cprs, consultation: saved_consultation, provider: create(:provider))
      end
    end

    trait :with_medical_record_letters do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, :with_new_appointments, provider: create(:provider),
                        referral_appointment_time: 1.week.ago,
                        consultation: saved_consultation)
      end
    end

    trait :without_medical_record_letters do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, :complete, :with_new_appointments, provider: create(:provider),
                        referral_appointment_time: 1.week.ago,
                        consultation: saved_consultation)
      end
    end

    trait :new_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'New').first)
      end
    end

    trait :prep_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Prep').first)
      end
    end

    trait :assigned_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Assigned').first)
      end
    end

    trait :accepted_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Accepted').first)
      end
    end

    trait :review_pending_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Review Pending').first)
      end
    end

    trait :information_needed_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Information Needed').first)
      end
    end

    trait :complete_referral_status do
      after(:create) do |saved_consultation|
        FactoryBot.create(:referral, consultation: saved_consultation, referral_status: ReferralStatus.where(name: 'Complete').first)
      end
    end
  end

end
