namespace :exam_requests do

  # NOTE: yep, there is only one in the array; setting a pattern to
  #       to be used in a set of common support tasks to be used by other tasks.
  REQUIRED_UTILITIES = %w[ curl ]

  desc "Copy existing exam_request data to claims;"
  task copy_existing_data_to_claims: :environment do
    ExamRequest.where.not(claim_id: nil).each do |exam_request|
      begin
        if claim = Claim.find(exam_request.claim_id)
          if claim.exam_request_data.blank?
            if claim.update_attributes!(exam_request_data:  Hash.from_xml(exam_request.xml))
            puts "Adding exam_request_data to claim: #{claim.id}  (ExamRequest: #{exam_request.id.to_s})" if Rails.env.production? || Rails.env.development?
            end
          end
        end
      rescue Exception => e
        puts "#{e.message} (ExamRequest: #{exam_request.id.to_s})" if Rails.env.production? || Rails.env.development?
        Rails.logger.info "Rescued #{e.inspect} -- ExamRequest #{exam_request.id} has a dangling claim id."
      end
    end # ExamRequest.where.not(claim_id: nil).each do |exam_request|
  end # task copy_existing_data_to_claims: :environment do


  #############################################
  # exam_requests:seed
  # Assumes: 1) you are running Puma, not any other server
  #          2) your machine recognizes the curl command
  #
  desc "Seed a development environment by sending a standard set of exam requests to a running CUI app"
  task seed: %w[ environment exam_requests:check_required_utilities ] do
    unless Rails.env.development?  &&  DBSEED
      STDERR.puts <<~EOM

        As a safety precaution, this task can only run when DBSEED is true
        and RAILS_ENV is development.

        Current values are:
          DBSEED:    #{DBSEED}
          RAILS_ENV: #{Nenv.RAILS_ENV}

      EOM
      exit(-1)
    end

    pid_path    = Rails.root + "tmp" + "pids" + "puma.pid"
    seed_path   = Rails.root + "db"  + "exam_requests"

    begin
      pid = File.read(pid_path).to_i
      Process.getpgid pid  # NOTE: Will throw an exception if the pid is not running

      # NOTE: This command intentionally targets the localhost as a safety
      #       measure to help prevent polluting non-development databases
      #       __filename__ is a placeholder that will be replaced with the
      #       full path to the file to be uploaded.
      command = %Q[
        curl -S -s --data @__filename__ \
             --header "Content-Type: application/xml" \
             --header "Accept: application/xml" \
             -X POST #{Nenv.PUMA_PROTOCOL.gsub('tcp','http')}localhost:#{Nenv.PUMA_PORT}#{Nenv.CLAIMS_API_URL}
      ] # TODO: check the puma protocol for https it might be ssl://

      seed_path.children.
        select{|a_path| !a_path.directory? && '.xml' == a_path.extname}.
        each do |filename|
          print "Seeding exam request: #{filename.basename} ... "
            `#{command.gsub('__filename__', filename.to_s)}`
          puts "done"
        end # seed_path.children

    rescue Errno::ESRCH, Errno::ENOENT
      STDERR.puts <<~EOM

        ERROR:  The CUI is not running on this machine.
                Please start the application in the development environment
                then attempt to seed your exam requests again.

      EOM
      exit(-1)
    end
  end # task seed: :environment do


  # TODO: checking to see that the requred command line utilities is a common
  #       thing to do across several rake tasks; consider extract this kind
  #       of task into a common core of utility tasks and just pass
  #       an array of the specific tasks required utilities to the common task.
  task :check_required_utilities do
    missing_utilities = []
    REQUIRED_UTILITIES.each do |cli_utility|
      # TODO: Find out if the Windoze environment supports 'which' as a standard
      #       shell command
      missing_utilities << cli_utility if `which #{cli_utility}`.empty?
    end

    unless missing_utilities.empty?
      STDERR.puts <<~EOM

        ERROR:  The following command line utilities are required but
                are not found on this machine:

                #{missing_utilities.join(', ')}

      EOM
      exit(-1)
    end
  end # task :check_required_utilities do


  # This task is used to generate a large number of exam requests for testing
  # by duplicating a standard exam request template file.  The template file
  # contains static text and placeholders that are denoted by the use of
  # three underscore character "_" as prefix and suffix to the placeholder
  # field name.  For example:   ___FIRST_NAME___
  #
  # This task takes two parameters:
  #   total ...... an integer > 0 -- the total number of exam requests to generate
  #   site_name .. the valid name of a VA Site.  If specified all generated exam
  #                requests will be for that specific site.
  #                (default) when not present the generated exam requests are
  #                randomly distributed across several different VA sites.  See
  #                the variable array "site_array" for a list of the hard-coded
  #                sites used.
  desc "Generates total>0 exam requests for a site_name(optional) site. "
  task :generate, [:total, :site_name] => [:environment] do |task_name, args|

    site_array = [
      "QTC",
      "VetFed",
      "VES",
      "Atlanta VA Medical Center",
      "Chicago Medical Center"
    ]

    usage = <<~USAGE

      Usage:  rake #{task_name}[total,'site_name']

      Where:
        total     is an integer > 0 (the number of exam requests to generate)
        site_name must be a valid VA Site name.  If it is missing the generated
                  exam requests are randomized amoung these Sites:
                    #{site_array.join("\n              ")}

      Note: if you specify a site_name it must be in single quotes -=> 'site_name'
            AND there can be no sapce after the comma that seperates the total
            exam requests and the site_name.

    USAGE

    total = args.total.to_i

    if args.total.blank?  &&  args.site_name.blank?
      puts usage
      exit(0)
    end

    template_path = Rails.root + 'lib/tasks/exam_request/exam_request_template_iepd_v2.0.xml'

    # Validate parameters and required resources

    errors = Array.new

    unless total > 0
      errors << "total must be an integer > 0 not: '#{args.total}'"
    end

    if args.site_name.blank?
      site_name = ""
    else
      unless Site.find_by_name(args.site_name).present?
        errors << "Unknown VA Site Name: '#{args.site_name}'"
      end
    end # if args.site_name.blank?


    unless template_path.exist?
      errors << "File does not exist: #{template_path}"
    end

    unless errors.empty?
      puts <<~ERRORS

        The follow errors were detected:

          #{errors.join("\n  ")}

      ERRORS
      puts usage
      exit(-1)
    end


    template_contents = template_path.read

    puts "Generating  #{total}  ExamRequests ..."

    total.times do |loop_index|
      print "#{loop_index}\r"  if total <= 1000 || 0 == loop_index%100

      contents = template_contents.clone

      first_name  = "FName" + (SecureRandom.random_number(1000)).to_s
      last_name   = "LName" + (SecureRandom.random_number(1000)).to_s
      file_number = SecureRandom.random_number(10000000)
      claim_id    = SecureRandom.hex

      contents.gsub!('___ER_UUID___',           SecureRandom.uuid)
      contents.gsub!('___EVENT_UUID___',        SecureRandom.uuid)
      contents.gsub!('___CONTENTION_UUID___',   SecureRandom.uuid)
      contents.gsub!('___CONTENTION_UUID2___',  SecureRandom.uuid)
      contents.gsub!('___CONTENTION_UUID3___',  SecureRandom.uuid)

      contents.gsub!('___FIRST_NAME___',  first_name)
      contents.gsub!('___LAST_NAME___',   last_name)
      contents.gsub!('___FILE_NUMBER___', file_number.to_s)
      contents.gsub!('___CLAIM_ID___',    claim_id)

      contents.gsub!( '___SITE___',
        site_name.blank? ? site_array.sample : site_name
      )

      # Tell the application about a new exam request

      begin
        VlerDASJob.postNotification(contents)
      rescue Errno::ECONNREFUSED
        puts <<~SYSTEM_ERROR

          SYSTEM ERROR:

            Is the application running?
            Check: #{Nenv.app_url}

        SYSTEM_ERROR
        exit(-1)
      rescue Exception => e
        puts <<~SYSTEM_ERROR

          SYSTEM ERROR:

            The following problem was reported while attempting
            to generate a new exam request:

            #{e}

        SYSTEM_ERROR
        exit(-1)
      end
    end # total.times do |loop_index|
    puts "Done."
  end # task generate: :environment do
end # namespace :exam_requests do