# frozen_string_literal: true
#
# This rake task is for manually adding users until
# SSOi/SSOe are fully deployed at all sites.
#
# Input is read from spreadsheet (xlsx) expecting one user per row
# There are 6 columns per row expected as follows:
#
# Index 0 (Column A): First Name               (mandatory)
# Index 1 (Column B): Last Name                (mandatory)
# Index 2 (Column C): Email Address            (mandatory)
# Index 3 (Column D): Roles                    (mandatory)
# Index 5 (Column F): NPI for provider         (optional, non_vha role only)
# Index 6 (Column G): Send Confirmation Email  (optional, default to true)
#
# NOTE: Sample input file can be found in $app/utlities/templates folder
#       named CPP_Access_Request_Form.xlsx, data for users is placed on
#       2nd worksheet.

require 'awesome_print'

namespace :admin do
  desc 'Add users to database from input file'

  task add_users: :environment do

    require "#{Rails.root}/lib/tasks/helpers/add_users_helper"
    require 'roo'
    require 'fileutils'
    require 'logger'
    require 'active_support'
    require 'action_mailer'
    require 'set'

    include AddUsersHelper

    ARGV.shift # remove task name from args

    # command line usage / feedback / error handling
    if ARGV.count < 1
      # added rake command directly; couldn't properly output current task
      abort "USAGE: rake admin:add_users <spreadsheet_file1> [<spreadsheet_file2> ...]"
    end
    ARGV.each do |filename_arg|
      unless File.exists? filename_arg
        abort "Spreadsheet file name #{filename_arg} not found."
      end
    end

    LOGIN_URL = ENV['APP_LOGIN_URL'] .freeze

    # constant names for spreadsheet columns within a row e.g. sheet.column(xxx_COL]
    WORKSHEET_WITH_DATA = 1
    SKIP_ROWS           = 2

    FIRST_NAME_COL      = 0
    LAST_NAME_COL       = 1
    EMAIL_COL           = 2
    ROLES_COL           = 3
    NPI_COL             = 4
    SEND_EMAIL_FLAG_COL = 5

    NPI_LENGTH          = 10
    NPI_WEBSITE         = "https://npiregistry.cms.hhs.gov/".freeze

    ADD_USERS_LOG       = "log/admin-add-users.log".freeze
    ADD_USERS_REGION    = Rails.env.freeze

    #Read Devise config on minimum password length
    PASSWORD_LENGTH     = Rails.application.config.devise.password_length.min.freeze

    user_app      = "ERROR"
    total_added   = 0
    total_skipped = 0
    total_failed  = 0

    #setup logger
    log = ActiveSupport::Logger.new(ADD_USERS_LOG)
    start_time = Time.now.strftime('%H:%M:%S')
    ap "--Starting @ #{start_time}"
    log.info "--Starting @ #{start_time}"


    ARGV.each do |filename_arg|

      book   = Roo::Spreadsheet.open(filename_arg, disable_html_injection: true)
      sheet  = book.sheet(WORKSHEET_WITH_DATA)

      users_skipped = 0
      users_failed  = 0
      users_added   = 0

      sheet.each_with_index do |row, index|
        next if index < SKIP_ROWS



        if [FIRST_NAME_COL, LAST_NAME_COL, EMAIL_COL, ROLES_COL].any? {|col_indx| row[col_indx].empty?  }
          message = "----Error in row ##{index + 1 }: missing mandatory field (first name, last name, email, role) in columns A thru D, skipping row"
          ap message
          log.info message
          users_skipped+=1
          next
        end

        # sanitize row values in place to prevent unnecessary calls/object creation
        row.map! {|cell| AddUsersHelper::sanitize(cell)}
        row[EMAIL_COL].downcase!

        # check if email address already exists
        if User.exists?(email: row[EMAIL_COL].downcase)
          message = "User with email #{row[EMAIL_COL]} (row ##{index + 1}, col C in spreadsheet) already exists, skipping row"
          ap message
          log.info message
          users_skipped+=1
          next
        end

        ap "Processing #{row[EMAIL_COL]} ..."

        # check for presence of roles and inclusion in list of valid roles.
        role_tokens = []
        if AddUsersHelper::ROLES.has_value?(row[ROLES_COL])
          role_tokens << AddUsersHelper::ROLES.key(row[ROLES_COL])
        else
            message = "----Error in row ##{index + 1}: unknown role in column #{ROLES_COL}"
            log.info message
            ap message
            log.info "\t\t\tAllowed roles: #{AddUsersHelper::ROLES}"
            users_skipped+=1
            next
        end


        # check for presence of NPI and inclusion in list of valid sites
        if row[NPI_COL].present?
          row[NPI_COL] = row[NPI_COL].strip.gsub(/[^0-9]/,'')
          if row[NPI_COL].length != NPI_LENGTH
            message = "----Error in row ##{index+1}NPI must be 10 digit number, creating account without NPI (see #{NPI_WEBSITE} for details)"
            ap message
            log.info
          end

          unless role_tokens.include? "non_vha"
            message = "----Error in row ##{index+1}: only non_vha users can have an assigned NPI."
            ap message
            log.info message
            users_skipped+=1
            next
          end
        end


        # IMPORTANT: All passwords must be at least 12 chars long
        new_password = "Cui!#{Array.new(PASSWORD_LENGTH){rand(36).to_s(36)}.join}"

        user = User.create(
          first_name:           row[FIRST_NAME_COL],
          last_name:            row[LAST_NAME_COL],
          email:                row[EMAIL_COL],
          password:             new_password,
          roles:                (role_tokens || []).to_a,
          authorization_state:  'authorized',
          user_preference:      UserPreference.new,
          npi:                  row[NPI_COL].to_i || ""
        )

        begin
          ap "----Attempting to create account for #{user.email}"
          user.save!
          success_message = "----Account created for #{user.email}."
          ap success_message
          log.info success_message
        rescue Exception => e
          log.info "------ERROR: Account creation for #{row[EMAIL_COL]} failed:  #{e}"
          ap "------ERROR creating account, see log exception details"
        end

        # checking for empty cells (which Spreadsheet explicitly maps to nil) or
        # case-insensitive truthy values, and in either case (empty or truthy) we send an email.
        email_flag = row[SEND_EMAIL_FLAG_COL]
        user_app = AddUsersHelper::get_app_long_name_func(User.find_by_email(row[EMAIL_COL]))
        welcome_msg = AddUsersHelper::welcome_email_msg_func(new_password)
        if (email_flag.blank? || ['TRUE', 'YES', 'Y'].include?(email_flag.to_s.upcase))
          UserMailer.welcome_email(user_app,row[EMAIL_COL], welcome_msg).deliver_now
          message = "----Email sent for above account " + row[EMAIL_COL]
          ap message
          log.info message
          message = "------Password: #{new_password}"
          ap message
          log.info message
        end

        users_added += 1
      end
      log.info "----Added #{users_added} users from #{filename_arg}"
      total_added   += users_added
      total_failed  += users_failed
      total_skipped += users_skipped
      add_users_stats_msg = AddUsersHelper::statistics_email_msg_func(filename_arg,users_added,users_failed,users_skipped)
      UserMailer.send_report(ADD_USERS_REGION, add_users_stats_msg).deliver_now
    end

    log.info "Total Users added: #{total_added}"
    ap "Total Users added: #{total_added}"
    log.info "Total Users skipped: #{total_skipped}"
    ap "Total Users skipped: #{total_skipped}"
    log.info "Total Users failed: #{total_failed}"
    ap "Total Users failed: #{total_failed}"
    stop_time = Time.now.strftime('%H:%M:%S')
    log.info "--Stopping @ #{stop_time}"
    ap "--Stopping @ #{stop_time}"
    ap "-See log for more details: #{ADD_USERS_LOG}, statistics emailed to #{ENV['ACCOUNT_MGMT_INBOX']}"



  end

  #TODO move this task into separate file and have it reference AddUsersHelper instead of hardcoding roles
  task generate_users: :environment do

    ARGV.each {|a| task a.to_sym do ; end }
    count = ARGV[1]
    site = ARGV[2]
    roles  = ARGV[3]

    if count == nil or count == "" or site == nil or roles == nil
      puts "Please make sure you are invoking the command as specified below:"
      puts ""
      puts "rake admin:generate_users <number of users> <site> <roles>"
      puts ""
    else
      DEFAULT_AUTHORIZATION_STATE = "authorized".freeze
      ROLES = ["medical_assistant",
             "supervisor",
             "examiner",
             "administrator",
             "quality_assurance",
             "supervisory_clinician",
             "clinician",
             "triage",
             "scheduling",
             "site_user",
             "app_admin",
             "site_admin"]
      site_from_db = Site.find_by_name(site)

      role_tokens = roles.split(",")

      original_role_tokens = role_tokens.clone
      role_tokens.reject! { |n| !AddUsersHelper::ROLES.include?  n.lstrip.rstrip }

      if (original_role_tokens.length < 1 or original_role_tokens.length != role_tokens.length)
        puts "Invalid Roles"
        puts "Please select from the list ::: " + AddUsersHelper::ROLES.to_s
      elsif (site_from_db == nil)
        puts "Invalid site ::: " + site
        puts "Please select one from the list ::: [QTC, VES, VetFed, Atlanta VA Medical Center, Chicago Medical Center]"
      else
        $i = 1
        $num = count.to_i

        File.open('lib/tasks/admin/users.txt', 'w') do |file|
         while $i <= $num  do

          first_name = "FName" + (SecureRandom.random_number(1000)).to_s
          last_name = "LName" + (SecureRandom.random_number(1000)).to_s
          email = "test" + (SecureRandom.random_number(100000)).to_s + "@gmail.com"
          password = "CUI-#{Array.new(6){rand(36).to_s(36)}.join}"

          u = User.find_by_email(email)
          if u.present?
            puts email + " already exists!!!"
          else
             new_user = User.create(
                     :first_name           => first_name,
                     :last_name            => last_name,
                     :email                => email,
                     :password             => password,
                     :roles                => role_tokens,
                     :authorization_state  => DEFAULT_AUTHORIZATION_STATE
                   )

             new_user.add_site(site_from_db)

             site_roles = new_user.get_site_roles(site_from_db)

             site_roles.triage = false
             if (role_tokens.include? 'triage')
               site_roles.triage = true
             end
             site_roles.admin = false
             if (role_tokens.include? 'admin')
               site_roles.admin = true
             end
             site_roles.scheduling = false
             if (role_tokens.include? 'scheduling')
               site_roles.scheduling = true
             end
             site_roles.clinician = false
             if (role_tokens.include? 'clinician')
               site_roles.clinician = true
             end
             site_roles.super_clinician = false
             if (role_tokens.include? 'supervisory_clinician')
               site_roles.super_clinician = true
             end
             site_roles.qa = false
             if (role_tokens.include? 'quality_assurance')
              site_roles.qa = true
             end
             site_roles.save

            puts("Generated and Created...#$i" )

            file.puts first_name + "  "  + last_name + "  " + email + "  " + password

          end
          $i +=1
         end
        end
        puts "**********************************"
        puts "The following file has been generated:"
        puts "Location ::: /lib/tasks/admin"
        puts "File     ::: users.txt "
        puts "**********************************"
      end
    end
  end
end
