# lib/sql_seeder.rb

# Support utility for db/seeds.rb
module SqlSeeder

  # Files found in the db/sql directory can be used in the db:seed process
  SQL_DIR = Rails.root + 'db' + 'sql'
  LOG_DIR = Rails.root + 'log'
  TMP_DIR = Rails.root + 'tmp'

  # Using this method to load seed data from SQL files stored in the SQL_DIR
  # parameter can be a single table name, list of table names or an array of
  # table names.  If a list or an array, the files are loaded in the order that
  # was presented in the parameter.
  #
  # Set reset_only to true if the SQL seeding should only take place
  # when DBRESET is true.
  #
  # Returns true if SQL seeding was attempted
  # Returns false if SQL seeding was NOT attempted

  def seed_from_sql(*klasses, reset_only: false)

    tables = klasses.map {|klass| klass.tableize}

    return false unless reset_only && DBRESET
    table_paths = tables.flatten.map {|table_name| SQL_DIR + "#{table_name}.sql"}

    # if `psql` command is available from command line and all SQL files exist, then
    # Seed data the fast way using psql if:
    # 1. the `psql` command is available from the command line; i.e. `which psql` does not
    #    return an empty string.
    # 2. all the SQL files in table_paths exist in the filesystem.
    avoid_slow_way = (`which psql`.present?) && table_paths.all? {|sql_file| sql_file.exist? }

    if avoid_slow_way
      print "... loading #{table_paths.map{|table| table.basename.to_s}.join(', ')} ... "
      base_command = "psql --single-transaction --host=#{ENV['DBHOST']} --username=#{ENV['DBUSER']} --dbname=#{DATABASE_NAME} --file=" #  #{table_pathname} > /dev/null"

      table_paths.each do |sql_file|
        log_file_path = LOG_DIR + "seed_#{sql_file.basename.to_s.gsub('sql','log')}"
        command = base_command + sql_file.to_s + " > #{log_file_path}"
        system command
      end
      puts "done"
    end
    avoid_slow_way
  end

end # module SqlSeeder
