# seeder.rb
#
# NOTE: *_id is actual record id whereas
#       *_number is just a repeatable sequence to add uniqueness

module Seeder

  CHERYL_HOWARD_NPIS = %w(
      1043303563
      1083089809
      1225182579
      1225340383
      1295124063
      1295862415
      1366499295
      1598199259
      1659696045
      1710342811
    )

  class VeteranSeed

    attr_reader :veteran

    VETERAN_BASE_ATTRS = [:ssn, :first_name, :middle_name, :last_name,
      :other_health_insurance_id, :other_health_insurance_name]

    def initialize(veteran_number, default_attrs = {})
      # separating @veteran ActiveRecord attrs from content attrs
      base_attrs = default_attrs.extract!(*VETERAN_BASE_ATTRS)
      content_attrs = default_attrs

      @veteran = Cpp::Local::Veteran.new base_attrs

      @veteran.ssn          ||= sprintf("3212377%02d", veteran_number)

      @veteran.first_name   ||= veteran_number.even? ? 'Jane' : 'Joe'
      @veteran.middle_name  ||=
        case veteran_number % 3
          when 0
            'Doe'
          when 1
            'Billy'
          when 2
            'Bob'
          else
            'Buddy'
        end

      @veteran.last_name    ||=
        case veteran_number % 3
          when 0
            'Smith'
          when 1
            'Jones'
          when 2
            'vonNeuman'
          else
            'Trump'
        end + ", #{veteran_number}#{veteran_number.ordinal}"

      @veteran.other_health_insurance_id ||= 1 + (veteran_number % OtherHealthInsurance.count)

      # MAGICE_NUMBER: 1 means record id 1 of the OtherHealthInsurance table which is the "Yes" title
      @veteran.other_health_insurance_name ||= (1 == @veteran.other_health_insurance_id) ?
        sprintf("Insurance Provider #%d",veteran_number) : ''

      @veteran.content = content(veteran_number, content_attrs)
      @veteran.save!
    end

    def content(veteran_number, content_attrs = {})
      # TODO: fill in with additional items required for the new views
      #       use static data elements that include the veteran_number.
      #       see name fields as an example
      pn_str = sprintf("%04d", (1000 + (veteran_number * 79)) % 10000)
      return {
        veteran_number: veteran_number,
        gender:         veteran_number.even? ? 'F' : 'M',
        date_of_birth:  sprintf("03/%02d/1953", 1+(veteran_number % 30)),
        address:        ["#{pn_str} Any Street Blvd.", "P.O. Box #{pn_str}"],
        city:           'Columbia',
        state:          'MD',
        zip:            '21046',
        phone:          "240-555-#{pn_str}"
      }.merge(content_attrs)
    end
  end # class VeteranSeed


  class ProviderSeed
    attr_reader :provider
    def initialize(provider_number)
      @provider                 = Provider.new
      @provider.npi             = sprintf("19530603-%04d",        provider_number)
      @provider.name            = sprintf('Provider Name #%04d',  provider_number)
      @provider.physician_name  = sprintf('Physician Name #%04d', provider_number)
      @provider.facility_id     = 1 + (provider_number % Facility.count)
      base_ms = provider_number % (MedicalSpecialty.count-3)

      @provider.medical_specialties << MedicalSpecialty.find(base_ms + 1)
      @provider.medical_specialties << MedicalSpecialty.find(base_ms + 2)
      @provider.medical_specialties << MedicalSpecialty.find(base_ms + 3)
      @provider.content = content(provider_number)
      @provider.save!
    end

    def content(provider_number)
      # TODO: fill in with additional items required for the new views
      #       use static data elements that include the provider_number.
      #       see name fields as an example
      pn_str = sprintf("%04d", provider_number)
      pn_str2 = sprintf("%04d", (rand 10_000))
      {
        provider_number:  provider_number,
        address:          ["#{pn_str} Any Street Blvd.", "P.O. Box #{pn_str}"],
        city:             'Bossier City',
        state:            'LA',
        zip:              '71111',
        phone:            "318-555-#{pn_str}",
        fax:              "202-555-#{pn_str}"
      }
    end
  end # class ProviderSeed

  class FacilitySeed
    attr_reader :facility
    def initialize(facility_number)
      @facility                             = Facility.new
      @facility.name                        = sprintf('Facility Name #%04d',  facility_number)
      @facility.content                     = content(facility_number)
      @facility.visn_id                     = 1 + (facility_number % Visn.count)
      @facility.save
    end


    def content(facility_number)
      pn_str = sprintf("%04d", facility_number)
      {
        facility_number:  facility_number,
        address:          ["#{pn_str} Any Street Blvd.", "P.O. Box #{pn_str}"],
        city:             'Bossier City',
        state:            'LA',
        zip:              '71111',
        phone:            "318-555-#{pn_str}"
      }
    end
  end #class FacilitySeed

  class ConsultationSeed
    attr_reader :consultation
    def initialize(veteran, consultation_number)
      seed = veteran.id * consultation_number
      @consultation                         = Consultation.new
      @consultation.veteran                 = veteran
      # NOTE: Using the mpi_pid and local_pid from spec/support/via_api/match_by_name_city_state_ms_response.xml
      #       so that appointments of that patient will be loaded for any referral when VIA_ACTIVE_FOR_CPP is true.
      @consultation.veteran_mpi_pid         = "1009609054"
      @consultation.veteran_local_pid       = "7177262"
      @consultation.veteran_dob = veteran.content['date_of_birth']
      @consultation.veteran_other_health_insurance_name = veteran.other_health_insurance_name
      @consultation.consultation_number     = sprintf("%02d-%04d", veteran.id, consultation_number)
      @consultation.ordering_provider_id     = 1 + (seed % Provider.count)
      @consultation.care_category_id        = 1 + (seed % CareCategory.count)
      @consultation.consultation_type_id    = 1 + (seed % ConsultationType.count)
      @consultation.ordering_provider_id    = 1 + (seed % Provider.count)
      @consultation.consultation_order_id   = 1 + (seed % ConsultationOrder.count)
      @consultation.consultation_type_id    = 1 + (seed % ConsultationType.count)
      @consultation.consultation_status_id  = 1 + (seed % ConsultationStatus.count)
      # use month range from 0, 1, or 2 months ago as from-to date range
      months_ago_date = Time.now - (consultation_number % 3).months
      @consultation.valid_from              = months_ago_date.at_beginning_of_month
      @consultation.valid_to                = months_ago_date.at_end_of_month
      @consultation.content                 = content(veteran.id, consultation_number)
      @consultation.save!

    end


    def content(veteran_id, consultation_number)
       # TODO: fill in with additional items required for the new views
      return {
        npi:                              '',
        urgent_priority:                  consultation_number.even?,
        ordering_physician_name:          @consultation.ordering_provider.physician_name,
        requesting_provider_telephone:    @consultation.ordering_provider.phone,
        requesting_provider_fax:          @consultation.ordering_provider.fax,
        requesting_provider_contact_name: @consultation.ordering_provider.name
      }
    end
  end # class ConsultationSeed


  class ReferralSeed
    attr_reader :referral
    ICD10_CODES = []

    def initialize(veteran_id, consultation_id, referral_number)
      @coordinator_ids = User.coordinators.all.pluck('id')
      @referral = Referral.new
      @referral.referral_number           = sprintf("%02d-%04d-%04d",       veteran_id, consultation_id, referral_number)
      @referral.authorization_number      = sprintf("xyzzy-%02d-%04d-%04d", veteran_id, consultation_id, referral_number)
      @referral.coordinator_id            = @coordinator_ids[referral_number % @coordinator_ids.size]
      @referral.consultation_id           = consultation_id
      @referral.content                   = content(veteran_id, consultation_id, referral_number)
      @referral.referral_status_id        = 1 + ((consultation_id + referral_number) % ReferralStatus.count)
      @referral.referral_type_id          = 1 + ((consultation_id * referral_number) % ReferralType.count)
      @referral.referral_reason_id        = 1 + ((consultation_id * referral_number) % ReferralReason.count)
      @referral.provider_id               = provider_id(veteran_id, consultation_id, referral_number)
      @referral.save!
    end

    def content(veteran_id, consultation_id, referral_number)
      # populate ICD10_CODES preloaded list if this is getting called for the first time
      if ICD10_CODES.blank?
        DiagnosisCode.all.map {|dc| dc.code }.shuffle.each {|code| ICD10_CODES << code }
      end
      return {
        veteran_id:             veteran_id,
        diagnosis_codes:        ICD10_CODES.slice!(0, 1 + (rand 4)).join(', '),
        procedure_codes:        "X#{1 + (rand 10_000)}",
        length_of_stay:         "#{7 + (rand 14)} days",
        length_of_procedure:    "#{1 + (rand 7)} days",
        treatment_frequency:    random_frequency,
        number_of_visits:       1 + (rand 6),
        cost_estimate:          sprintf("%04.2f", ((rand 990_000).to_f / 100.0) + 100.0)
      }
    end

    def random_frequency
      case (rand 6)
      when 0
        "every day"
      when 1
        "once every #{(rand 4) + 2} days"
      when 2
        "once per week"
      when 3
        "every #{(rand 2) + 2} weeks"
      when 4
        "#{(rand 7) + 2} times per day"
      when 5
        "once per month"
      end
    end

    def provider_id(veteran_id, consultation_id, referral_number)
      cheryl_howard_provider_ids ||= Provider.where(npi: CHERYL_HOWARD_NPIS).pluck(:id)
      case ((veteran_id + consultation_id) % 2)
      when 0
        1 + ((consultation_id * referral_number) % Provider.count)
      when 1
        cheryl_howard_provider_ids[1 + rand(CHERYL_HOWARD_NPIS.length - 1)]
      end
    end

  end # class ReferralSeed


  class ReferralNoteSeed
    attr_reader :referral
    def initialize(veteran_id, consultation_id, referral_id, referral_note_number)
      @referral_note              = ReferralNote.new
      @referral_note.referral_id  = referral_id
      @referral_note.content      = content(veteran_id, consultation_id, referral_id, referral_note_number)
      @referral_note.save!
    end

    def content(veteran_id, consultation_id, referral_id, referral_note_number)
      # TODO: fill in with additional items required for the new views
      {
        veteran_id:             veteran_id,
        consultation_id:        consultation_id,
        referral_id:            referral_id,
        referral_note_number:   referral_note_number,
        user:                   'Joe Bob Billy Boy',
        notes:                  "This is referral_note ##{referral_note_number} " +
                                "for referral ##{referral_id} " +
                                "for consultation ##{consultation_id} " +
                                "for veteran ##{veteran_id} "
      }
    end
  end # class ReferralNoteSeed


  class ReferralAppointmentSeed
    attr_reader :referral

    def initialize(veteran_id, consultation_id, referral_id, referral_appointment_number)
      @referral_appointment                  = ReferralAppointment.new
      @referral_appointment.referral_id      = referral_id
      @referral_appointment.appointment_time = get_appointment_date(veteran_id)

      @referral_appointment.referral_appointment_status_id = referral_appointment_status_id

      @referral_appointment.content = content(veteran_id, consultation_id, referral_id, referral_appointment_number)
      @referral_appointment.save!
    end

    def content(veteran_id, consultation_id, referral_id, referral_appointment_number)
      # TODO: fill in with additional items required for the new views
      {
        veteran_id:             veteran_id,
        consultation_id:        consultation_id,
        referral_id:            referral_id,
        referral_appointment_number:   referral_appointment_number,
        appointment_type:       referral_appointment_number.even? ? "type one" : "type two",
        scheduled_by:           referral_appointment_number.even? ? "Joe Bob, Jr." : "Billy Boy, III",
        book_notes:             "This is referral_appointment ##{referral_appointment_number} " +
                                "for referral ##{referral_id} " +
                                "for consultation ##{consultation_id} " +
                                "for veteran ##{veteran_id} "
      }
    end

    APPOINTMENT_TIME_ROUNDING = 5*60 # 5 minutes

    # get random date between today and 4 weeks from now
    def get_appointment_date(id)
      @@all_dates ||= {}
      @@all_dates[id] ||= []
      @@start_time ||= Time.now.to_i
      @@end_time ||= (Time.now + 28.days).to_i
      t = rand(@@start_time..@@end_time)
      date = round_time(t).utc
      date = get_appointment_date(id) if appointment_conflict?(date, id)
      #recurse till we find a good date.  Theoretically will never end if we build more records than valid appointments across 28 days (per vet). (200 appointments per vet)
      #adding an artificial stop only results in a validation failure.  Mathematically, we will stop, but if we ever don't increase the 28 days parameter above.
      @@all_dates[id] << date
      date
    end

    def appointment_conflict?(time, id)
      before = time - ReferralAppointment::APPOINTMENT_TIME_STEP.minutes
      after = time + ReferralAppointment::APPOINTMENT_TIME_STEP.minutes
      #@@all_dates[id].inject(false) {|result, used_time| result ||= used_time.between?(before, after)}(alternative, can't break)
      @@all_dates[id].each do |used_time|
        return true if used_time.between?(before, after)
      end
      false
    end

     def referral_appointment_status_id
      @referral_appointment_status_count  ||= ReferralAppointmentStatus.count
      (1 + rand(@referral_appointment_status_count - 1))
    end

    def round_time(time)
      Time.at(time.to_i - (time.to_i % ReferralAppointment::APPOINTMENT_TIME_STEP.minutes))
    end

    module ClassMethods
      #intended for use in testing
      #never returns the same date twice
      #start_time and end_time form the boundary (but the top of a 15 minute interval must be present)
      #note the floor is taken so a start time of 10:34 is the same as 10:30, and end_time of 11:56 is 11:45
      #this is across all vets, which should suffice for testing
      def fetch_valid_appointment_time(start_time:, end_time: )
        raise ArgumentError.new("Please provide a valid end time.") unless end_time > start_time
        raise ArgumentError.new("Please provide a time span greater than #{ReferralAppointment::APPOINTMENT_TIME_STEP} minutes") unless end_time - start_time > ReferralAppointment::APPOINTMENT_TIME_STEP.minutes
        @@times ||= Array.new
        start_time = Time.at((start_time.to_f / ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).floor * ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).utc
        end_time = Time.at((end_time.to_f / ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).floor * ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).utc
        current_times = ReferralAppointment.all_within_time(start_time, end_time - start_time).map(&:appointment_time).map(&:to_time)
        iteration = 1
        valid_time = nil
        loop do
          potential_time = start_time + (ReferralAppointment::APPOINTMENT_TIME_STEP*iteration).minutes
          break if potential_time > end_time
          if(!current_times.include?(potential_time) && !@@times.include?(potential_time))
            valid_time = potential_time
            @@times << valid_time
            break
          end
          iteration +=1
        end
        valid_time
      end
    end
    extend ClassMethods
  end # class ReferralAppointmentSeed


  class ReferralApprovalSeed
    attr_reader :referral
    def initialize(veteran_id, consultation_id, referral_id, referral_approval_number)
      @referral_approval              = ReferralApproval.new
      @referral_approval.referral_id  = referral_id
      @referral_approval.content      = content(veteran_id, consultation_id, referral_id, referral_approval_number)
      @referral_approval.save!
    end

    def content(veteran_id, consultation_id, referral_id, referral_approval_number)
      # TODO: fill in with additional items required for the new views
      {
        veteran_id:             veteran_id,
        consultation_id:        consultation_id,
        referral_id:            referral_id,
        referral_approval_number:   referral_approval_number,
        notes:                  "This is referral_approval ##{referral_approval_number} " +
                                "for referral ##{referral_id} " +
                                "for consultation ##{consultation_id} " +
                                "for veteran ##{veteran_id} "
      }
    end
  end # class ReferralApprovalSeed


  class ReferralDocumentSeed
    attr_reader :referral
    def initialize(veteran_id, consultation_id, referral_id, referral_document_number)
      is_approved = referral_document_approved?(referral_id, referral_document_number)
      @referral_document                            = ReferralDocument.new
      @referral_document.referral_id                = referral_id
      @referral_document.referral_document_type_id  = ReferralDocumentType.all.sample.id # select random id
      @referral_document.content                    = content(veteran_id, consultation_id, referral_id, referral_document_number)
      @referral_document.uploader_id                = User.all.sample.id
      @referral_document.approver_id                = is_approved ? User.all.sample.id : nil
      @referral_document.approved_at                = is_approved ? Time.now : nil
      @referral_document.save!
    end

    # set docs as approved based on status and
    def referral_document_approved?(referral_id, referral_document_number)
      referral_code = Referral.find(referral_id).referral_status.code
      if referral_document_number.even?
        return ["ACCEPTED", "COMPLETE"].include? referral_code
      else
        return ["ACCEPTED", "COMPLETE", "REVIEW_PENDING"].include? referral_code
      end
    end

    def content(veteran_id, consultation_id, referral_id, referral_document_number)
      # TODO: fill in with additional items required for the new views
      {
        veteran_id:             veteran_id,
        consultation_id:        consultation_id,
        referral_id:            referral_id,
        referral_document_number:   referral_document_number,
        document_name:          referral_document_number.even? ? "1-10-500-500-100.jpg" : "2016_rubykaigi.pdf",
        document_type:          referral_document_number.even? ? "image/jpeg" : "application/pdf",
        location:               "/fake_docstore/",
        vha_document:           "false",
        metadata:               {
                                  placeholder:  "For future stuff",
                                  notes:        "This is referral_document ##{referral_document_number} " +
                                                "for referral ##{referral_id} " +
                                                "for consultation ##{consultation_id} " +
                                                "for veteran ##{veteran_id} "
                                }
      }
    end
  end # class ReferralDocumentSeed


end # module Seeder
