# lib/ntm/news.rb
# NOTE: supports the /news route
#       also known as the news and training material website (NTM)
#       intent to be dynamic always presenting the user with the
#       most uptodate information

# Used in development and testing
require_relative 'ntm_storage/filesystem_access.rb'

# For production and UAT-level testing
require_relative 'ntm_storage/s3_access.rb'

class News

  # TODO: testing in irb with explicit methods; will add the deligate when done
  # deligate a bunch of stuff to, @storage

  # Used to select which files to process as News articles.
  # NOTE: file_path is a Pathname object.
  FILE_SELECTER = Proc.new {|file_path| '.txt' == file_path.extname.downcase }

  def initialize(user_group)
    @user_group   = user_group

    if NTM_S3_ENABLED
      # Access News articles from AWS S3 (Simple Storage System)
      root_path = Pathname.new('/s3') 
      storage_system = NtmStorage::S3_Access
    else
      # Access News articles from local filesystem
      root_path = Rails.root + 'public'
      storage_system = NtmStorage::FileSystem_Access
    end

    base_path = root_path + ENV['NTM_ROOT_DIR'] + ENV['NTM_NEWS_DIR']

    @storage = storage_system.new(
                @user_group, 
                base_path, 
                News::FILE_SELECTER
              )
   @storage.reverse_keys! # to access most recent date first
  end # def initialize

  def latest
    file_path, file_content = @storage.latest
    return path_to_date(file_path), file_content
  end
  alias :first  :latest
  alias :newest :latest

  def oldest
    file_path, file_content = @storage.oldest
    return path_to_date(file_path), file_content
  end
  alias :last :oldest

  def each_pair(&block)
    @storage.each_pair do |file_path, file_content|
      block.call(path_to_date(file_path), file_content)
    end
  end
  alias :each :each_pair

  def keys
    @storage.keys
  end

  def dates
    @storage.keys.map{|key| key_to_date(key)}
  end

  def [](key)
    @storage[key]
  end

  def path_to_date(a_pathname)
    key = a_pathname.basename.to_s
    key_to_date(key)
  end

  def key_to_date(key)
    DateTime.parse(key.split('.').first)
  end

end # class News
