# db/seeds.rb
# This file is executed during `rake db:reset` and `db:setup` and `db:seed` tasks

PaperTrail.enabled = false

# If this file is executed when the RAILS_ENV is development it will
# only seed the static text files unless the DBRESET environment variable is true
# WHen DBRESET is true and RAILS_ENV is development, the database will have
# been reset to a blank condition prior to this file being executed and the
# this file will populate the entire development dataset.

require 'seeder.rb'

require 'csv_seeder.rb'
include CsvSeeder

require 'sql_seeder.rb'
include SqlSeeder


#########################################################
## seed static text tables from csv files located in db/csv

puts "\nLoading static text tables."

unless CPP_DISABLED  # Turn off CPP features prior to deprecating code
  [
    CareCategory,
    ConsultationOrder,
    ConsultationStatus,
    ConsultationType,
    OtherHealthInsurance,
    ReferralDocumentType,
    ReferralReason,
    ReferralType,
    ReferralAppointmentStatus

  ].each do |model_class_name|
    load_static_text_table_from_csv(klass: model_class_name,
      common_fields: { deleted_at: nil },
      column_mappings: ['sequence','title'],
      use_copy_from: false
    )
  end
end # unless CPP_DISABLED

unless seed_from_sql("Visn", reset_only: false)
  load_static_text_table_from_csv(
    klass:            Visn,
    column_mappings: ['sequence','deleted_at','region','name'],
    use_copy_from: false
  )
end

unless seed_from_sql("Site", reset_only: true)
  load_static_text_table_from_csv(
    klass:            Site,
    column_mappings:  ['name','address','city','state','zip_code','site_station_number'],
    common_fields:    {country: 'USA'},
    use_copy_from:    false
  )
end

unless seed_from_sql("DiagnosisCode", reset_only: true)
  load_static_text_table_from_csv(
    klass:  DiagnosisCode,
    column_mappings: ['version_code','description'],
    use_copy_from:    false
  )
end

unless seed_from_sql( "Clinic", reset_only: true)
  load_static_text_table_from_csv(
    klass:            Clinic,
    column_mappings:  ['site_id','clinic_identifier','name'],
    use_copy_from:    false
  )
end

##########################################################
  ## Create three dbqs for CUI

  puts "... Seeding new dbqs in CUI"

  seed_from_sql(
    "DbqForm",
    "DbqSection",
    "DbqElement",
    reset_only: true
  )

  puts "Done"


### Load issue categories for DOM-127 Request for Support ###
%w[
  Performance
  Security
  Functional
  Access
  Forgot_Password
  Other
].each do |category|
  SupportRequestCategory.find_or_create_by(name: category.gsub('_', ' ').titlecase)
end

org1 = SupportRequestOrganization.create(short_name: "C&P", long_name:"Compensation and Pension")

org2 = SupportRequestOrganization.create(short_name: "CPP", long_name:"Community Provider Portal")

#############################################################


#########################################################
## define the ReferralStatus

unless CPP_DISABLED  # Turn off CPP features prior to deprecating code
  print "Loading static text for ReferralStatus ... "

  [
    {
      status:               "new",
      update_description:   "",
      referral_queue:       "vha_cc",   # Only users who have one of these roles will have this status in the referral_queue
      filterable_by_roles:  "vha_cc"    # Only users who have one of these roles will have this status in the detail filter pane
    },
    {
      status:               "prep",
      update_description:   "The Referral is being prepared for the Community Provider.",
      referral_queue:       "vha_cc",
      filterable_by_roles:  "vha_cc"
    },
    {
      status:               "assigned",
      update_description:   "The Referral contains all medical information needed for the first visit and has been assigned to a Community Provider. <b>Note:</b> The Communtiy Provider will receive a system-generated notification email as soon as the <b>Submit</b> button below is clicked.",
      referral_queue:       "vha_cc, non_vha",
      filterable_by_roles:  "vha_cc, non_vha"
    },
    {
      status:               "accepted",
      update_description:   "The Community Provider has successfully returned medical records for the first appointment. Additional appointments and medical documents are expected.",
      referral_queue:       "non_vha",
      filterable_by_roles:  "vha_cc, non_vha"
    },
    {
      status:               "review_pending",
      update_description:   "The Referral has been sent to the VA for approval.",
      referral_queue:       "vha_cc, non_vha",
      filterable_by_roles:  "vha_cc, non_vha"
    },
    {
      status:               "information_needed",
      update_description:   "The Referral has been reviewed and returned to the Community Provider with a request for additional information.",
      referral_queue:       "vha_cc, non_vha",
      filterable_by_roles:  "vha_cc, non_vha"
    },
    {
      status:               "complete",
      update_description:   "The Referral is complete and all medical document have been received.",
      referral_queue:       "",
      filterable_by_roles:  "vha_cc, non_vha"
    }
  ].each do |entry|
    ReferralStatus.find_or_create_by(
      name:                 entry[:status].gsub('_', ' ').titlecase,
      code:                 entry[:status].upcase,
      update_description:   entry[:update_description],
      referral_queue:       entry[:referral_queue].downcase,
      filterable_by_roles:  entry[:filterable_by_roles].downcase
    )
  end

  puts "done"
end # unless CPP_DISABLED


#########################################################
## define the test ExaminationState

print "Loading static text for ExaminationState ... "

# FIXME: spelling error: Canceled not Cancelled
%w[
  Pending
  Ready_to_be_Scheduled
  Scheduled
  In_Progress
  Completed
  Reviewed
  Submitted
  Rejected
  Pending_Cancellation
  Canceled
  Reschedule
  Pending_Clarification
].each do |status|
  ExaminationState.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

puts "done"


#########################################################
## define the test ExamRequestState

print "Loading static text for ExamRequestState ... "

# FIXME: spelling error Canceled not Cancelled
%w[
  New
  In_Progress
  Pending_Clarification
  Pending_Reported
  Pending_Cancellation
  Canceled
  Canceled_by_MAS
  Canceled_by_RO
  Completed
  Completed_Printed_by_RO
  Released_to_RO_Not_Printed
  Rerouted_New
  Rerouted_Accepted
  Rerouted_Rejected
  Clarification_Received
].each do |status|
  ExamRequestState.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

puts "done"


#########################################################
## define the test ClarificationType

print "Loading static text for ClarificationType ... "

%w[
  Request
  Response
].each do |status|
  ClarificationType.find_or_create_by(name: status.gsub('_', ' ').titlecase, code: status.upcase)
end

puts "done"


#########################################################
## Loading the data from db/ecm_seeds.rb via
## rake eval_con_man:seed
##
## The following models are impacted:
##      DiagnosisModifier
##      Diagnosis
##      DmAssignment
##      Symptom
##      MinorSystem
##      MajorSystem
##      EvaluationTemplate


  print "Invoking rake task eval_con_man:seed ... "
  unless EvaluationTemplate.exists?
    Rake::Task["eval_con_man:seed"].invoke

    puts "done"

    print "Invoking rake task eval_con_man:load_html ..."

    Rake::Task["eval_con_man:load_html"].invoke

    puts "done"
  end

# #########################################################
# ## Loading the data from db/vbms_r_fg_init.rb via
# ## rake vbms_r_fg:init
# ##
# ## The following models are impacted:
# ##      VbmsRFactGroup

 print "Invoking rake task vbms_r_fg:init ... "

 Rake::Task["vbms_r_fg:init"].invoke

 puts "done"

  ###############################################################################
  #### Sites - The below 5 sites were provided earlier in the project
  #### These sites have to be created outside the 'development' test data section
  #### as they are valid sites
  #### (These sites cannot be moved to the sites csv file as they are being used
  #### (See below for details on these sites usage)
  ###############################################################################

  print "Creating the 7 additional sites ... "

  site1 = Site.find_or_create_by(
              name: "Chicago Medical Center",
              address: "1111 Silver Avenue",
              city: "Chicago",
              state: "IL",
              zip_code: "60611",
              country: "USA")

  site2 = Site.find_or_create_by(
              name: "Dallas VA Clinic",
              address: "2222 Rush Avenue",
              city: "Dallas",
              state: "TX",
              zip_code: "75418",
              country: "USA")

  site3 = Site.find_or_create_by(
              name: "QTC",
              site_station_number: "QTC VBA Vendor",
              address: "21700 Copley Dr., Ste 200",
              city: "Diamond Bar",
              state: "CA",
              zip_code: "91765-2219",
              country: "USA")

  site4 = Site.find_or_create_by(
              name: "VES",
              site_station_number: "VES VBA Vendor",
              address: "2707 North Loop W., Suite 1000",
              city: "Houston",
              state: "TX",
              zip_code: "77008",
              country: "USA")

  site5 = Site.find_or_create_by(
              name: "VetFed",
              site_station_number: "VetFed VBA Vendor",
              address: "21700 Copley Dr., Ste 200",
              city: "Diamond Bar",
              state: "CA",
              zip_code: "91765-2219",
              country: "USA")

  site6 = Site.find_or_create_by(
              name: "Atlanta VA Medical Center",
              site_station_number: "508",
              address: "1670 Clairmont Road",
              city: "Decatur",
              state: "GA",
              zip_code: "30033",
              country: "USA")

  site7 = Site.find_or_create_by(
              name: "CSRA ITC",
              site_station_number: "777",
              address: "6300 Texas Ave.",
              city: "Bossier City",
              state: "LA",
              zip_code: "71111",
              country: "USA")

  site_manchester_va = Site.find_by_site_station_number("608")

  # Resets private key sequences on database tables
  ActiveRecord::Base.connection.tables.each do |t|
    ActiveRecord::Base.connection.reset_pk_sequence!(t)
  end

  puts "done"
#########################################################
## make up some test data for development

if Rails.env.development?  &&  DBRESET

  puts "\nLoading development test data."

  # CPP-specific test users
  cheryl_howard_npis = Seeder::CHERYL_HOWARD_NPIS.dup

  #########################################################
  ## define test facilities and providers

  unless CPP_DISABLED  # Turn off CPP features prior to deprecating code
    unless seed_from_sql( "Facility", "Provider", "MedicalSpecialty", "MedicalSpecialtiesProvider",reset_only: true )
      puts "\n... Facilities table."

      50.times do |facility_number|
        facility_record = Seeder::FacilitySeed.new(facility_number)
      end

      puts "\n... Provider table and Medical Specialty table from NPI Registry with zip codes ..."

      # NOTE: loading the cheryl howard NPI records
      #       into the providers table.
      cheryl_howard_npis.each do |npi|
        NpiRegistry.filter(npi: npi)
      end

      zipcodes = %w[20002 23233 71111 73501]

      zipcodes.each do |postal_code|
        printf "%s ", postal_code
        NpiRegistry.filter(postal_code: postal_code)
      end

      puts "done"
    end
  end # unless CPP_DISABLED

  #########################################################
  ## define test users

  puts "... User table."

  medical_assistant = User.create(
              email: "team@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Medical",
              last_name: "Assistant",
              authorization_state: 'authorized',
              roles: ['medical_assistant']) unless User.exists?(email: "team@adhocteam.us")

  supervisor = User.create(
              email: "supervisor@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Super",
              last_name: "Visor",
              authorization_state: 'authorized',
              roles: ['supervisor']) unless User.exists?(email: "supervisor@adhocteam.us")

  examiner = User.create(
              email: "examiner@adhocteam.us",
              password: "Sms!123456789",
              first_name: "Exam",
              last_name: "Iner",
              authorization_state: 'authorized',
              roles: ["examiner"]) unless User.exists?(email: "")

  app_admin = User.create(
              email: "samantha@smith.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Samantha",
              last_name: "Smith",
              # FIXME: at the end of CPP sprint 15 these two lines were removed
              #        from the integration branch.  So if they are truly not needed,
              #        come back here and delete this dead code in a future sprint.
              # authorization_state: 'authorized',
              roles: ["app_admin"]) unless User.exists?(email: "samantha@smith.com")

  site_user1 = User.create(
              email: "mike@wallace.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Mike",
              last_name: "Wallace") unless User.exists?(email: "mike@wallace.com")

  site_user2 = User.create(
              email: "eric@wilson.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Eric",
              last_name: "Wilson") unless User.exists?(email: "eric@wilson.com")

  all_roles_QTC_user = User.create(
              email: "all@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "All",
              last_name: "Qtc") unless User.exists?(email: "all@qtc.com")

  admin_QTC_user = User.create(
              email: "admin@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Admin",
              last_name: "Qtc") unless User.exists?(email: "admin@qtc.com")

  triage_QTC_user = User.create(
              email: "triage@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Triage",
              last_name: "Qtc") unless User.exists?(email: "triage@qtc.com")

  scheduling_QTC_user = User.create(
              email: "scheduling@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Scheduling",
              last_name: "Qtc") unless User.exists?(email: "scheduling@qtc.com")

  clinician_QTC_user = User.create(
              email: "clinician@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Clinician",
              last_name: "Qtc") unless User.exists?(email: "clinician@qtc.com")

  sclinician_QTC_user = User.create(
              email: "super_clinician@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Super",
              last_name: "Qtc") unless User.exists?(email: "super_clinician@qtc.com")

  qa_QTC_user = User.create(
              email: "qa@qtc.com",
              password: "Sms!123456789",
              authorization_state: "authorized",
              first_name: "Qa",
              last_name: "Qtc") unless User.exists?(email: "qa@qtc.com")

  unless CPP_DISABLED
    vha_cc = User.create(
                email: "vha_cc@example.com",
                password: "Sms!123456789",
                first_name: "VHA CC",
                last_name: "TestUser",
                authorization_state: 'authorized',
                roles: ["vha_cc"]) unless User.exists?(email: "vha_cc@example.com")

    non_vha = User.create(
                email: "non_vha@example.com",
                password: "Sms!123456789",
                first_name: "Non VHA",
                last_name: "TestUser",
                authorization_state: 'authorized',
                npi: cheryl_howard_npis.pop(5),
                roles: ["non_vha"]) unless User.exists?(email: "non_vha@example.com")

    vha_cc2 = User.create(
                email: "vhacc.cpptestuser@gmail.com",
                password: "Sms!123456789",
                first_name: "VHA CC2",
                last_name: "TestUser",
                authorization_state: 'authorized',
                roles: ["app_admin","vha_cc"]) unless User.exists?(email: "vhacc.cpptestuser@gmail.com")

    non_vha2 = User.create(
                email: "nonvha.cpptestuser@gmail.com",
                password: "Sms!123456789",
                first_name: "Non VHA2",
                last_name: "TestUser",
                authorization_state: 'authorized',
                npi: cheryl_howard_npis.pop(5),
                roles: ["non_vha"]) unless User.exists?(email: "nonvha.cpptestuser@gmail.com")
  end # unless CPP_DISABLED


  # add empty UserPreference object to each user, with time zone
  # corresponding to index
  US_TIME_ZONES = ActiveSupport::TimeZone.us_zones.map &:name
  User.all.each_with_index do |user, index|
    time_zone_string     = US_TIME_ZONES[index % US_TIME_ZONES.count]
    user.user_preference = UserPreference.new(time_zone: time_zone_string)
    if user.is_vha_cc?
      user.user_preference.vista_duz       = 'mLb+ZHYXCWt7R9WvfFhWHBwdOKeT5ZsuZJOmjLIU8ko='
      user.user_preference.vista_user_name = 'WASH,PAUL E'
      user.user_preference.vista_site_id   = '516'
      user.user_preference.save(validation: false)
    end
    user.save(validation: false)
  end


  #########################################################
  ## Updating test user's site and roles

  puts "... Adding site and site roles to CUI-users."

  app_admin.present? ? (app_admin.add_site(site3); app_admin.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  app_admin.present? ? (app_admin.add_site(site6); app_admin.get_site_roles(site6).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  app_admin.present? ? (app_admin.add_site(site_manchester_va); app_admin.get_site_roles(site_manchester_va).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  site_user1.present? ? (site_user1.add_site(site3); site_user1.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  site_user2.present? ? (site_user2.add_site(site3); site_user2.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  all_roles_QTC_user.present? ? (all_roles_QTC_user.add_site(site3); all_roles_QTC_user.get_site_roles(site3).update(admin: true, triage: true, scheduling: true, clinician: true, super_clinician: true, qa: true)) : false
  admin_QTC_user.present? ? (admin_QTC_user.add_site(site3); admin_QTC_user.get_site_roles(site3).update(admin: true)) : false
  triage_QTC_user.present? ? (triage_QTC_user.add_site(site3); triage_QTC_user.get_site_roles(site3).update(triage: true)) : false
  scheduling_QTC_user.present? ? (scheduling_QTC_user.add_site(site3); scheduling_QTC_user.get_site_roles(site3).update(scheduling: true)) : false
  clinician_QTC_user.present? ? (clinician_QTC_user.add_site(site3); clinician_QTC_user.get_site_roles(site3).update(clinician: true)) : false
  sclinician_QTC_user.present? ? (sclinician_QTC_user.add_site(site3); sclinician_QTC_user.get_site_roles(site3).update(super_clinician: true)) : false
  qa_QTC_user.present? ? (qa_QTC_user.add_site(site3); qa_QTC_user.get_site_roles(site3).update(qa: true)) : false


  #########################################################
  ## Adding the Evaluation specs

  puts "... EvaluationSpec table."

  #########################################################
  ## Seed the Evaluation Specs

  seed_specs_log = File.open(LOG_DIR+'seed_specs.log','w')

  print "... Executing EvaluationSpec.sync_specs ... "
  EvaluationSpec.sync_specs(nil, "app/exams/*.yml", seed_specs_log)
  puts "done"

  seed_specs_log.close

  ##########################################################
  ## Send exam requests to CUI

  puts "... Seeding exam requests in CUI"

  seed_from_sql(
    "Claim",
    "ExamRequest",
    "Contention",
    "ContentionDetail",
    "ContentionsDbqInformation",
    "ContentionObject",
    "RequestObject",
    reset_only: true
  )

  puts "Done"

  unless CPP_DISABLED  # Turn off CPP features prior to deprecating code

    #########################################################
    ## define test Veteran, consultation, referral
    ## TODO: add test data for the sub-referral tables


    puts "\n... Veteran, Consultation, Referral tables."

    RANDOMIZED_VETERAN_COUNT = 10

    # PP-496: adding more test veterans named after famous U.S. Generals who died in WWII
    PII

    TOTAL_VETERAN_COUNT = (RANDOMIZED_VETERAN_COUNT + US_GENERALS.count)

    RANDOMIZED_VETERAN_COUNT.times do |veteran_number|

      puts "Fake Veteran #{veteran_number+1} of #{RANDOMIZED_VETERAN_COUNT}."

      veteran_record  = Seeder::VeteranSeed.new(veteran_number).veteran
      veteran_id      = veteran_record.id
      Referral.skip_callbacks = true
      5.times do |consultation_number|
        consultation_record = Seeder::ConsultationSeed.new(veteran_record, consultation_number).consultation
        consultation_id     = consultation_record.id


        20.times do |referral_number|
          referral_seed_record  = Seeder::ReferralSeed.new(veteran_id, consultation_id, referral_number)
          referral_id           = referral_seed_record.referral.id

          2.times do |referral_note_number|
            Seeder::ReferralNoteSeed.new(veteran_id, consultation_id, referral_id, referral_note_number)
          end

          2.times do |referral_appointment_number|
            Seeder::ReferralAppointmentSeed.new(veteran_id, consultation_id, referral_id, referral_appointment_number)
          end

          2.times do |referral_approval_number|
            Seeder::ReferralApprovalSeed.new(veteran_id, consultation_id, referral_id, referral_approval_number)
          end

          2.times do |referral_document_number|
            Seeder::ReferralDocumentSeed.new(veteran_id, consultation_id, referral_id, referral_document_number)
          end

        end # 20.times do |referral_number|
      end # 5.times do |consultation_number|
    end # RANDOMIZED_VETERAN_COUNT.times do |veteran_number|
    Referral.skip_callbacks = false

    print "Adding World War II heroes with fake SSNs ... "
    US_GENERALS.each_with_index do |us_general_data, usg_index|
      vet_number = RANDOMIZED_VETERAN_COUNT + usg_index + 1
      Seeder::VeteranSeed.new((vet_number), us_general_data)
    end

    puts "Done."

    # Adding customer test accounts for CPP.

    print "Adding customer test accounts ..."
    customer_test_accounts = %w[
      PII                      
      PII                    
      PII              
    ].each do |email|
      name = email.split('@').first.split('.')
      User.create(
                email:                email,
                password:             "Sms!123456789",
                first_name:           name.first,
                last_name:            name.last,
                authorization_state:  'authorized',
                roles:                ["vha_cc"]) unless User.exists?(email: email)
    end

    puts"Done."
  end # unless CPP_DISABLED
end # if Rails.env == "development"
