module DAS
  module ECRUD
    include Faraday

    EVENT_PATH = '/ecrud/v1/core/examManagement'
    DBQ_PATH   = '/ecrud/v1/core/disabilityBenefitsQuestionnaires/transform'
    URL        = ENV['DAS_API_URL']

    class DasEcrudError < RuntimeError; end

    # response contains fault tag 
    class EcrudFaultError < DasEcrudError; end

    class << self
      # Send DBQ to DAS eCRUD
      def send_dbq(dbq_path_and_file_name)
        dbq_send(dbq_path_and_file_name)
      end

      # Send Exam Management Event to DAS eCRUD
      def send_event(event_xml)
        event_send(event_xml)
      end

      private

      # Faraday connection to base URL
      def connection
        Faraday.new(:url => URL) do |builder|
          builder.request :multipart
          builder.response :logger
          builder.adapter Faraday.default_adapter
        end
      end

      # Send DBQ to DAS eCrud
      def dbq_send(dbq_path_and_file_name)
        xml_body = { :file => Faraday::UploadIO.new(dbq_path_and_file_name, 'text/xml') }
        # POST to DBQ_PATH
        response = connection.post DBQ_PATH do |req|
          req.body = xml_body
          req.headers['Content-Type'] = 'multipart/form-data'
        end
        status = response.status
        process_error(status) unless status == 201
        response
      end

      # Send Exam Management Event to DAS eCrud
      def event_send(xml_body)
        # POST to EVENT_PATH
        response = connection.post EVENT_PATH do |req|
          req.body = xml_body
          req.headers['Content-Type'] = 'application/xml'
        end
        status = response.status
        process_error(status) unless status == 201
        response
      end

      def process_error(status)
        msg = 'Unknown DAS service error'
        case status
        when 404
          msg = 'DAS service is shut down'
        when 502
          msg = 'DAS service is timing out'
        when 503
          msg = 'DAS service has changed endpoints'
        when 422
          msg = 'DAS service schema validation failed'
        end
        raise EcrudFaultError.new("DAS Ecrud APU Error: status is #{status} - " + msg)
      end

    end # self
  end # module ECRUD
end # module DAS

