# == Schema Information
#
# Table name: users
#
#  id                     :integer          not null, primary key
#  email                  :string           default(""), not null
#  encrypted_password     :string           default(""), not null
#  reset_password_token   :string
#  reset_password_sent_at :datetime
#  remember_created_at    :datetime
#  sign_in_count          :integer          default(0), not null
#  current_sign_in_at     :datetime
#  last_sign_in_at        :datetime
#  current_sign_in_ip     :inet
#  last_sign_in_ip        :inet
#  created_at             :datetime
#  updated_at             :datetime
#  roles                  :text
#  first_name             :string
#  last_name              :string
#  is_under_review        :boolean          default(TRUE)
#  failed_attempts        :integer          default(0)
#  unlock_token           :string
#  locked_at              :datetime
#  provider               :string
#  uid                    :string
#  authorization_state    :string           default("none")
#  action_token           :string
#

class User < ActiveRecord::Base
  include ActiveModel::Validations

  # For Validations
  ALPHA_NUMERICS=/[0-9A-Za-z,.]/
  DIGITS=/\d/
  LC_LETTERS=/[a-z]/
  SPECIAL_CHARS=/[\$\^\*~!@#%&]/
  UC_LETTERS=/[A-Z]/
  MINIMUM_PASSWORD_LENGTH = 12  # per VA-6500 Appx. F

  # Include default devise modules. Others available are:
  # :confirmable, :lockable, :timeoutable and :omniauthable
  devise  :database_authenticatable,
          :recoverable,
          :rememberable,
          :trackable,
          :validatable,
          :lockable,
          :registerable,
          :timeoutable,
          :omniauthable,
          # :confirmable,
          omniauth_providers: [:saml]

  # FIXME: commented code below should either be fixed or removed.

  # attr_accessible :email, :password, :password_confirmation, :remember_me, :username
  #
  # validates_presence_of :username
  # validates_uniqueness_of :username


  # NPI - National Provider Index - is a string of NPI values with some non-digit separation
  attr_accessor :npi  # only applies to role 'non_vha' users; not part of the user record

  attr_accessor :duz, :via_user_name, :site_id # VIA_API related. Used for storing the duz session key and site id.
  serialize :roles, Array

  has_one :user_preference, dependent: :destroy

  has_many :referrals, foreign_key: 'coordinator_id'
  has_many :evaluations
  has_many :examination_notes, foreign_key: :from_id
  has_many :claims, -> { distinct }, through: :evaluations
  has_many :consultation_comments

  has_many :incomplete_claims, -> {
    where(completed_at: nil).order(created_at: :desc).distinct
  }, through: :evaluations, source: :claim

  has_many :site_role_sets

  has_many :sites, through: :site_role_sets do
    def <<(s)
      self.push(s) unless self.include?(s)
    end
  end

  has_and_belongs_to_many :supervising,
                          class_name:   "User",
                          join_table:   :supervised_clinicians,
                          foreign_key:  :user_id,
                          association_foreign_key: :supervised_id do
    def <<(u)
      # SMELL: why not use an exception?
      return "Use add_supervised_clinician()"
    end

    def delete(u)
      # SMELL: why not use an exception?
      return "Use remove_supervised_clinician()"
    end
  end # has_and_belongs_to_many :supervising,

  has_and_belongs_to_many :supervisors,
                          class_name:   "User",
                          join_table:   :supervising_clinicians,
                          foreign_key:  :user_id,
                          association_foreign_key: :supervisor_id do
    def <<(u)
      # SMELL: why not use and exception?
      return "Use add_supervising_clinician()"
    end

    def delete(u)
      # SMELL: why not use an exception?
      return "Use remove_supervising_clinician()"
    end
  end # has_and_belongs_to_many :supervisors,

  # many-to-many association between Provider and User data records.
  # The "providers" association has no relation to the Omniauth "provider" field
  # that indicates which identity provider (SAML, etc.) we use for authentication.
  has_many :providers_users, dependent: :destroy
  has_many :providers, through: :providers_users

  before_create :set_default_parameters

  after_save do |user|
    user.create_preferences_if_empty
    if user.is_cpp_user?          &&
       (0 == user.sign_in_count)  &&   # TODO: This only applies to creation and not update of NPI
       user.errors.empty?         &&
       !(user.npi.blank?)
      npi_array     = user.npi.split(/[^\d]/).uniq.reject{|entry| entry.empty?}
      provider_ids  = Provider.where(npi: npi_array).pluck(:id)
      provider_ids.each do |provider_id|
        # TODO: During the roles refactor we will add the roles of the
        #       CPP user w/r/t each provider to this table.
        begin
          ProvidersUser.new(user_id: user.id, provider_id: provider_id).save!
        rescue Exception => e
          logger.error "Got exception while adding ProvidersUser record for user/provider (#{user.id}/#{provider_id}) => #{e}"
        end
      end
    end
  end

  # TODO: What about when the user record is updated with changed NPI values?
  validates_with NpiValidator, on: :create, if: :is_non_vha?

  validates_presence_of :first_name, :last_name
  validates :authorization_state, inclusion: {in: ["none", "pending", "rejected", "authorized", nil]}
  validate :password_complexity, if: :password_required?

  scope :coordinators, -> {where("roles like ?", "% vha_cc\n%")}

  scope :by_last_name, -> { order(last_name: :asc, first_name: :asc) }

  # Scopes that duplicate functionality of User#is_cpp_user? and User#is_non_vha?
  # but perform faster due to being run as queries instead of via AR.
  scope :cpp_users, -> { where("roles like ? or roles like ?", "% vha_cc\n%", "% non_vha\n%") }
  scope :cui_users, -> { where.not(id: cpp_users) }

  scope :site_ids, -> (user_id) { find(user_id).sites.pluck(:id) }


  #############################################################
  ###
  ##  All instance methods
  #

  # This web-application has two different personalities/feature-sets
  # defined as CPP and CUI.  The two feature sets do not share the same
  # user community.  This method returns the identifier for the feature set
  # with which the user interacts.
  def app_feature_set
    is_cpp_user? ? 'CPP' : 'CUI'
  end

  def send_devise_notification(notification, *args)
   devise_mailer.send(notification, self, *args).deliver_later
  end


  ##
  #   password-related
  ##

  def password_required?
    super && provider.blank? # NOTE: provider is a SAML provider NOT associated with the Provider model
  end

  def password_complexity
    # if password.present? and not password.match(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)./)
    #   errors.add :password, "must include at least one lowercase letter, one uppercase letter, and one digit"
    # end
    if password.present? && has_minimum_length? && has_uppercase_letters? && has_digits? &&
      has_special_chars? && has_downcase_letters? && has_only_permitted_characters?
      true
    else
      errors.add :password, "must be at least 12 characters long, must include at least one lowercase letter, " \
        "one uppercase letter, one digit and one special character from the " \
        "following: ~!@#$%^&*"
      false
    end
  end

  def update_with_password(params, *options)
    if encrypted_password.blank?
      update_attributes(params, *options)
    else
      super
    end
  end


  ##
  #   decorators
  ##

  def name
    "#{first_name} #{last_name}"
  end


  def last_name_first_name
    "#{last_name}, #{first_name}"
  end


  def human_cpp_rolename
    is_vha_cc? ? 'VHA User' : 'Non VHA Provider'
  end


  def npi_list
    (providers || []).pluck(:npi).join(', ')
  end



  ##
  #   do stuff to a user record
  ##


  # Safe way to add new roles to user (prevents roles array from
  # getting duplicate or typo values). Should always use this method.
  # FIXME: refactoring of the roles subsystem will remove risk of rypos.

  VALID_MAGIC_USER_ROLES =  %w[ site_user
                                app_admin
                                vha_cc
                                non_vha
                              ]

  def add_role(role)
    return false if VALID_MAGIC_USER_ROLES.exclude?(role)
    roles << role if roles.exclude?(role)
  end # def add_role(role)


  def create_preferences_if_empty
    create_user_preference if user_preference.blank?
  end


  # Adds a clinician to be supervised. Also ensures the current user is added to that clinician's supervisors.
  # SMELL:  What is 'c' a record id?  a name?
  def add_supervised_clinician(c)
    # SMELL: why is there such an obsession with "self" ??  its not necessary.
    if self.supervising.exclude?(c)
      self.supervising.push c
      c.supervisors.push self
    end
  end


  # Removes a clinician from the group the current user is supervising.
  # Also ensure the current user is removed from that clinician's supervisors.
  def remove_supervised_clinician(c)
    self.supervising.destroy(c)
    c.supervisors.destroy(self)
  end


  # Add a clinician to group of supervisors for current user.
  # Also ensure that that clinician is now supervising the current user.
  def add_supervising_clinician(c)
    if self.supervisors.exclude? c
      self.supervisors.push c
      c.supervising.push self
    end
  end


  # Remove a clinician from the group of supervisors for current user.
  # Also ensure that current user is removed from that clinician's supervised group.
  def remove_supervising_clinician(c)
    self.supervisors.destroy(c)
    c.supervising.destroy(self)
  end


  # Returns 'Ready for Review' exams that have a clinician who is supervised by
  # the current user at the site of the exam.
  def my_supervised_exams (all_ready_for_review_exams)
    supervised_exams = []
    all_ready_for_review_exams.each do |exam|
      clinician_of_exam = User.find(exam.clinician)
      my_supervised_clinicians_at_exam_site = self.supervising.includes(:sites)
                                                              .where(:sites => {id: exam.site_id})
      # We must check that we supervise the exam's clinician at the exam site. If we supervise
      # them at a different site, the exam should *not* be returned for us to view.

      if my_supervised_clinicians_at_exam_site.include? clinician_of_exam
        supervised_exams << exam
      end
    end
    return supervised_exams
  end


  ##
  #   Role-based conditional queries
  ##

  # builds all of the is_<role>? instance methods

  ROLE_TESTING_METHOD_NAME_REGEX = /is_(.*)\?/

  def method_missing(method_sym, *arguments, &block)
    method_name = method_sym.to_s
    if method_name =~ ROLE_TESTING_METHOD_NAME_REGEX
      role = method_name.match(ROLE_TESTING_METHOD_NAME_REGEX)[1]
      instance_eval <<~EOM
        def #{method_name}
          has_role?('#{role}')
        end
      EOM
      send(method_name)
    else
      super
    end
  end # def method_missing(method_name, *arguments, &block)


  def is_vha?
    !is_non_vha?
  end


  def is_cpp_user?
    is_non_vha? || is_vha_cc?
  end

  def is_cui_user?
    !is_cpp_user?
  end

  #new user is one who has been granted access to the app, but has not been associated with sites/roles
  def is_new_user?
    is_authorized? && site_role_sets.none?
  end


  def has_role?(role)
    self.roles.include?(role)
  end


  # This is an OR conditional not an AND conditional
  # returns true if the user has any of the specified roles
  def has_roles?(roles)
    self.roles.any?{|role| roles.include?(role) }
  end


  ##
  #   attribute-based conditional queries
  ##

  def has_active_alerts?
    self.is_app_admin? && Alert.exists?(:active => true)
  end


  def has_no_sites?
    # SMELL: seems to me that if the count is zero it does not matter whether the user is an admin or not
    if self.sites.count == 0 && self.is_admin? == false
      return true
    else
      return false
    end
  end


  def is_authorized?
    # SMELL: "self." is not necessary
    #        could be simplified to just return('authorized' == authorization_state)
    if self.authorization_state && self.authorization_state == "authorized"
      return true
    else
      return false
    end
  end



  ##
  #   site oriented
  ##


  def is_site_admin? (site)
    if site
      site_roles = self.get_site_roles(site)
      if site_roles.admin == true
        return true
      else
        return false
      end
    else
      return false
    end
  end


  def add_site(site)
    self.sites << site
    self.save
  end


  def remove_site(site)
    self.sites.delete(site)
  end


  # List all sites for "CUI Users" display
  def list_sites
    sites.pluck(:name).join " • "
  end


  # Get the roles that this user has on a given site
  def get_site_roles(site)
    return self.site_role_sets.find_by_site_id(site.id)
  end


  def get_roles_string(site)
    self.get_site_roles(site).try(:get_roles_string) || ''
  end


  def has_this_role_on_any_site?(role)
    self.sites.each do |s|
      roles = self.get_site_roles(s).get_roles_string
      if roles && roles.include?(" < " + role + " > ")
        return true
      end
    end
    return false
  end


  #############################################################
  ###
  ##  All class methods
  #

  class << self

    def old_from_omniauth(auth)
      where(provider: auth.provider, uid: auth.uid).first_or_create do |user|
        puts "from_omniauth"
        puts user
        # NOTE: setting Onmiauth identity provider here, NOT community or VA provider
        user.provider = auth.provider
        user.uid = auth.uid
        user.first_name = auth["info"]["first_name"]
        user.last_name = auth["info"]["last_name"]
        user.email = auth["extra"]["raw_info"].attributes["adEmail"]

        # SMELL: Is all of this debug stuff still needed?
        puts "*****************************************"
        puts "from_omniauth"
        puts auth

        puts "**** auth.info"
        puts auth.info
        puts auth["info"]
        puts auth["info"]["first_name"]

        puts "**** auth.credentials"
        puts auth.credentials
        puts auth["credentials"]

        puts "**** auth.extra"
        puts auth.extra
        puts auth["extra"]
        puts auth["extra"]["raw_info"]
        puts auth["extra"]["raw_info"].attributes["adEmail"]

        puts "*****************************************"
      end
    end

    def from_omniauth(response_attrs)
      email = response_attrs['adEmail'].downcase
      user_with_email = where(email: email).first
      if user_with_email.present?
        # DO NOTHING!!!
      else
        # Create the user account
        first_name = response_attrs['firstName']
        last_name = response_attrs['lastName']
        password = "Sms!123456789"
        user_with_email = User.create(first_name: first_name,
                                      last_name:  last_name,
                                      email:      email,
                                      password:   password,
                                      roles:      ['site_user'])
      end
      return user_with_email
    end

    def new_with_session(params, session)
      if session["devise.user_attributes"]
        new(session["devise.user_attributes"], without_protection: true) do |user|
          user.attributes = params
          # SMELL: method invoked with no action upon the returned value
          user.valid?
          # SMELL: is this debug stuff still needed?
          puts "*****************************************"
          puts "new_with_session"
          puts params
          puts "*****************************************"
        end
      else
        super
      end
    end


    # Temp Method: Similar to User::from_omniauth but takes a OneLogin::RubySaml::Attributes
    # parameter and performs an email-based search based on the email and UUID parameters passed in.
    # A User record with the same email address must already exist in the system in
    # order to be validated, and the UUID field must either match or be empty if the user
    # logs in for the first time -- in which case the UUID will get set to the incoming value.

    def from_idme(response_attrs)
      user_with_email = where(email: response_attrs['email']).first
      # on first-time login only: set User#uid field with ID.me UUID from response.
      if user_with_email.present? && user_with_email.uid.nil?
        user_with_email.uid = response_attrs['uuid']
        user_with_email.save!
      end
      return user_with_email if (user_with_email.uid == response_attrs['uuid'])
    end


    # pretty sure we can't define this as a scope, due to the serialization of
    # roles. Furthermore, postgres can't natively handle YAML, so we have to load
    # all users and filter in memory, unfortunately.

    def examiners
      # FIXME: self is User; so we can just start this line with "all."
      User.all.order(last_name: :asc, first_name: :asc).select { |u| !u.roles.index("examiner").nil? }
    end

    def get_triage_users_by_site(site_id)
      User.joins(:site_role_sets => :site)
          .where(sites: {id: site_id})
          .where(site_role_sets: {triage: true})
    end


    def get_clinicians_by_site(site_id)
      User.joins(:sites)
          .where(sites: {id: site_id})
          .where(site_role_sets: {clinician: true})
    end


    def get_all_clinicians
      User.joins(:site_role_sets => :site)
          .where(site_role_sets: {clinician: true})
    end

    def to_csv
      str = 'last_name first_name email last_sign_in_at last_sign_in_ip' \
            ' roles created_at updated_at is_under_review provider'
      attributes = str.split(' ')

      CSV.generate(headers: true) do |csv|
        csv << attributes

        all.each do |user|
          csv << attributes.map{ |attr| user.send(attr) }
        end
      end
    end
  end # class << self

  #############################################################
  ###
  ##  All private methods
  #

  private

# FIXME: dead code needs to be removed.

#  def set_default_parameters
#    if ENV.has_key? "CUI_DISABLE_UNDER_REVIEW"
#      self.roles = ["medical_assistant", "examiner"]
#      self.is_under_review = false
#    end
#
#    # need this so that we don't return false
#    nil
#  end
# UC_LETTERS=/[A-Z]/
# LC_LETTERS=/[a-z]/
# DIGITS=/\d/
# SPECIAL_CHARS=/[\$\^\*~!@#%&]/
# ALPHA_NUMERICS=/[0-9A-Za-z,.]/


  def set_default_parameters
    self.add_role("site_user")

    # SMELL: nil is conditionally the same as false.
    # need this so that we don't return false
    nil
  end

  def has_uppercase_letters?
    password.match(UC_LETTERS) ? true : false
  end

  def has_digits?
    password.match(DIGITS) ? true : false
    end

  def has_special_chars?
    password.match(SPECIAL_CHARS) ? true : false
  end

  def has_downcase_letters?
    password.match(LC_LETTERS) ? true : false
  end

  def has_only_permitted_characters?
    working_str = password.dup

    special_chars = working_str.scan(SPECIAL_CHARS)
    special_chars.each do |x|
      working_str.gsub!("#{x}", '')
    end

    working_str.gsub!(ALPHA_NUMERICS, '')

    working_str.length == 0
  end

  def has_minimum_length?
    password.length >= MINIMUM_PASSWORD_LENGTH ? true : false
  end
end
