# == Schema Information
#
# Table name: referral_documents
#
#  id                        :integer          not null, primary key
#  referral_document_type_id :integer
#  uploader_id               :integer
#  approver_id               :integer
#  approved_at               :datetime
#  content                   :json
#  created_at                :datetime         not null
#  updated_at                :datetime         not null
#  referral_id               :integer
#  deleted_at                :datetime
#

class ReferralDocument < ActiveRecord::Base
  acts_as_paranoid
  has_paper_trail

  belongs_to :referral
  belongs_to :referral_document_type
  belongs_to :approver, class_name: 'User', foreign_key: 'approver_id', validate: true
  belongs_to :uploader, class_name: 'User', foreign_key: 'uploader_id', validate: true

  validates_presence_of   :uploader_id, :document_name, :doc_type_name
  autowire_content_fields :document_name, :location, :document_type, :referral_document_number, :metadata, :deleted_reason, :deleted_by, :vha_document

  def file_link
    # TODO: Enable storage on DAS
    # if DAS_ACTIVE_FOR_CUI
      # result = documentURN provided by DAS
    # else
    File.join((location || ''), '/', (document_name || ''))
    # end
  end

  def doc_type_name
    referral_document_type.try :title
  end

  def uploader_name
    uploader.try :name
  end

  def uploader_is_vha?
    uploader.try :is_vha?
  end

  def approver_name
    approver.try :name
  end

  def approved?
    approved_at.present? and approver.present?
  end

  def referral_status
    referral.referral_status
  end

  # Return :delete_own_documents if document is yours, you are not a VHA user, and the document is unapproved
  # Return nil if no aforementioned conditions holds true
  def visible_action_links(current_user)
    ret = ''

    # if the current user is NON-VA and is the uploader then allow them to delete the document unapproved documents
    if current_user.is_non_vha? && uploader == current_user && !approved?
      ret = :delete_own_documents
    end

    # allow VA care coordinators to delete any non-approved document
    if current_user.is_vha_cc?
      ret = :delete_own_documents
    end

    ret
  end

  # Rules for determining document visibility, as per PP-245:
  #
  # When VHA Coordinators and VHA Leads upload new documents, and the referral state is:
  # [Active, Pending] -> The new documents can only be seen by other VHA users.
  #
  # Once the referral status (for VHA-uploaded docs) is changed to:
  # [Scheduled, Review Pending, Request Information, Complete, Complete No Visits] ->
  #   The newly uploaded documents can be seen by Community Providers.
  #
  # When Community Providers upload documents, and the referral status is:
  # [Scheduled, Request Information] -> The documents cannot be seen by VHA users. (self only)
  #
  # (Non-VHA) Once the referral status (for non-VHA-uploaded docs) is changed to:
  # [Review Pending] -> The newly uploaded documents can be seen by VHA Coordinators and VHA Leads.
  #
  # In both scenarios, older documents that have already been through the approval cycle are visible
  # to all who have permission to open the referral.
  def visible?(current_user)
    return true if approved?
    if uploader_is_vha?
      return current_user.is_vha? || referral_status.visible_for_non_vha?
    else
      if referral_status.vha_valid_for_approve?
        return true
      else
        return current_user.is_non_vha?
      end
    end
  end

end
