# == Schema Information
#
# Table name: referral_appointments
#
#  id               :integer          not null, primary key
#  content          :json
#  created_at       :datetime         not null
#  updated_at       :datetime         not null
#  referral_id      :integer
#  appointment_time :datetime
#  added_to_cprs_at :datetime
#  added_to_cprs_id :integer
#

class ReferralAppointment < ActiveRecord::Base
  has_paper_trail
  include AuditTrail

  belongs_to :referral
  belongs_to :added_to_cprs_user, class_name: 'User', foreign_key: 'added_to_cprs_id', validate: true
  belongs_to :referral_appointment_status, validate: true

  has_many :referral_notes

  validates_presence_of :appointment_type, message: "Please add a appointment description"
  validates_presence_of :appointment_time, message: "Please select a date and time for the appointment."

  validate :valid_appointment_time?, :valid_referral_provider?

  autowire_content_fields :appointment_type, :scheduled_by, :book_notes, :purpose, :appointment_end_time, :veteran_id

  scope :needs_reminder, -> {where("appointment_time >= ? AND appointment_time <= ?",
                      (Time.now - 3.days).beginning_of_day, (Time.now - 1.days).end_of_day) }
  scope :all_by_vet, ->(veteran_ssn){joins(referral: :consultation).where('veteran_ssn = ?', veteran_ssn)}
  scope :all_by_scheduler, ->(scheduler) {where("content ->>'scheduled_by' = ?", scheduler.to_s)}
  scope :all_within_time, ->(start_time, interval) {where("appointment_time > ? AND appointment_time < ?", (start_time - interval), (start_time + interval))}

  after_save do
    if (content_changed? || appointment_time_changed? || referral_appointment_status_id_changed?) && skip_callbacks != true
      notes_changes = humanize_the_id_field_values.values.last
      time = notes_changes["fields_changed"]["appointment_time"].present? ? notes_changes["fields_changed"]["appointment_time"].last : self['appointment_time']
      type = notes_changes["fields_changed"]["appointment_type"].present? ? notes_changes["fields_changed"]["appointment_type"].last : self['content']['appointment_type']
      appointment_status = notes_changes["fields_changed"]["referral_appointment_status_id"].present? ? get_appointment_status(notes_changes["fields_changed"]["referral_appointment_status_id"].last) : self.appointment_status
      ReferralNote.create(user: notes_changes["changed_by"], notes: "#{time} #{type} with #{self.referral.provider.name} - #{appointment_status}", referral: self.referral)
    end
  end

  APPOINTMENT_TIME_STEP = 15
  APPOINTMENT_TIME_FORMAT = '%m/%d/%Y %H:%M %p'
  VET_TIME_CONFLICT_MESSAGE = 'the selected appointment time conflicts with another appointment for this veteran.  Please select another time.'.freeze

  def added_to_cprs?
    added_to_cprs_at.present?
  end

  def added_to_cprs_username
    added_to_cprs_user.try! :name
  end

  def appointment_status
    referral_appointment_status.title if referral_appointment_status.present?
  end

  def get_appointment_status(referral_appointment_status_id)
    referral_appointment_status.title
  end

  def vista_appointment?
    false
  end

  # returns an array of fields that appear in 'Details' section of appointment datatable
  def details
    [
       "Scheduled By: " + scheduled_by,
      "Specialty: " + referral.provider.medical_specialties.pluck(:title).join(', ')
    ]
  end
  protected

  def valid_appointment_time?
  return unless appointment_time
    valid_vet = true
    veteran_ssn = self.referral.consultation.veteran_ssn rescue nil
    if veteran_ssn
      vet_conflicts = ReferralAppointment.all_within_time(appointment_time, ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).all_by_vet(veteran_ssn).to_a +
                      ReferralAppointment.all_within_time(appointment_time, ReferralAppointment::APPOINTMENT_TIME_STEP.minutes).select{|ra| ra.referral.consultation.veteran_ssn.eql? veteran_ssn}
      valid_vet = vet_conflicts.reject {|ra| self.id.eql?(ra.id)}.empty?
      errors.add('Appointment time is not valid,', VET_TIME_CONFLICT_MESSAGE) unless valid_vet

    else
      #if we don't have a vet we cannot have a conflict.
    end
    valid_vet
  end

  def valid_referral_provider?
    return true unless referral.provider_id.nil?
    errors.add(:base, "Please select a Provider on the Referral Details tab before creating an appointment.")
    false
  end

end # class ReferralAppointment < ActiveRecord::Base
#load('./app/models/referral_appointment.rb')