# == Schema Information
#
# Table name: examinations
#
#  id                                              :integer          not null, primary key
#  title                                           :string
#  exam_date                                       :datetime
#  state                                           :integer
#  exam_id                                         :integer
#  clinician                                       :integer
#  created_at                                      :datetime         not null
#  updated_at                                      :datetime         not null
#  site_id                                         :integer
#  examination_state_id                            :integer
#  active                                          :boolean
#  examfile                                        :string
#  exam_request_id                                 :integer
#  purpose                                         :string
#  evaluation_id                                   :integer
#  examination_state_start_date                    :datetime
#  reviewed_at                                     :datetime
#  reviewed_by                                     :integer
#  interview                                       :text
#  plan                                            :text
#  examfiles                                       :string           default([]), is an Array
#  claim_id                                        :integer
#  cancellation_reason                             :text
#  identifier                                      :string
#  acknowledged                                    :boolean
#  clinician_acknowledged                          :boolean
#

class Examination < ActiveRecord::Base
  belongs_to :examination_state
  belongs_to :site
  belongs_to :claim
  belongs_to :parent, class_name: 'ExamRequest', foreign_key: 'exam_request_id'
  belongs_to :clinician_user, class_name: 'User', foreign_key: 'clinician'
  belongs_to :exam_request

  delegate :name, :code, to: :examination_state
  delegate :participant_id, to: :claim
  delegate :request_date, to: :parent
  delegate :file_number, :edipi, :date_of_birth, :first_name, :last_name, :gender, to: :claim

  # has_one :evaluation, dependent: :destroy
  has_one :full_body_evaluation, dependent: :destroy
  has_one :examination_review_questionnaire
  has_and_belongs_to_many :contentions

  has_many :examination_schedules
  has_many :examination_notes
  has_many :dbq_forms

  after_create :build_associated_evaluation

  accepts_nested_attributes_for :examination_review_questionnaire

  # before_validation :create_guid

  # attr_accessor :examfile
  mount_uploaders :examfiles, ExamfileUploader
  validate :examfiles_size

  scope :scheduled_site_exams, -> (current_user) { joins(:site => :site_role_sets).joins(:examination_state)
                                                    .where(examination_states: {code: ['SCHEDULED']})
                                                    .where(site_role_sets: {user_id: current_user.id, scheduling: true}) }

  RECORD_FIELD_LABEL_MAP = {
    :reviewed_not_indicated => "Not indicated",
    :reviewed_military_service_treatment_records => "Military service treatment records",
    :reviewed_military_service_personnel_records => "Military service personnel records",
    :reviewed_military_enlistment_examination => "Military enlistment examination",
    :reviewed_military_separation_examination => "Military separation examination",
    :reviewed_military_post_deployment_questionnaire => "Military post-deployment questionnaire",
    :reviewed_dod_form_214_separation_documents => "Department of Defense Form 214 Separation Documents",
    :reviewed_vha_treatment_records => "Veterans Health Administration medical records (VA treatment records)",
    :reviewed_civilian_medical_records => "Civilian medical records",
    :reviewed_interviews_with_collateral_witnesses => "Interviews with collateral witnesses (family and others who have known the veteran before and after military service)"
  }

  FIELDS_FOR_RESULTSET = {
    0 => 'examination_schedules.exam_date_time',
    1 => ['claims.first_name', 'claims.last_name'],
    2 => 'contention_details.name',
    3 => 'examination_states.name',
    5 => 'sites.name',
    6 => 'users.name'
  }

  class << self
    def cancel_filter(params={}, current_user)

      query = Examination.scheduled_site_exams(current_user)
                .joins(:claim).joins(:contentions => :contention_details).joins(:examination_state)
                .joins(:site).joins(:clinician_user)
                .joins(:exam_request).joins(:examination_schedules).where(examination_schedules: {active: true})

      return query
    end
  end

  def assign_a_clinician(clinician_user_id)
    self.clinician = clinician_user_id
   # evaluation.user_id = clinician_user_id
   # evaluation.save
  end

  def set_the_creating_user(creator_user_id)
    # if evaluation.present?
    #   evaluation[:assigner_id] = creator_user_id
    #   evaluation.save
    if full_body_evaluation.present?
      full_body_evaluation[:assigner_id] = creator_user_id
      full_body_evaluation.save
    end
  end

  def reviewed?
    reviewed_at.nil? ? false : true
  end

  def review!(reviewed_by = nil)
    return if self.reviewed?
    self.update_attributes(reviewed_at: Time.now, reviewed_by: reviewed_by)
  end

  def one_or_more_reviewed_checkboxes_selected?
    RECORD_FIELD_LABEL_MAP.keys.any? { |k| self[k] }
  end

  def already_submitted?
    !EvaluationLog.where(evaluation_id: self.full_body_evaluation.id, message: "VLER-DAS evaluation submission").empty?
  end

  ##
  #   decorators
  ##

  def clinician_name
    clinician_user.name
  end

  def most_recent_active_appointment
    examination_schedules.where(active: true).last
  end

  def contention_descriptions
    contentions.map { |con| con.contention_description }
  end

  private

  # def create_guid
  #   self.guid = SecureRandom.uuid
  # end

  def build_associated_evaluation
    contention = contentions.first
    claim = contention.claim
    evaluation_spec = find_evaluation_spec

    evaluation = FullBodyEvaluation.new(evaluation_spec: evaluation_spec, claim: claim, examination: self)

    if evaluation.save(:validate => false)
      contention.evaluations << evaluation
      # Should this be self.full_body_evaluation?
      # It seems that the association changed but the evaluation_id db column remains
      self.evaluation_id = evaluation.id
    end
    evaluation.set_highlighting_based_on_associated_dbqs

    self.save(:validate => false)

    true
      # Always return true in callbacks as the normal 'continue' state
      # Assumes that the default_profile can **always** be created.
      # or
      # Check the validation of the evaluation. If it is not valid, then
      # return false from the callback. Best to use a before_validation
      # if doing this. View code should check the errors of the child.
      # Or add the child's errors to the User model's error array of the :base
      # error itemend

  end

  def find_evaluation_spec
    all_evaluation_specs = EvaluationSpec.records_by_body_system
    evaluation_spec = all_evaluation_specs.fetch("Fullbody").first
    evaluation_spec
  end

  def examfiles_size
    size_array = self.examfiles.collect { |e| e.size }
    if size_array.inject(0, :+) > 125.megabytes
      self.errors.add(:examfiles, "combined file size should be less than 125MB")
    end
  end
end
