class VBMSJob < ActiveJob::Base
  queue_as :vbms

  class DBLogger
    attr_reader :evaluation_id

    def initialize(evaluation_id)
      @evaluation_id = evaluation_id
    end

    def log(event, data)
      if event == :decrypted_message
        message = "connect_vbms decrypted response"
        request_body = ""
        response_body = data[:decrypted_data]
      elsif event == :request
        message = "connect_vbms request received status #{data[:response_code]}"
        request_body = data[:request_body]
        response_body = data[:response_body]
      elsif event == :unencrypted_request
        message = "Unencrypted request body: #{data[:unencrypted_body]}"
        request_body = ""
        response_body = ""
      else
        message = "connect_vbms logged unknown event #{event} with data #{data}"
      end

      EvaluationLog.create(message: message,
                           submitted_data: request_body,
                           response_body: response_body,
                           evaluation_id: @evaluation_id)
    end
  end

  def perform(evaluation)
    if evaluation.complete?
      pdf = evaluation.to_pdf
      output_to_temp_file(pdf, evaluation.temp_pdf_path)
      send_to_vbms(evaluation)
    else
      EvaluationLog.create(submitted_data: nil,
                           response_body: nil,
                           evaluation_id: evaluation.id,
                           message: "Evaluation was incomplete and could not be sent")
    end
  end

  private

  def output_to_temp_file(pdf, file_path)
    Dir.mkdir(File.dirname(file_path)) unless Dir.exists? File.dirname(file_path)
    File.open(file_path, "wb") do |f|
      f.write(pdf)
    end
  end

  def send_to_vbms(evaluation)
    env = ENV['VBMS_UPLOADER_ENV'] || "test"
    received_date = Time.now.utc
    subject = "VA Disability Assessment #{evaluation.evaluation_spec.evaluation_builder_title}"

    args = [
      evaluation.claim.file_number,
      received_date,
      evaluation.claim.first_name,
      evaluation.claim.middle_name,
      evaluation.claim.last_name,
      subject,
      evaluation.temp_pdf_path,
      "356", # Mary Kate Alber told us via email that the doctype should be "C&P Exam",
             # and a getDocumentTypes call shows this as the proper docType
      "VHA_CUI", # source
      true,      # new_mail
    ]

    client = VBMS::Client.from_env_vars(logger: DBLogger.new(evaluation.id), env_name: env)
    request = VBMS::Requests::UploadDocumentWithAssociations.new(*args)

    begin
      doc = client.send(request)

    # Handle errors in client.send by logging them and re-raising them
    rescue VBMS::HTTPError => e
      EvaluationLog.create(message: "VBMS error code #{e.code}",
                           response_body: e.body,
                           evaluation_id: evaluation.id
                          )
      raise
    rescue VBMS::ExecutionError => e
      EvaluationLog.create(message: "Execution error, unable to submit evaluation",
                           submitted_data: "Error executing command `#{e.cmd}`",
                           response_body: e.output,
                           evaluation_id: evaluation.id
                          )
      raise
    rescue VBMS::ClientError => e
      EvaluationLog.create(message: "VBMS error #{e.class}: #{e.message}",
                           evaluation_id: evaluation.id)
      raise
    rescue => e
      EvaluationLog.create(message: "VBMS threw unexpected exception #{e.class}: #{e.message}",
                           evaluation_id: evaluation.id)
      raise
    end

    EvaluationLog.create(submitted_data: args.inspect,
                         response_body: doc.to_s,
                         evaluation_id: evaluation.id,
                         message: "VBMS evaluation submission")
    doc
  end
end
