module ExamResultsHelper

  class ExamResultsHelperError < RuntimeError; end
  class ExamResultsValidationError < RuntimeError; end

  class << self

  def build_exam_results exam_id
    ActiveRecord::Base.transaction do
      @exam = Examination.find(exam_id)

      create_contention_dbq_hash

      @exam.dbq_forms.each do |dbq|
        build_exam_result(dbq)
        Rails.logger.debug "exam results xml"
        Rails.logger.debug @doc.to_s
        validate_exam_result_xml
        write_exam_result_to_file(dbq)
      end
    end
  end

  private

  def build_exam_result(dbq)
    path = File.join(Rails.root.to_s, 'app', 'xml', 'templates', 'dbq.xml')
    @doc = Nokogiri::XML(File.open(path))

    @doc.at_xpath('//cld:DocumentTypeVersion').content = '1.1.2'

    claim_id = @doc.at_xpath('//cld:ClaimID').content = @exam.file_number

    build_common_data(dbq)
    build_attachments(dbq)
  end

  def build_common_data(dbq)
    build_document(dbq)
    build_exam_detail
    build_client
    build_service_provider
    build_persons
    build_facility
  end

  def build_document(dbq)
    document_node = @doc.at_xpath('//nc:Document')
    document_node.at_xpath('//nc:DateTime').content = DateTime.now.strftime("%Y-%m-%dT%H:%M:%SZ")
    document_node.at_xpath('//nc:DocumentDescriptionText').content = dbq.title

    document_reference_id = "#{SecureRandom.hex(4)}-#{SecureRandom.hex(2)}-#{SecureRandom.hex(2)}-#{SecureRandom.hex(6)}"
    document_node.at_xpath('//nc:IdentificationID').content = document_reference_id
    contention = Contention.find(@con_dbq_hash[dbq.title.parameterize('_')])
    contention.document_reference_ids.create(efolder_identifier: document_reference_id)

    document_node.at_xpath('//nc:DocumentTitleText').content = dbq.title
  end

  def build_exam_detail
    exam_detail_node = @doc.at_xpath('//vler:ExamDetail')
    exam_detail_node.at_xpath('//vler:ExamId').content = @exam.exam_request.exam_request_uuid
    exam_detail_node.at_xpath('//vler:StatusDate//nc:DateTime').content = DateTime.now.strftime("%Y-%m-%dT%H:%M:%SZ")
  end

  def build_client
    @doc.at_xpath('//vler:ClientIdentifier//nc:IdentificationID').content = @exam.try(:edipi).to_s
    @doc.at_xpath('//vler:ClientIdentifier//vler:AssigningAuthority').content = '2.16.840.1.113883.3.42.10001.100001.12'
  end

  def build_service_provider
    # Adding in test data for ID and assigning authority
    @doc.at_xpath('//vler:ServiceProvider//nc:IdentificationID').content = "12345887"
    @doc.at_xpath('//vler:ServiceProvider//vler:AssigningAuthority').content = "2.16.840.1.113883.3.42.10001.100001.12"
    @doc.at_xpath('//nc:IdentificationJurisdictionText').content = "TBD"
  end

  def build_persons
    # build physican section
    @doc.at_xpath('//nc:PersonGivenName').content = @exam.clinician_user.first_name
    @doc.at_xpath('//nc:PersonSurName').content = @exam.clinician_user.last_name

    # build veteran section
    @doc.at_xpath('//nc:PersonBirthDate//nc:Date').content = @exam.date_of_birth.strftime("%Y-%m-%d")
    @doc.xpath('//nc:PersonName//nc:PersonGivenName').last.content = @exam.first_name
    @doc.xpath('//nc:PersonName//nc:PersonSurName').last.content = @exam.last_name
    @doc.at_xpath('//nc:Person//nc:PersonSexCode').content = @exam.gender[0]
    @doc.at_xpath('//nc:Person//nc:PersonSSNIdentification//nc:IdentificationID').content = "123123123"
  end

  def build_facility
    @doc.at_xpath('//nc:FacilityIdentification//nc:IdentificationID').content = @exam.site.site_station_number.to_s
    @doc.at_xpath('//nc:FacilityIdentification//nc:IdentificationJurisdictionText').content = @exam.site.name.to_s
    @doc.at_xpath('//nc:FacilityName').content = @exam.site.name.to_s
  end

  # Modify to include dbq pdf
  def build_attachments(dbq)
    build_dbq_attachment dbq
    build_examfile_attachments dbq
  end

  def build_dbq_attachment dbq
    dbq_pdf_name = "#{@exam.file_number}_" + "#{@exam.most_recent_active_appointment.exam_date_time.strftime('%Y%m%d')}_" + "#{dbq.title.parameterize('_')}" + ".pdf"
    dbq_pdf_location = Rails.root.to_s + '/files/dbq_pdfs/'
    if !File.file?(dbq_pdf_location + dbq_pdf_name)
      raise ExamResultsHelperError.new("Completed DBQ pdf, #{dbq_pdf_name}, was not found. Cannot continue building exam results.")
    end

    add_new_attachment_section
    fill_dbq_details(dbq, dbq_pdf_location + dbq_pdf_name)
  end

  def build_examfile_attachments dbq
    if !@exam.examfiles.empty? && @exam.examfiles_dbqs.empty?
      raise ExamResultsHelperError.new("Uploaded exam files are not associated to DBQs. Cannot continue building exam results.")
    end
    @exam.examfiles_dbqs.each_with_index do |dbq_id, index|
      if dbq.id == dbq_id
        build_attachment(@exam.examfiles[index], dbq)
      end
    end
  end

  def build_attachment(examfile, dbq)
    add_new_attachment_section
    fill_examfile_details(examfile)
  end

  def fill_dbq_details dbq, dbq_full_path_filename
    attachments_element = @doc.at_xpath("//cld:Attachments")

    attachments_element.xpath('//nc:BinaryBase64Object').last.content = Base64.encode64(File.read(dbq_full_path_filename))
    attachments_element.xpath('//nc:BinaryDescriptionText').last.content = dbq.title
    attachments_element.xpath('//nc:BinaryLocationURI').last.content = dbq_full_path_filename
    attachments_element.xpath('//nc:BinarySizeValue').last.content = "1"
    attachments_element.xpath('//nc:BinaryCategoryText').last.content = "DBQ"
  end

  def add_new_attachment_section
    new_doc = Nokogiri::XML(File.open("app/xml/templates/exam_results_attachment.xml"))
    attachments_element = @doc.at_xpath("//cld:Attachments")
    if attachments_element.children.empty?
      attachments_element.add_child(new_doc.root)
    else
      attachments_element.last_element_child.after(new_doc.root)
    end
  end

  def fill_examfile_details examfile
    attachments_element = @doc.at_xpath("//cld:Attachments")
    new_attachment = attachments_element.last_element_child

    new_attachment.xpath('//nc:BinaryBase64Object').last.content = Base64.encode64(File.read(examfile.file.file))
    new_attachment.xpath('//nc:BinaryDescriptionText').last.content = "None provided"
    new_attachment.xpath('//nc:BinaryLocationURI').last.content = examfile.file.file.split("/").last
    new_attachment.xpath('//nc:BinarySizeValue').last.content = "1"
    new_attachment.xpath('//nc:BinaryCategoryText').last.content = "Not available"
  end

  def validate_exam_result_xml
    xsd_path = Rails.root.join("app", "xsd", "Disability_Benefit_Questionnaire_Claims", "exchange", "Claim-1.1.2.xsd")
    xsd = Nokogiri::XML::Schema(File.open(xsd_path))

    errors = xsd.validate(@doc).map { |error| error.message }
    if !errors.empty?
      error_str = errors.join("\n")
      raise ExamResultsValidationError.new("Exam Result XML is not compatible with DBQ IEPD 1.1.2.final\n"\
                                              "#{errors.length} validation error(s):\n"\
                                              "#{error_str}")
    end
  end

  def write_exam_result_to_file dbq
    filename = "#{@exam.file_number}_" + "#{@exam.most_recent_active_appointment.exam_date_time.strftime('%Y%m%d')}_" + "#{dbq.title.parameterize('_')}" +".xml"
    FileUtils.mkdir_p Rails.root.to_s + '/files/exam_results/'
    File.open(Rails.root.to_s + "/files/exam_results/#{filename}", "w") do |fh|
      fh.write(@doc.to_xml)
    end 
    Rails.root.to_s + "/files/exam_results/#{filename}"
  end

  def create_contention_dbq_hash
    @con_dbq_hash = {}
    @exam.contentions.each do |con|
      con.dbq_informations.each do |dbq|
        @con_dbq_hash[dbq.title.parameterize('_')] = con.id
      end
    end
  end

end
end