class MinorSystemsController < ApplicationController
  include DataTablePaginator

  before_action :set_minor_system,
    only: [:show, :edit, :update, :destroy, :diagnoses_sort,
           :general_question_sort, :symptoms_sort, :sort]

  # GET /minor_systems
  # GET /minor_systems.json
  def index
     minor_system_to_fields_func = -> (ms) do
      {
        id:                       ms.id,
        name:                     ms.name.try(:humanize),
        row_label:                ms.label.try(:humanize),
        edit:                     edit_minor_system_path(ms),
        destroy:                  minor_system_path(ms)
      }
    end
    ordered_minor_systems = MinorSystem.all.order('name asc')
    @return_data = datatables_json_for_query_using_func(
      ordered_minor_systems, minor_system_to_fields_func
    ).to_json
  end

  # GET /minor_systems/1
  # GET /minor_systems/1.json
  def show
  end

  # GET /minor_systems/new
  def new
    @minor_system = MinorSystem.new
  end

  # GET /minor_systems/1/edit
  def edit
    @minor_system = MinorSystem.find(params[:id])
    if params[:delete_diagnosis_id].present?
      d = @minor_system.diagnoses.find_by_id(params[:delete_diagnosis_id])
      @minor_system.diagnoses = @minor_system.diagnoses.reject{|diag| diag == d}
      flash.now[:notice] = "Diagnosis successfully removed."
    elsif params[:delete_symptom_id].present?
      s = @minor_system.symptoms.find_by_id(params[:delete_symptom_id])
      @minor_system.symptoms = @minor_system.symptoms.reject{|symp| symp == s}
      flash.now[:notice] = "Symptom successfully removed."
    elsif params[:delete_general_question_id].present?
      q = @minor_system.general_questions.find_by_id(params[:delete_general_question_id])
      @minor_system.general_questions = @minor_system.general_questions.reject{|gquest| gquest == q}
      flash.now[:notice] = "General Question successfully removed."
    end
    set_diagnoses_not_yet_added @minor_system
    set_symptoms_not_yet_added @minor_system
    set_general_questions_not_yet_added @minor_system
    @generated_html = @minor_system.build_html()
  end

  def generated_html
    @minor_system = MinorSystem.find(params[:minor_system_id])
    @generated_html = @minor_system.build_html()
    render partial: 'evaluation_templates/displayed_result'
  end

  # POST /minor_systems
  # POST /minor_systems.json
  def create
    @minor_system = MinorSystem.new(minor_system_params)

    respond_to do |format|
      if @minor_system.save
        @minor_system.update_attribute(:position, @minor_system.id)
        format.html { redirect_to @minor_system, notice: 'Minor system was successfully created.' }
        format.json { render action: 'show', status: :created, location: @minor_system }
      else
        format.html { render action: 'new' }
        format.json { render json: @minor_system.errors, status: :unprocessable_entity }
      end
    end
  end

  # PATCH/PUT /minor_systems/1
  # PATCH/PUT /minor_systems/1.json
  def update
    if params[:identifier].present?
      redirect_to edit_minor_system_path(@minor_system, :identifier => params[:identifier])
    else
      respond_to do |format|
        if @minor_system.update(minor_system_params)
          if params[:minor_system] && params[:minor_system][:diagnosis_ids]
            params[:minor_system][:diagnosis_ids].each do |d|
              diagnosis = Diagnosis.find_by_id(d)
              if diagnosis
                @minor_system.diagnoses << diagnosis
              end
            end
          end
          if params[:minor_system] && params[:minor_system][:symptom_ids]
            params[:minor_system][:symptom_ids].each do |s|
              symptom = Symptom.find_by_id(s)
              if symptom
                @minor_system.symptoms << symptom
              end
            end
          end
          if params[:minor_system] && params[:minor_system][:general_question_ids]
            params[:minor_system][:general_question_ids].each do |q|
              general_question = GeneralQuestion.find_by_id(q)
              if general_question
                @minor_system.general_questions << general_question
              end
            end
          end
          @minor_system.save
          format.html { redirect_to edit_minor_system_path(@minor_system), notice: 'Minor system successfully updated.' }
          format.json { head :no_content }
        else
          set_diagnoses_not_yet_added @minor_system
          set_symptoms_not_yet_added @minor_system
          set_general_questions_not_yet_added @minor_system
          @generated_html = @minor_system.build_html
          format.html { render action: 'edit' }
          format.json { render json: @minor_system.errors, status: :unprocessable_entity }
        end
      end
    end
  end

  # DELETE /minor_systems/1
  # DELETE /minor_systems/1.json
  def destroy
    @minor_system.destroy
    respond_to do |format|
      format.html { redirect_to minor_systems_url }
      format.json { head :no_content }
    end
  end

  def sort
    params[:minor_system].each_with_index do |id, index|
      MinorSystem.where(id: id).update_all({position: index+1})
    end
    render nothing: true
  end

  def diagnoses_sort
    params[:diagnosis].each_with_index do |id, index|
      @minor_system.diagnoses.where(id: id).update_all({position: index+1})
    end
    render nothing: true
  end

  def symptoms_sort
    params[:symptom].each_with_index do |id, index|
      @minor_system.symptoms.where(id: id).update_all({position: index+1})
    end
    render nothing: true
  end


  def general_question_sort
    params[:general_question].each_with_index do |id, index|
      @minor_system.general_questions.where(id: id).update_all({position: index+1})
    end
    render nothing: true
  end
  private
    # Use callbacks to share common setup or constraints between actions.
    def set_minor_system
      @minor_system = MinorSystem.find(params[:id])
    end

    # Never trust parameters from the scary internet, only allow the white list through.
    def minor_system_params
      params.require(:minor_system).permit(:name, :label, :position)
    end

    def set_diagnoses_not_yet_added minor_system
      @diagnoses_not_yet_added = Diagnosis.all.order("name ASC")
      minor_system.diagnoses.each do |d|
        @diagnoses_not_yet_added = @diagnoses_not_yet_added.reject{|diag| diag == d}
      end
    end

    def set_symptoms_not_yet_added minor_system
      @symptoms_not_yet_added = Symptom.all.order("name ASC")
      minor_system.symptoms.each do |s|
        @symptoms_not_yet_added = @symptoms_not_yet_added.reject{|symp| symp == s}
      end
    end

    def set_general_questions_not_yet_added minor_system
      @general_questions_not_yet_added = GeneralQuestion.all.order("name ASC")
      minor_system.general_questions.each do |q|
        @general_questions_not_yet_added = @general_questions_not_yet_added.reject{|gquest| gquest == q}
      end
    end

end
