class ConsultationsController < ApplicationController
  include DataTablePaginator
  include ApplicationHelper
  include ReferralAppointmentsHelper

  before_action :set_consultation, only: [:show, :edit, :update, :destroy, :add_referral, :add_referral_appointment]

  # The before_action methods below are nested; methods should only call one action below.
  before_action :set_consultation_prefs, only: :index
  before_action :set_filtered_consultations, only: :paged_list
  before_action :set_filtered_consultations_and_user_prefs, only: :search
  before_action :set_referral_and_appointment, only: [:update_referral_appointment, :add_to_cprs]

  # hook up authorization and user permissions
  check_authorization
  authorize_resource

  FILE_UPLOAD_DIR = 'fake_docstore'
  FILE_UPLOAD_PATH = File.join(Rails.root, "public/#{FILE_UPLOAD_DIR}")

  # GET /consultations
  # GET /consultations.json
  def index
    set_global_counts
    @consultation_comment = ConsultationComment.new
    # setting pre-loaded dropdown options before index page is rendered
    if @consultation_filter['facility_id'].blank?
      @facilities = Facility.none
    else
      @facilities = Visn.find(@consultation_filter["visn_id"]).facilities
    end
  end

  # GET /consultations/1
  # GET /consultations/1.json
  def show
    @consultation_comment = ConsultationComment.new
    @referral_note        = @referral.referral_notes.new
    @referral_appointment = ReferralAppointment.new

    cpp_appointments = @consultation.referral_appointments

    if VIA_ACTIVE_FOR_CPP

      # current_veteran = @consultation.veteran(session[:vista])

      # NOTE: This struct is added to avoid the mpi_lookup call in current_veteran as it does not work in pre-prod.
      current_veteran = Struct.new(:mpi_pid, :local_pid).new(@consultation.veteran_mpi_pid, @consultation.veteran_local_pid)

      vista_appointments = VIA_API::EmrService.get_appointments(current_veteran,
        session[:vista], {future_only: true, active_status_only: true})
      @all_appointments = (cpp_appointments + vista_appointments).sort_by(&:appointment_time)
    else
      @all_appointments = cpp_appointments.sort_by(&:appointment_time)
    end

    unique_referral_ids = @all_appointments.map{|appt| appt.referral_id}.uniq
    referrals = Referral.where('id IN (?)',unique_referral_ids )
    @referral_details = {}
    referrals.each do |referral|
      @referral_details[referral.id] = "#{referral.consultation.care_category.title} <br /> #{referral.referral_type.title} <br /> #{referral.authorization_number}".html_safe
    end
    @referral_details
  end

  def add_consultation_comment
    comment = ConsultationComment.new(consultation_comment_create_params)
    comment.user = current_user
      if comment.save
        comment_data = comment.as_json
        comment_data['comment_creator'] = comment.user.name
        comment_data['formatted_date'] = formatted_datetime(comment.created_at, current_user)
        render json: comment_data.to_json
      else
        render json: { validation_errors: comment.errors.full_messages }
      end
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def consultation_comment_create_params
    params.permit(:comments, :consultation_id)
  end

  #Method adds referral notes in the history tab
  def add_referral_note
    @referral = Referral.find(params[:referral_note][:referral_id])
    @referral_note = @referral.referral_notes.new(referral_note_params)
    @referral_note.user = current_user.name
    respond_to do |format|
      if @referral_note.save
        format.html do
          flash[:notice] = 'Referral note was successfully created.'
          @consultation = Consultation.find(params[:consultation_id])
          redirect_to consultation_path(@consultation, tabpage: 'history')
        end
        format.json { render json: @consultation, status: :created }
      else
        format.html do
          flash[:alert] = 'Text field cannot be empty'
          @consultation = Consultation.find(params[:consultation_id])
          redirect_to consultation_path(@consultation, tabpage: 'history')
        end
        format.json { render json: @consultation.errors, status: :unprocessable_entity }
      end
    end
  end

  # GET /consultations/new
  def new
    @consultation = Consultation.new(
        veteran_first_name: params[:first_name],
        veteran_last_name: params[:last_name],
        veteran_middle_name: params[:middle_name],
        veteran_ssn: params[:ssn],
        veteran_mpi_pid: params[:mpi_pid],
        veteran_local_pid: params[:local_pid],
        consultation_number: params[:consult_id],
        consultation_status_id: ConsultationStatus.find_by_title('New').id,
        consultation_title: params[:consult_name],
        veteran_dob: params[:dob],
        via_created_date: params[:consult_created_at],
        consultation_text: params[:consult_text],
        veteran_other_health_insurance_name: params[:other_health_insurance_name])
    @veteran_ssn = nil

    # GET /referrals/new
    @referral = Referral.new
  end

  # POST /consultations
  # POST /consultations.json
  def create
    clean_params = consultation_params.merge!(ordering_provider_id: 1)
    @consultation = Consultation.new(clean_params)
    @consultation = urgent_priority(@consultation)
    if !clean_params[:valid_from].blank?
      begin
        @consultation[:valid_from]  = DateTime.strptime(clean_params[:valid_from],'%m/%d/%Y')
        @consultation[:valid_to]    = DateTime.strptime(clean_params[:valid_to],'%m/%d/%Y')
      rescue
      end
    end
    respond_to do |format|
      if @consultation.save
        format.html { redirect_to @consultation, notice: 'A new consultation was successfully created.' }
        format.json { render action: 'show', status: :created, location: @consultation }
      else
        @consultation.valid_from = @consultation.valid_from.to_date.strftime('%m/%d/%Y') if @consultation.valid_from
        @consultation.valid_to = @consultation.valid_to.to_date.strftime('%m/%d/%Y') if @consultation.valid_to
        @veteran_ssn = params[:veteran_ssn]
        format.html { render action: 'new', veteran_ssn: @veteran_ssn }
        format.json { render json: @consultation.errors, status: :unprocessable_entity }
      end
    end
  end

  def upload_vha_doc
    @consultation = Consultation.find(params[:id])
    if(referral_document_params[:document_name].nil? or referral_document_params[:document_name].empty?)
      flash[:alert] = "No Document attached"
      redirect_to @consultation
    else
      move_file_to_uploads_dir
      @referral = Referral.find(referral_document_params[:referral_id])
      @referral_document = @referral.referral_documents.new(referral_document_params)
      @referral_document.location = "/#{FILE_UPLOAD_DIR}"
      @referral_document.uploader_id = current_user.id
      if current_user.is_vha_cc?
        @referral_document.approved_at = Time.now
      end
      respond_to do |format|
        if @referral.save
          format.html do
            flash[:notice] = 'Document was successfully created.'
            redirect_to @consultation
          end
          format.json {render json: @referral, status: :created}
        else
          format.html do
            flash[:alert] = @referral.errors.full_messages.join(" ")
            redirect_to @consultation
          end
          format.json {render json: @referral.errors, status: :unprocessable_entity}
        end
      end
    end
  end

  # add new referral to consultation with basic params (referral typem auth #, provider)
  # and preset params: arandomized referral number, status set to Pending, and default reason ID
  def add_referral
    clean_params = referral_params.merge!(content: {}, referral_number: (rand 999), referral_status_id: 1, referral_reason_id: 1)
    referral = Referral.where(authorization_number: clean_params[:authorization_number])
    if referral.empty? && !(clean_params[:referral_type_id].empty?)
      new_referral = @consultation.referrals.create(clean_params) if referral.empty?

      respond_to do |format|
        if new_referral
          auth_num = new_referral.authorization_number
          success_notice = "Referral with authorization number #{auth_num} was successfully created."
          format.html do
            auth_num = new_referral.authorization_number
            redirect_to @consultation, notice: success_notice
          end
          format.json do
            provider_data = ProviderDataPresenter.new(new_referral)
            new_referral_data = {
              referral_type: new_referral.referral_type.title,
              authorization_number: new_referral.authorization_number,
              coordinator_name: new_referral.coordinator.name,
              provider_name: provider_data.name,
              date: new_referral.created_at.strftime('%D'),
              referral_status: new_referral.referral_status.name
            }
            render json: {data: new_referral_data, message: success_notice, id: new_referral.id}, status: :created
          end
        else
          format.html {redirect_to @consultation, error: @consultation.errors}
          format.json {render json: {error: @consultation.errors}, status: :unprocessable_entity}
        end
      end
    elsif clean_params[:referral_type_id].empty?
      respond_to do |format|
        format.html {redirect_to @consultation, error: "Please select Referral type"}
        format.json {render json: {error: "Please select Referral type"}, status: :unprocessable_entity}
      end
    elsif !(referral.empty?)
      respond_to do |format|
        format.html {redirect_to @consultation, error: "The Authorization number entered is already in use. Please enter a unique Authorization Number."}
        format.json {render json: {error: "The Authorization number entered is already in use. Please enter a unique Authorization Number."}, status: :unprocessable_entity}
      end
    end
  end


  # PATCH/PUT /consultations/1
  # PATCH/PUT /consultations/1.json
  def update
    formatted_params = consultation_params
    formatted_params = format_update_parameters(formatted_params)
    respond_to do |format|
      if @consultation.update(formatted_params)
        format.html {redirect_to @consultation, notice: 'Consultation was successfully updated.'}
        format.json {head :no_content}
      else
        format.html {render action: 'show'}
        format.json {render json: @consultation.errors, status: :unprocessable_entity}
      end
    end
  end


  # DELETE /consultations/1
  # DELETE /consultations/1.json
  def destroy
    @consultation.destroy
    respond_to do |format|
      format.html {redirect_to consultations_url}
      format.json {head :no_content}
    end
  end

  def reset_filter
    save_to_user_preference({})
    render json: {success: @saved}
  end

  def search
    render json: {success: @saved}
  end


  def export_consultations
    respond_to do |format|
      format.csv {send_data Consultation.to_csv(current_user.user_preference.consultation_filter)}
    end
  end

  # Returns a paginated list of Consultation records, based on DataTables
  # parameters passed in.  Params keys include:
  # :start - the offset when querying users in the database
  # :length - the number of records being queried
  # :draw - DataTables internal counter userd to sort async responses
  def paged_list
    consultation_to_fields_func = -> (consultation) do
      {
        id:                 consultation.id,
        patient_name:       consultation.veteran_full_name,
        ssn:                consultation.formatted_ssn,
        episode_of_care:    consultation.care_category.title,
        status:             consultation.status_name,
        ordering_physician: ProviderDataPresenter.new(consultation).name,
        date:               consultation.created_at.strftime(DATE_FORMAT)
      }
    end
    ordered_consultations = apply_ordering_to_query(
      @filtered_consultations, Consultation::FIELDS_FOR_RESULTSET, params['order']
    )
    render json: datatables_json_for_query_using_func(
      ordered_consultations, consultation_to_fields_func
    )
  end

  # Ajax method to retrieve Ordering Physician autocomplete options.
  # Search term is passed in as 'term' param
  def physician_select
    providers = Provider.none
    if param_present?(:term)
      if param_present?(:facility_id)
        providers = Facility.find(params['facility_id'])
                            .providers
                            .where("providers.physician_name ilike ?", "#{params['term']}%")
      elsif param_present?(:visn_id)
        providers = Provider.filter visn_id: params['visn_id'], physician_name: params['term']
      end
    end

    # returning values as JSON-ified (id, name) key-value pairs.
    # NOTE: performing pluck/uniq because providers table in Postgres doesn't
    # like distinct() and pluck is faster for small numbers of columns.
    render json: providers.order('providers.name asc')
                     .limit(50)
                     .pluck('providers.id, providers.physician_name')
                     .uniq()
                     .map {|arr| {id: arr[0], name: arr[1]}}
                     .to_json
  end

  # If VIA_ACTIVE_FOR_CPP: Search VISTA for consultations to be created locally.
  # Otherwise, create a temporary consult with "New User" as name, and user can
  #   fill out the other fields.
  # Returns JSON with format: {"data": [<list of Consultation json objects>] }
  def get_patient_consult_details
    merged_consultations = []
    consult_details = []
    temp_consult = Consultation.new(veteran_ssn: params[:ssn])
    veteran = temp_consult.veteran(session[:vista])

    # If Veteran lookup with Vista fails then the ViaApiError is added to the consultation errors and nil is returned.
    vet_error = temp_consult.errors.messages[:via_api_error] if veteran.nil?

    begin
      if veteran.present? && vet_error.blank?
        if VIA_ACTIVE_FOR_CPP
          consult_details = VIA_API::EmrService.get_consults_for_patient(veteran, session[:vista])
        else
          consult_details = [{
            id: veteran.generate_consultation_number,
            date_of_birth: veteran.date_of_birth,
            city: '',
            zip: '',
            timestamp: Time.now.strftime(TIMESTAMP_FORMAT),
            title: '',
            text: "Consultation created in CPP for patient: #{veteran.last_name}, #{veteran.first_name}"
          }]
        end

        consult_details.each do |consult_detail|
          # hashifying veteran and consult_detail, and merging the two hashes
          merged_consultations << JSON.parse(veteran.to_json).merge(JSON.parse(consult_detail.to_json))
        end

        render json: {data: merged_consultations, status: :ok}, status: :ok, message: :success
      elsif vet_error.present?
        render json: {error: vet_error, status: :error}, status: :unprocessable_entity
      else
        render json: {error: "Veteran with SSN #{params[:ssn]} not found.", status: :error}, status: :unprocessable_entity
      end

    rescue VIA_API::ViaApiError => e
      render json: {error: e.to_s, status: :error}, status: :unprocessable_entity

    end

  end

  def add_referral_appointment
    ra_params_hash = convert_appointment_time(referral_appointment_params.to_h)
    @referral = Referral.find(params[:referral_appointment][:referral_id])
    @referral_appointment = @referral.referral_appointments.new(ra_params_hash)
    respond_to do |format|
      if @referral_appointment.save
        format.html do
          flash[:notice] = 'Appointment was successfully created.'
          if ReferralAppointmentStatus::ADD_TO_CPRS_STATUS == @referral_appointment.referral_appointment_status.title
            add_to_cprs
          else
            redirect_to consultation_path(@consultation, tabpage: 'appointments')
          end
        end
        format.json { render json: @referral_appointment, status: :created }
      else
        if Referral.find(params[:referral_appointment][:referral_id]).provider.blank?
          @referral_appointment.errors.clear
          @referral_appointment.errors.add(:base, "Please select a Provider on the Referral Details tab before creating an appointment.")
        end
        format.html do
          flash[:alert] = @referral_appointment.errors.messages.map do |e|
            e.last.map do |e|
              e.split('.').map(&:strip).map(&:capitalize).join('. ')
            end
          end.join(", ")
          redirect_to consultation_path(@consultation, tabpage: 'appointments')
        end
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end

  # Method called via Ajax call that edits appointment data in database.
  # If VIA_ACTIVE_FOR_CPP and record was previously saved (Approved), then appointment
  # also needs to be cancelled and re-scheduled.
  def update_referral_appointment
    request_body = JSON.parse(request.body.read)

    attributes_hash = convert_appointment_time({
      'appointment_type' => request_body['table_appointment_description'],
      'scheduled_by' => current_user.name,
      'appointment_time' => request_body['table_appointment_time'],
      'referral_appointment_status_id' => request_body['table_referral_appointment_status_id']
    })

    @referral_appointment.assign_attributes(attributes_hash)

    if @referral_appointment.valid?
      if VIA_ACTIVE_FOR_CPP && @referral_appointment.added_to_cprs?
        response_hash = cancel_appointment_in_vista
        notice_text = (response_hash[:error_message] if response_hash[:error])
      else
        response_hash = {}
      end

      respond_to do |format|
        format.json do
          if (not response_hash[:error]) && @referral_appointment.save
            if (ReferralAppointmentStatus::ADD_TO_CPRS_STATUS == @referral_appointment.referral_appointment_status.title)
              add_to_cprs
            else
              render json: @referral_appointment, status: :no_content
            end
          else
            render json: (notice_text || @referral_appointment.errors), status: :unprocessable_entity
          end
        end
      end
    else
      render json: @referral_appointment.errors.full_messages.join(', '), status: :unprocessable_entity
    end
  end

  def add_to_cprs
    # first make appointment in VISTA and set added flags only if that operation succeeds.
    if VIA_ACTIVE_FOR_CPP
      response_hash = make_appointment_in_vista
      notice_text = (response_hash[:error_message] if response_hash[:error])

      @referral_appointment.referral_appointment_status = ReferralAppointmentStatus.find_by_title(ReferralAppointmentStatus::NEW_STATUS) if response_hash[:error]
    else
      response_hash = {}
    end

    respond_to do |format|
      if (not response_hash[:error]) && @referral_appointment.save
        format.html do
          notice_text = "Appointment ##{@referral_appointment.appointment_type} was successfully approved."
          redirect_to consultation_path(@consultation, tabpage: 'appointments'), notice: notice_text
        end
        format.json do
          result_html = created_in_cprs_datacell_html(@referral_appointment, current_user)
          render json: {appointment_status: result_html}, status: :ok
        end
      else
        @referral_appointment.save
        format.html do
          flash[:alert] = notice_text || @referral_appointment.errors.full_messages.join(" ")
          redirect_to consultation_path(@consultation, tabpage: 'appointments')
        end
        format.json { render json: (notice_text || @referral_appointment.errors), status: :unprocessable_entity }
      end
    end

  end


  #########################################################################
  ## Don't Peek at my Privates!!

  private

  # Basic before_action: only set consultation filter preferences.
  def set_consultation_prefs
    @consultation_filter = current_user.user_preference.consultation_filter
  end

  # Set consultation filter preferences and filtered list.
  def set_filtered_consultations
    set_consultation_prefs
    @filtered_consultations = Consultation.filter(@consultation_filter)
  end

  # Save search params to user preferences first, then set consultation filter
  # preferences and filtered list.
  def set_filtered_consultations_and_user_prefs
    save_to_user_preference
    set_filtered_consultations
  end

  # Save parameters (passed in or from HTTP request hash) to user preferences.
  # The 'params' hash passed in from the request is used as the default.
  # Pass in an empty hash to reset prefs.
  def save_to_user_preference(consultation_params = nil)
    consultation_params ||= consultation_filter_params.to_h
    consultation_params['status'] ||= []
    consultation_params['referral_status'] ||= []
    @saved = current_user.user_preference.save_filter(:consultation, consultation_params)
    current_user.reload if @saved
  end

  # Set up existing consultation, and create temporary new referral for
  # rendering "Add New Referral" form
  def set_consultation
    if params.key? :consultation_id
      @consultation = Consultation.find(params[:consultation_id])
    else
      @consultation = Consultation.find(params[:id])
    end
    @referral = Referral.new
    @referral_document = ReferralDocument.new
    @ordering_provider = ProviderDataPresenter.new(@consultation)
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def consultation_params
    params.require(:consultation).permit(:consultation_number,
      :care_category_id, :urgent_priority, :valid_from, :valid_to,
      :consultation_order_id, :veteran_id, :ordering_physician_name,
      :requesting_provider_telephone, :requesting_provider_fax,
      :requesting_provider_contact_name, :consultation_status_id,
      :ordering_provider_id, :veteran_ssn, :veteran_mpi_pid, :veteran_local_pid, :veteran_first_name, :veteran_last_name,
      :veteran_middle_name, :consultation_title,:veteran_dob, :via_created_date, :consultation_text,
      :veteran_other_health_insurance_name, :comment)
  end


  # Never trust parameters from the scary internet, only allow the white list through.
  def consultation_filter_params
    params.permit(:from_date, :to_date, :first_name, :last_name, :ssn, :dob,
                  :coordinator_id, :care_category_id, :ordering_provider_text,
                  :ordering_provider_id, :consultation_number, :visn_id,
                  :facility_id, :authorization_number,:referral_type_id, {referral_status: []}, 
                  {status: []}, :expired_data, :new_appointments, :medical_record_letters, :stat_data)
  end

  def referral_params
    params.require(:referral).permit(:referral_type_id,
      :authorization_number, :provider_id, :coordinator_id, {referral_status: []})
  end

  def referral_document_params
    params.require(:referral_document).permit(:document_name, :referral_document_type_id, :referral_id, :vha_document)
  end

  def referral_doc_file_from_params
    doc = params[:referral_document][:document]
    params[:referral_document].extract! :document
    doc
  end

  def move_file_to_uploads_dir
    @uploaded_file = referral_doc_file_from_params
    file_name = @uploaded_file.original_filename
    tmpfile = @uploaded_file.tempfile
    tmpfile.close #Unix might let you move an open file, but not windows.
    FileUtils.mv tmpfile, File.join(FILE_UPLOAD_PATH, file_name)
  end

  #Changing urgent priority to boolean
  def urgent_priority(consultation)
    consultation.urgent_priority = (consultation.urgent_priority.to_s.downcase == "true")
    return consultation
  end

  #End of changing urgent priority to boolean

  def format_update_parameters(parameters)
    parameters[:urgent_priority] = (parameters[:urgent_priority].to_s.downcase == "true")
    unless parameters[:valid_from].blank?
      begin
        parameters[:valid_from] = DateTime.parse(Time.strptime(parameters[:valid_from], '%m/%d/%Y').to_s)
        parameters[:valid_to] = DateTime.parse(Time.strptime(parameters[:valid_to], '%m/%d/%Y').end_of_day.to_s)
      rescue
        Rails.logger.error "Date parameters (from #{parameters[:valid_from]} " +
                               "to #{parameters[:valid_to]}) are invalid."
      end
    end

    return parameters
  end

  def param_present?(key)
    params[key.to_s].present?
  end

  def referral_note_params
    params.require(:referral_note).permit(:user, :notes)
  end

  # convert 'appointment_time' to TimeWithZone if found in params hash.
  # Return params hash with converted appointment time.
  def convert_appointment_time(params_hash)
    if (params_hash['appointment_time'])
      params_hash['appointment_time'] = datetime_string_to_time_with_zone(
        params_hash['appointment_time'],
        ReferralAppointment::APPOINTMENT_TIME_FORMAT,
        current_user
      )
    end
    params_hash
  end


  # Never trust parameters from the scary internet, only allow the white list through.
  def referral_appointment_params
    params.require(:referral_appointment).permit( :appointment_type,
                                                  :scheduled_by,
                                                  :appointment_time,
                                                  :referral_appointment_status_id)
  end

  def set_referral_and_appointment
    @referral = Referral.find(params[:referral_id])
    if params.has_key?(:id)
      @referral_appointment = ReferralAppointment.find(params[:id])
    elsif params.has_key?(:referral_appointment_id)
      @referral_appointment = ReferralAppointment.find(params[:referral_appointment_id])
    else
      raise Exception, "Referral appointment ID not found."
    end
    # check if referral appointment is contained in referral
    unless @referral.referral_appointments.include? @referral_appointment
      raise Exception, "Invalid Referral appointment ##{@referral_appointment.id} for referral ##{@referral.id}."
    end
  end

  # handle VIA_API calls to cancel appointment.  Returns deleted appointment if successful.
  def cancel_appointment_in_vista
    # get clinic id from site id saved in user preferences
    clinic = Clinic.get_clinic_from_user(@referral.coordinator)
    patient = Struct.new(:mpi_pid, :local_pid).new(@referral.consultation.veteran_mpi_pid, @referral.consultation.veteran_local_pid)

    begin
      # cancel appointment through VISTA.  The cancel_code, cancel_reason, and
      # remarks parameter are always set to default values.
      appointment_list = VIA_API::SchedulingService.cancel_appointment(
        @referral_appointment.appointment_time.utc,
        # NOTE: The patient struct is added to avoid the mpi_lookup call in the below line as it does not work in pre-prod.
        # @referral.veteran(session[:vista]),
        patient,
        clinic.clinic_identifier,
        11,
        'OTHER',
        'REBOOKED',
        session[:vista]
      )
      cancelled_appointment = (appointment_list.first if appointment_list.present?)

      if cancelled_appointment.present?
        return {error: false}
      else
        return {error: true, error_message: "Error cancelling appointment in VISTA."}
      end

    rescue VIA_API::ViaApiError => e
      Rails.logger.error(notice_text)
      return {error: true, error_message: "Error cancelling appointment in VISTA: #{e.message}"}
    end

  end

  # handle VIA_API calls to make appointment, and save response id successful.
  # Returns {success: <boolean>} hash, where success is true if VIA_API call and
  # internal save are both successful.
  def make_appointment_in_vista
    # first make appointment in VISTA and set added flags only if that operation succeeds.
    saved_appointment, notice_text = nil, nil
    begin
      # get clinic id from site id saved in user preferences
      clinic = Clinic.get_clinic_from_user(@referral.coordinator)

      # make appointment through VISTA.  The nil parameter is the appointment type, which
      # should get set to the default.

      patient = Struct.new(:mpi_pid, :local_pid).new(@referral.consultation.veteran_mpi_pid, @referral.consultation.veteran_local_pid)
      appointment_list = VIA_API::SchedulingService.make_appointment(
        # NOTE: The patient struct is added to avoid the mpi_lookup call in the below line as it does not work in pre-prod.
        # @referral.veteran(session[:vista]),
        patient,
        clinic.clinic_identifier,
        nil,
        @referral_appointment.appointment_time.utc,
        @referral_appointment.book_notes,
        session[:vista]
      )
      saved_appointment = (appointment_list.first if appointment_list.present?)

      if saved_appointment.present?
        end_time = @referral_appointment.appointment_time + (saved_appointment.appointment_length.to_i).minutes
        @referral_appointment.purpose = saved_appointment.purpose
        @referral_appointment.appointment_end_time = end_time
        @referral_appointment.added_to_cprs_at = Time.now
        @referral_appointment.added_to_cprs_id = current_user.id
        return {error: false}
      else
        return {error: true, error_message: "Error saving appointment to VISTA."}
      end

    rescue VIA_API::ViaApiError => e
      Rails.logger.error(notice_text)
      return {error: true, error_message: e.message}
    end
  end

end # class ConsultationsController < ApplicationController
